package ru.yandex.wmtools.common.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import ru.yandex.wmtools.common.data.info.RegionInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.util.geobase.RegionInfoByNameComparator;

/**
 * Date: May 13, 2009
 *
 * @author baton
 */
abstract public class CustomRegionService<HostInfoType> extends AbstractDbService {
    public static final int MAX_NUMBER_OF_KEY_REGIONS = 25;

    private RegionsTreeCacheService regionsTreeCacheService;

    public void setRegionsTreeCacheService(RegionsTreeCacheService regionsTreeCacheService) {
        this.regionsTreeCacheService = regionsTreeCacheService;
    }

    public void addCustomRegionForHost(HostInfoType hostInfo, int regionId) throws InternalException, UserException {
        List<RegionInfo> keyRegions = getKeyRegionsForHost(hostInfo);
        if (keyRegions.size() >= MAX_NUMBER_OF_KEY_REGIONS) {
            throw new UserException(UserProblem.TOO_MANY_CUSTOM_REGIONS_FOR_HOST, "There are already " + MAX_NUMBER_OF_KEY_REGIONS + " key regions. Cannot add any more.");
        }

        List<Integer> customRegionIds = getKeyRegionIdsForHost(hostInfo);
        if (customRegionIds.size() == 0) {
            // we are adding first custom region. Let's add to database all default regions.
            addAllDefaultKeyRegions(hostInfo);
        }

        internalAddKeyRegionForHost(hostInfo, regionId);
    }

    private void addAllDefaultKeyRegions(HostInfoType hostInfo) throws UserException, InternalException {
        for (RegionInfo region : regionsTreeCacheService.getDefaultKeyRegionsInfo()) {
            internalAddKeyRegionForHost(hostInfo, region.getId());
        }
    }

    public void internalAddKeyRegionForHost(HostInfoType hostInfo, int regionId) throws InternalException, UserException {
        int rowsUpdated = internalAddKeyRegionForHostQuery(hostInfo, regionId);

        if (rowsUpdated != 1) {
            if (rowsUpdated != 0) {
                throw new AssertionError("This should never happen: updated not 0 and not 1 rows.");
            }

            // no queries added to list
            throw new UserException(UserProblem.SUCH_CUSTOM_REGION_ALREADY_ADDED, "Custom region " + regionId + " is already added for this host.");
        }
    }

    abstract protected int internalAddKeyRegionForHostQuery(HostInfoType hostInfo, int regionId) throws InternalException;

    public int internalAddKeyRegionsForHostQuery(HostInfoType hostInfo, List<Integer> regionIds) throws InternalException {
        int count = 0;
        for (Integer regiodId : regionIds) {
            count += internalAddKeyRegionForHostQuery(hostInfo, regiodId);
        }
        return count;
    }

    public void removeCustomRegionsForHost(HostInfoType hostInfo, List<Integer> regionIds) throws InternalException, UserException {
        List<Integer> customRegionIds = getKeyRegionIdsForHost(hostInfo);
        if (customRegionIds.size() == 0) {
            // we are making first action. Let's add to database all default regions.
            addAllDefaultKeyRegions(hostInfo);
        }

        if (internalRemoveCustomRegionsForHost(hostInfo, regionIds) != regionIds.size()) {
            // some queries were not removed from list
            throw new UserException(UserProblem.NO_SUCH_CUSTOM_REGION_FOUND, "Some regions to be deleted are not found.");
        }
    }

    abstract protected int internalRemoveCustomRegionsForHost(HostInfoType hostInfo, List<Integer> regionIds) throws InternalException;

    abstract protected List<Integer> getKeyRegionIdsForHost(HostInfoType hostInfo) throws InternalException;

    public List<RegionInfo> getKeyRegionsForHost(HostInfoType hostInfo, boolean includeZero) throws InternalException {
        List<RegionInfo> res = new ArrayList<RegionInfo>();
        res.addAll(getKeyRegionsForHost(hostInfo));

        if (!includeZero) {
            RegionInfo toRemove = null;

            for (RegionInfo info : res) {
                if (info.getId() == 0) {
                    toRemove = info;
                }
            }

            if (toRemove != null) {
                res.remove(toRemove);
            }
        }

        return res;
    }

    public List<RegionInfo> getKeyRegionsForHost(HostInfoType hostInfo) throws InternalException {
        List<Integer> regionsList = getKeyRegionIdsForHost(hostInfo);

        List<RegionInfo> res = regionsTreeCacheService.getRegionsInfo(regionsList);
        if (res.size() == 0) {
            res = regionsTreeCacheService.getDefaultKeyRegionsInfo();
        }

        Collections.sort(res, new RegionInfoByNameComparator());

        return res;
    }

    abstract protected List<Integer> getKeyRegionIdsForHostAndDate(HostInfoType hostInfo, Date date) throws InternalException;

    public List<RegionInfo> getKeyRegionsForHostAndDate(HostInfoType hostInfo, Date date) throws InternalException {
        List<Integer> regionsList = getKeyRegionIdsForHostAndDate(hostInfo, date);

        List<RegionInfo> res = regionsTreeCacheService.getRegionsInfo(regionsList);
        if (res.size() == 0) {
            res = regionsTreeCacheService.getDefaultKeyRegionsInfo();
        }

        Collections.sort(res, new RegionInfoByNameComparator());

        return res;
    }

    abstract protected List<Integer> getKeyRegionIdsForHostAndAllTheTime(HostInfoType hostInfo) throws InternalException;

    public List<RegionInfo> getKeyRegionsForHostAndAllTheTime(HostInfoType hostInfo) throws InternalException {
        List<Integer> regionsList = getKeyRegionIdsForHostAndAllTheTime(hostInfo);

        List<RegionInfo> res = regionsTreeCacheService.getRegionsInfo(regionsList);
        if (res.size() == 0) {
            res = regionsTreeCacheService.getDefaultKeyRegionsInfo();
        }

        Collections.sort(res, new RegionInfoByNameComparator());

        return res;
    }
}
