package ru.yandex.wmtools.common.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import wmc.stubs.Stubs;

import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;

/**
 * @author avhaliullin
 */
public class LightDispatcherHttpService extends AbstractExternalHttpService {
    private static final Logger log = LoggerFactory.getLogger(LightDispatcherHttpService.class);

    protected static final int SOCKET_TIMEOUT = 15000;
    private String clientName;
    private static final String DEFAULT_CLIENT_NAME = "wmconsole";

    public void setClientName(String clientName) {
        this.clientName = clientName;
    }

    public String getClientName() {
        if (clientName == null) {
            log.warn("DispatcherHttpService.clientName is not specified");
            return DEFAULT_CLIENT_NAME;
        } else {
            return clientName;
        }
    }

    public String getMainMirror(final String hostname) throws InternalException {
        return getMainMirror(hostname, null);
    }

    public String getMainMirror(final String hostname, final String clientName) throws InternalException {
        final String extendedClientName = getExtendedClientName(clientName);
        log.debug("Asking for main mirror: " + hostname + " from " + extendedClientName);
        Stubs.string_msg.Builder ans;
        try {
            ans = Stubs.string_msg.newBuilder().mergeFrom(httpGetResponse("getMainMirror", createParams()
                    .addParam("hostname", hostname)
                    .addParam("client", extendedClientName), SOCKET_TIMEOUT));
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
        if (!ans.hasVal()) {
            throw assertServantIsUnavailable();
        }
        return ans.getVal();
    }

    public String getExtendedClientName(final String extendedName) {
        if (extendedName == null) {
            return getClientName();
        } else {
            return getClientName() + "::" + extendedName;
        }
    }

    public List<String> getMainMirrors(List<String> hosts) throws InternalException {
        Stubs.string_msg.Builder ans;
        try {
            String hostsParam = getDelimitedWith(hosts, "\n");
            if (hostsParam.length() > 4 * 1024 * 1024) {
                throw new InternalException(InternalProblem.ILLEGAL_ARGUMENT, "\"hostnames\" param for getMainMirrors request must be less than 4MB");
            }
            ans = Stubs.string_msg.newBuilder().mergeFrom(httpPostResponse("getMainMirrors", createParams()
                    .addParam("hostnames", hostsParam), 2 * SOCKET_TIMEOUT));
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }

        if (!ans.hasVal()) {
            throw assertServantIsUnavailable();
        }

        List<String> res = new ArrayList<String>();
        log.debug(ans.getVal().replaceAll("\n", "\\\\n"));
        for (StringTokenizer st = new StringTokenizer(ans.getVal(), "\n"); st.hasMoreTokens(); ) {
            res.add(st.nextToken());
        }
        return res;
    }

    public long getPagesInIndex(String hostName) throws InternalException {
        log.debug("Asking for pages in index for \"" + hostName + "\"");
        Stubs.host_info_search_docs_count.Builder ans;
        try {
            ans = Stubs.host_info_search_docs_count.newBuilder().mergeFrom(httpGetResponse("getSearchDocsCount", createParams()
                    .addParam("hostname", hostName)
                    .addParam("client", clientName), SOCKET_TIMEOUT));
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
        long result;
        if (ans.hasVal()) {
            result = ans.getVal().getVal();
        } else {
            if (ans.hasErr()) {
                if (ans.getErr().getVal() == Stubs.error_code.HostNotFound) {
                    result = 0;
                } else {
                    throw assertServantIsUnavailable(ans.getErr());
                }
            } else {
                throw assertServantIsUnavailable();
            }
        }
        if (result < 0) {
            log.warn("Negative pages in index count. Treat as unsigned int32");
            result += (-((long) Integer.MIN_VALUE)) * 2;
        }
        return result;
    }


    protected InternalException assertServantIsUnavailable(Stubs.error_msg errorMsg) throws InternalException {
        //Как показала практика, errorMsg.hasVal() верить нельзя
        String code = "\"" + String.valueOf(errorMsg.getVal()) + "\"";
        String desc = "\"" + String.valueOf(errorMsg.getDesc()) + "\"";
        return new InternalException(InternalProblem.SERVANT_FAILED, "External servant is unavailable: " +
                "dispatcher returned error with code: " + code + ", description: " + desc);
    }
}
