package ru.yandex.wmtools.common.service;

import org.apache.http.entity.AbstractHttpEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.util.HttpConnector;

import java.io.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author avhaliullin
 */
public class MRTablesService extends AbstractExternalHttpService {
    private static final Logger log = LoggerFactory.getLogger(MRTablesService.class);

    private boolean debugMode = false;
    private String mrUserName;

    public void rewriteTable(String tableName, TableContentWriter tableWriter) throws InternalException {
        log.info("Rewriting MR table " + tableName + " in " + externalServantUrl +
                ". Debug mode is switched " + (debugMode ? "on" : "off"));
        String url = externalServantUrl + tableName.replace("/", "%2F") + ",replace/?username=" + mrUserName;
        if (debugMode) {
            log.info("Querying " + url);
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            tableWriter.writeTo(out);
            try {
                out.close();
                log.debug("Table content:\n" + out.toString("UTF-8"));
            } catch (IOException e) {
                //ignore
            }
        } else {
            InputStream content = null;
            try (InputStream in = new HttpConnector.RequestBuilder(new URL(url))
                    .method(HttpConnector.HttpMethod.POST)
                    .entity(new TableContentEntity(tableWriter))
                    .okStatusRequired(true)
                    .socketTimeout(30000)
                    .execute().getContent()) {
                //do nothing
            } catch (MalformedURLException e) {
                throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Bad MR url " + url, e);
            } catch (IOException e) {
                throw assertServantIsUnavailable(e);
            }
        }
    }

    private class TableContentEntity extends AbstractHttpEntity {
        private final AtomicBoolean streaming = new AtomicBoolean(false);
        private final TableContentWriter contentWriter;

        private TableContentEntity(TableContentWriter contentWriter) {
            super();
            this.contentWriter = contentWriter;
            setContentType("application/octet-stream");
            setChunked(true);
        }

        @Override
        public boolean isRepeatable() {
            return false;
        }

        @Override
        public long getContentLength() {
            return -1;
        }

        @Override
        public InputStream getContent() throws IOException, IllegalStateException {
            if (streaming.compareAndSet(false, true)) {
                PipedInputStream in = new PipedInputStream();
                PipedOutputStream out = new PipedOutputStream(in);
                writeTo(out);
                return in;
            } else {
                throw new IllegalStateException("This entity is not repeatable!");
            }
        }

        @Override
        public void writeTo(OutputStream out) throws IOException {
            if (streaming.compareAndSet(false, true)) {
                try {
                    contentWriter.writeTo(out);
                } finally {
                    out.close();
                }
            } else {
                throw new IllegalStateException("This entity is not repeatable!");
            }
        }

        @Override
        public boolean isStreaming() {
            return true;
        }
    }

    public interface TableContentWriter {
        public void writeTo(OutputStream out);
    }


    public void setDebugMode(boolean debugMode) {
        this.debugMode = debugMode;
    }

    @Required
    public void setMrUserName(String mrUserName) {
        this.mrUserName = mrUserName;
    }
}
