package ru.yandex.wmtools.common.service;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.util.URLUtil;
import ru.yandex.wmtools.common.SupportedProtocols;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 10.12.2007
 * Time: 10:59:21
 */
public class OwnerDomainService implements IService {
    protected static final Logger log = LoggerFactory.getLogger(OwnerDomainService.class);

    private Tree tree;
    private static final String DOMAIN_SPLIT_PATTERN = "[.]";

    private static OwnerDomainService INSTANCE;

    private static OwnerDomainService getInstance() throws InternalException {
        if (INSTANCE == null) {
            INSTANCE = new OwnerDomainService();
        }
        return INSTANCE;
    }

    private OwnerDomainService() throws InternalException {
        InputStream ldListStream = getClass().getResourceAsStream("/data/areas.lst");
        BufferedReader reader = new BufferedReader(new InputStreamReader(ldListStream));
        tree = new Tree();
        try {
            try {
                int count = 0;
                while (reader.ready()) {
                    String line = reader.readLine().trim();
                    if (line.length() != 0) {
                        if (tree.add(line)) {
                            count++;
                        }
                    }
                }
                log.debug("OwnerDomainService: " + count + " domais loaded");
            }
            finally {
                reader.close();
            }
        } catch (FileNotFoundException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "FileNotFoundException", e);
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "IOException", e);
        }
    }

    /**
     * Получить домен-владелец
     *
     * @param domain    имя хоста без протокола и без порта
     * @return          домен-владелец
     * @throws InternalException
     */
    public static String getOwnerDomain(String domain) throws InternalException {
        LinkedList<String> owner = getInstance().find(domain);
        log.debug("owner = " + owner);
        List<String> list = parseSegments(domain);
        String result = getLastNSegments(list, Math.max(owner.size() + 1, 2));
        log.debug("getOwnerDomain for " + domain + " is " + result);
        return result;
    }

    /**
     * Получить хост-владелец с протоколом и портом (если есть).
     *
     * Получает на вход домен с протоколом или без, возвращает домен-владелец с таким же протоколом как на входе
     * или без протокола, если на входе не было протокола
     *
     * @param domain    имя хоста (с протоколом или без, с портом или без)
     * @return
     * @throws InternalException
     */
    public static String getOwnerHost(final String domain) throws InternalException {
        try {
            final URL url = SupportedProtocols.getURL(domain);
            final String resultDomain = getOwnerDomain(url.getHost());
            final URL resultUrl = new URL(url.getProtocol(), resultDomain, url.getPort(), "");
            return URLUtil.getHostName(resultUrl, false);
        } catch (MalformedURLException e) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "malformed url", e);
        } catch (URISyntaxException e) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "uri syntax", e);
        } catch (SupportedProtocols.UnsupportedProtocolException e) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "unsupported protocol", e);
        }
    }

    private static String getLastNSegments(List<String> segments, int n) {
        int size = segments.size();
        String owner = segments.get(size - 1);
        if (segments.size() > 1) {
            int last = size - n;
            if (last < 0) {
                last = 0;
            }
            for(int i = size - 2; i >= last; i--) {
                owner = segments.get(i) + "." + owner;
            }
        }
        return owner;
    }

    private LinkedList<String> find(String domain) {
        return tree.find(domain);
    }

    public static String getOwnerDomainRPattern(String domain) throws InternalException {
        String ownerDomain = getOwnerDomain(domain);
        try {
            URL url = AbstractServantlet.prepareUrl(ownerDomain, true);
            // отбрасываем схему и порт
            ownerDomain = url.getHost();
        } catch (UserException e) {
            log.error("Unable to parse URL for owner domain");
        }
        List<String> list = parseSegments(ownerDomain);
        Collections.reverse(list);
        String rOwner = list.get(0);
        if (list.size() > 1) {
            for(int i = 1; i < list.size(); i++) {
                rOwner += ("." + list.get(i));
            }
        }
        if (parseSegments(domain).size() > list.size()) {
            rOwner += ".*";
        }
        log.debug("getOwnerDomainRPattern for " + domain + " is " + rOwner);
        return rOwner;
    }

    static List<String> parseSegments(String domain) {
        return Arrays.asList(domain.split(DOMAIN_SPLIT_PATTERN));
    }

    private static class Tree {
        public Tree() {
            this.fRoot = new Node(null);
        }
        public Node fRoot;

        public boolean add(String domain) {
            LinkedList<String> list = new LinkedList<String>(parseSegments(domain));
            Collections.reverse(list);
            return fRoot.add(list);
        }

        public LinkedList<String> find(String domain) {
            LinkedList<String> segments = new LinkedList<String>(parseSegments(domain));
            Collections.reverse(segments);
            Node node = fRoot.find(segments);
            LinkedList<String> result = new LinkedList<String>();
            if (node.equals(fRoot)) {
                return result;
            }
            result.add(node.getName());
            Node parent = node.getParent();
            while (parent != null) {
                String parentName = parent.getName();
                if (parentName != null) {
                    result.add(parentName);
                }
                parent = parent.getParent();
            }
            return result;
        }
    }

    private static class Node {
        private Map<String, Node> children = new HashMap<String, Node>();
        private Node parent;
        private String name;

        public Node(String name) {
            this.name = name;
        }

        public boolean add(LinkedList<String> domainSegments) {
            String firstSegment = domainSegments.getFirst();
            Node child = children.get(firstSegment);
            if (child == null) {
                child = new Node(firstSegment);
                child.parent = this;
                children.put(firstSegment, child);
            }
            domainSegments.removeFirst();
            if (domainSegments.isEmpty()) {
                return true;
            }
            else if (child.add(domainSegments)) {
                return true;
            }
            return false;
        }

        public String getName() {
            return name;
        }

        public Map<String, Node> getChildren() {
            return children;
        }

        Node getParent() {
            return parent;
        }

        public Node find(LinkedList<String> domainSegments) {
            String firstSegment = domainSegments.get(0);
            Node child = children.get(firstSegment);
            if (child == null) {
                return this;
            }
            domainSegments.removeFirst();
            if (domainSegments.isEmpty()) {
                return child;
            }
            return child.find(domainSegments);
        }
    }
}
