package ru.yandex.wmtools.common.service;

import org.apache.commons.lang.mutable.MutableInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.FileSystemResource;
import org.xml.sax.SAXException;
import ru.yandex.wmtools.common.data.info.RegionInfo;
import ru.yandex.wmtools.common.data.info.RegionInfoTreeNode;
import ru.yandex.wmtools.common.data.info.RegionName;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.util.geobase.GeobaseRegionInfoHandler;
import ru.yandex.wmtools.common.util.geobase.GeobaseRegionSaver;

import javax.xml.parsers.ParserConfigurationException;
import java.io.File;
import java.io.IOException;
import java.util.*;

/**
 * @author baton
 */
abstract public class RegionsTreeCacheService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(RegionsTreeCacheService.class);

    private static final List<Integer> defaultKeyRegions = Arrays.asList(
            RegionIds.RUSSIA,
            RegionIds.UKRAINE,
            RegionIds.BELARUS,
            RegionIds.KAZAKHSTAN,
            RegionIds.MOSCOW,
            RegionIds.SAINT_PETERSBURG
    );

    private static final List<Integer> oldKeyRegionsForRegionsCompare = Arrays.asList(
            RegionIds.SAINT_PETERSBURG,
            RegionIds.KALININGRAD,
            RegionIds.KRASNODAR,
            RegionIds.ROSTOV_NA_DONU,
            RegionIds.KAZAN,
            RegionIds.NIZNY_NOVGOROD,
            RegionIds.PERM,
            RegionIds.SAMARA,
            RegionIds.YEKATERINBURG,
            RegionIds.CHELYABINSK,
            RegionIds.KRASNOYARSK,
            RegionIds.IRKUTSK,
            RegionIds.KEMEROVO,
            RegionIds.NOVOSIBIRSK,
            RegionIds.OMSK,
            RegionIds.VLADIVOSTOK,
            RegionIds.BELARUS,
            RegionIds.KAZAKHSTAN,
            RegionIds.UFA,
            RegionIds.UKRAINE,
            RegionIds.VORONEZH,
            RegionIds.BARNAUL,
            RegionIds.MOSCOW,
            RegionIds.RUSSIA
    );

    private List<RegionInfo> defaultKeyRegionsInfo;
    private List<RegionInfo> oldKeyRegionsForRegionsCompareInfo;
    private Map<Integer, RegionInfoTreeNode> regionsTree;

    private GeobaseService geobaseService;

    private File geobaseCacheResource;

    /*
    * Spring init-method. Initializing regions tree.
    */
    public void init() throws InternalException, ParserConfigurationException, SAXException, IOException {
        List<RegionInfo> regions;
        try {
            regions = loadRegionsFromDB();
            loadRegionsFromGeobase(regions);
            saveLocalCache(regions);
        } catch (InternalException e) {
            log.error("Failed to initialize RegionsTreeCacheService", e);
            regions = loadLocalCache();
        } catch (IOException e) {
            log.error("Failed to initialize RegionsTreeCacheService", e);
            regions = loadLocalCache();
        }

        regionsTree = Collections.unmodifiableMap(convertToTreeNodes(regions));
        // initializing information for default regions
        defaultKeyRegionsInfo = getRegionsInfo(defaultKeyRegions);
        oldKeyRegionsForRegionsCompareInfo = getRegionsInfo(oldKeyRegionsForRegionsCompare);
        log.debug("Initialization was successful");
    }

    private List<RegionInfo> loadLocalCache() throws IOException, SAXException, ParserConfigurationException {
        log.info("Load regions from file cache");
        List<RegionInfo> regions = GeobaseRegionSaver.load(new FileSystemResource(geobaseCacheResource));
        log.info("Cache regions loaded: " + regions.size());
        return regions;
    }

    private void saveLocalCache(List<RegionInfo> regions) throws IOException, InternalException {
        log.info("Save regions cache");
        File tmpFile = new File(geobaseCacheResource.getParentFile(), geobaseCacheResource.getName() + ".tmp");
        GeobaseRegionSaver.save(regions, new FileSystemResource(tmpFile));
        if (!tmpFile.renameTo(geobaseCacheResource)) {
            log.error(
                    "Unable to update regions cache: " + tmpFile.getAbsolutePath() + " to " + geobaseCacheResource.getAbsolutePath());
        }
    }

    private List<RegionInfo> loadRegionsFromDB() throws InternalException {
        log.debug("Load regions from DB");
        List<RegionInfo> allRegions = getAllRegionsFromDb();
        log.debug("DB regions loaded: " + allRegions.size());
        return allRegions;
    }

    private void loadRegionsFromGeobase(List<RegionInfo> regions) throws InternalException {
        log.debug("Loaded regions from geobase");
        final Map<Integer, RegionInfo> regionsMap = new HashMap<Integer, RegionInfo>();
        for (RegionInfo region : regions) {
            regionsMap.put(region.getId(), region);
        }

        final MutableInt regionsFound = new MutableInt();
        final MutableInt regionsTotal = new MutableInt();
        geobaseService.loadRegionsInfo(new GeobaseRegionInfoHandler() {
            @Override
            public void region(int id, RegionName regionName) {
                regionsTotal.increment();
                RegionInfo regionInfo = regionsMap.get(id);
                if (regionInfo != null) {
                    regionsFound.increment();
                    regionInfo.setRegionName(regionName);
                }
            }
        });
        log.debug("Geobase regions found: " + regionsFound);
        log.debug("Geobase regions total: " + regionsTotal);
    }

    private Map<Integer, RegionInfoTreeNode> convertToTreeNodes(List<RegionInfo> allRegions) {
        Map<Integer, RegionInfoTreeNode> map = new HashMap<Integer, RegionInfoTreeNode>();
        for (RegionInfo regionInfo : allRegions) {
            RegionInfoTreeNode node = map.get(regionInfo.getId());
            if (node == null) {
                node = new RegionInfoTreeNode();
                map.put(regionInfo.getId(), node);
            }

            RegionInfoTreeNode parent = map.get(regionInfo.getParent());
            if (parent == null) {
                parent = new RegionInfoTreeNode();
                map.put(regionInfo.getParent(), parent);
            }

            parent.getChildren().add(node);
            node.setWholeRegionInfo(regionInfo, parent);
        }
        return map;
    }

    protected abstract List<RegionInfo> getAllRegionsFromDb() throws InternalException;

    public List<RegionInfo> getDefaultKeyRegionsInfo() throws InternalException {
        return defaultKeyRegionsInfo;
    }

    public List<RegionInfo> getOldKeyRegionsForRegionsCompareInfo() throws InternalException {
        return oldKeyRegionsForRegionsCompareInfo;
    }

    public final RegionInfo getRegionInfo(int regionId) throws InternalException {
        RegionInfoTreeNode regionInfoTreeNode = regionsTree.get(regionId);
        if (regionInfoTreeNode == null) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Unable to find region id=" + regionId);
        }
        return regionInfoTreeNode.getRegionInfo();
    }

    public final List<RegionInfo> getRegionsInfo(List<Integer> regionIds) throws InternalException {
        List<RegionInfo> res = new ArrayList<RegionInfo>();
        for (int regionId : regionIds) {
            res.add(getRegionInfo(regionId));
        }

        return res;
    }

    public final List<Integer> getRegionsIds(Collection<RegionInfo> regions) {
        List<Integer> ids = new ArrayList<Integer>();
        for (RegionInfo info : regions) {
            ids.add(info.getId());
        }
        return ids;
    }

    public final List<Integer> getSubTreeForRegionByInfoList(int regionId, List<RegionInfo> stopOnRegions) throws InternalException {
        List<Integer> stopOn = new ArrayList<Integer>();
        for (RegionInfo info : stopOnRegions) {
            stopOn.add(info.getId());
        }

        return getSubTreeForRegionByIntList(regionId, stopOn);
    }

    public final List<Integer> getSubTreeForRegionByIntList(int regionId, List<Integer> stopOn) throws InternalException {
        List<Integer> res = new ArrayList<Integer>();

        RegionInfoTreeNode root = regionsTree.get(regionId);
        if (root == null) {
            return null;
        }

        Set<Integer> stopOnSet = new HashSet<Integer>(stopOn);

        addSubTree(res, stopOnSet, root, false);

        return res;
    }

    private void addSubTree(final List<Integer> res, final Set<Integer> stopOnSet,
            final RegionInfoTreeNode node, final boolean canStop)
    {
        if ((canStop) && (stopOnSet.contains(node.getRegionInfo().getId()))) {
            return;
        }

        res.add(node.getRegionInfo().getId());
        for (RegionInfoTreeNode child : node.getChildren()) {
            if (child.getRegionInfo().getId() == node.getRegionInfo().getId()) {
                continue;
            }

            addSubTree(res, stopOnSet, child, true);
        }
    }

    public int getKeyRegionForRegion(int regionId, final Collection<Integer> keyRegions) throws InternalException {
        int initialRegionId = regionId;

        while (true) {
            if (keyRegions.contains(regionId)) {
                return regionId;
            }

            int newRegionId = getRegionInfo(regionId).getParent();
            if (newRegionId == regionId) {
                throw new AssertionError(
                        "Cannot find any key region for region: " + initialRegionId + "." +
                        " Finished searching at the loop on region: " + regionId + " which is parent for itself.");
            }

            regionId = newRegionId;
        }
    }

    private boolean containsInSubTree(Integer regionId, int subRegionId) throws InternalException {
        if (regionId == null) {
            return true;
        }
        while (true) {
            if (regionId == subRegionId) {
                return true;
            }

            int newRegionId = getRegionInfo(subRegionId).getParent();
            if (newRegionId == subRegionId) {
                return false;
            }

            subRegionId = newRegionId;
        }
    }

    private boolean containsInSubTreeAndHighest(Integer regionId, Integer subRegionId, Collection<Integer> keyRegions)
            throws InternalException
    {
        boolean first = true;
        while (true) {
            if (subRegionId.equals(regionId)) {
                return true;
            }

            if (!first && keyRegions.contains(subRegionId)) {
                return false;
            }
            first = false;

            int newRegionId = getRegionInfo(subRegionId).getParent();
            if (newRegionId == subRegionId) {
                //regionId==null means the whole world.
                return regionId == null || regionId == 0;
            }

            subRegionId = newRegionId;
        }
    }

    public List<Integer> getKeyRegionsFromSubTreeByIntList(Integer regionId, final Collection<Integer> keyRegions,
            boolean highestOnly) throws InternalException
    {
        List<Integer> regions = new ArrayList<Integer>();

        for (Integer currentRegion : keyRegions) {
            if ((highestOnly && containsInSubTreeAndHighest(regionId, currentRegion,
                    keyRegions)) || ((!highestOnly) && containsInSubTree(regionId,
                    currentRegion)) && regionId != currentRegion) {
                regions.add(currentRegion);
            }
        }
        return regions;
    }

    public final List<Integer> getKeyRegionsFromSubTreeByInfoList(Integer regionId,
            final Collection<RegionInfo> keyRegions, boolean highestOnly) throws InternalException
    {
        List<Integer> keyRegs = new ArrayList<Integer>();
        for (RegionInfo info : keyRegions) {
            keyRegs.add(info.getId());
        }

        return getKeyRegionsFromSubTreeByIntList(regionId, keyRegs, highestOnly);
    }

    /**
     * Builds a tree of regions and inserts countries as nodes.
     *
     * @param regionInfos regions, from which we have to build a tree.
     * @return List of roots, because we have a forest, not a single tree.
     */
    public List<RegionInfoTreeNode> getTreeWithCountries(Collection<RegionInfo> regionInfos) throws InternalException {
        // 1. init nodes
        List<RegionInfoTreeNode> nodes = new ArrayList<RegionInfoTreeNode>();
        for (RegionInfo info : regionInfos) {
            RegionInfoTreeNode node = new RegionInfoTreeNode();
            node.setRegionInfo(info);
            nodes.add(node);
        }

        // 2. build tree
        List<RegionInfoTreeNode> roots = new ArrayList<RegionInfoTreeNode>();
        for (int i = 0; i < nodes.size(); i++) {
            RegionInfoTreeNode node = nodes.get(i);

            RegionInfoTreeNode wholeTreeNode = regionsTree.get(node.getRegionInfo().getId());
            while (!Thread.interrupted()) {
                if ((wholeTreeNode.getParent() == null) || (wholeTreeNode.getParent() == wholeTreeNode)) {
                    roots.add(node);
                    break;
                }

                wholeTreeNode = wholeTreeNode.getParent();
                boolean found = false;
                for (RegionInfoTreeNode other : nodes) {
                    if (other.getRegionInfo().getId() == wholeTreeNode.getRegionInfo().getId()) {
                        other.getChildren().add(node);
                        node.setParent(other);
                        found = true;
                        break;
                    }
                }

                if (found) {
                    break;
                }

                // country?
                if (wholeTreeNode.getRegionInfo().getGeotype() == RegionInfo.GEOTYPE_COUNTRY) {
                    RegionInfoTreeNode country = new RegionInfoTreeNode(false);
                    country.setRegionInfo(wholeTreeNode.getRegionInfo());
                    country.getChildren().add(node);
                    node.setParent(country);
                    nodes.add(country);
                    break;
                }
            }
        }

        return roots;
    }

    public boolean containsRegion(int regionId) throws InternalException {
        return regionsTree.containsKey(regionId);
    }

    /**
     * Builds a tree of regions and inserts countries as nodes.
     *
     * @param regionIds ids of regions, from which we have to build a tree.
     * @return List of roots, because we have a forest, not a single tree.
     */
    public List<RegionInfoTreeNode> getTreeWithCountries(int[] regionIds) throws InternalException {
        Collection<RegionInfo> regionInfos = new ArrayList<RegionInfo>();
        for (int regionId : regionIds) {
            regionInfos.add(regionsTree.get(regionId).getRegionInfo());
        }

        return getTreeWithCountries(regionInfos);
    }

    public Map<Integer, RegionInfoTreeNode> getRegionsTree() {
        return regionsTree;
    }

    @Required
    public void setGeobaseService(GeobaseService geobaseService) {
        this.geobaseService = geobaseService;
    }

    @Required
    public void setGeobaseCacheResource(File geobaseCacheResource) {
        this.geobaseCacheResource = geobaseCacheResource;
    }
}
