package ru.yandex.wmtools.common.service;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.util.concurrent.CommonThreadFactory;
import ru.yandex.wmtools.common.data.info.VisitingStatInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.util.SqlUtil;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 20.03.2007
 * Time: 10:09:24
 */

abstract public class VisitingStatService extends AbstractDbService implements Runnable, IVisitingStatService {
    protected static final Logger log = LoggerFactory.getLogger(VisitingStatService.class);

    private static final long MILLISECONDS_IN_SECOND = 1000;
    private static final long AUTO_FIRE_TIME_IN_MILLIS = MILLISECONDS_IN_SECOND * 60 * 12;

    private IUserInfoService userInfoService;

    private static HashMap<String, Integer> request2id = new HashMap<String, Integer>();

    protected int getRequestId(String requestName) {
        if (request2id.isEmpty()) {
            loadRequest2IdMap();
        }
        Integer id = request2id.get(requestName);
        if (id == null) {
            log.warn("VisitingStatService: unknown request '" + requestName + "'");
            id = 0;
        }
        return id;
    }

    public static class RequestType {
        private int id;

        private String name;

        public RequestType(String name, int id) {
            this.name = name;
            this.id = id;
        }

        public int getId() {
            return id;
        }

        public String getName() {
            return name;
        }
    }

    protected void loadRequest2IdMap() {
        try {
            List<RequestType> types = getRequestTypeSelectQuery2();
            for (RequestType type : types) {
                request2id.put(type.getName(), type.getId());
            }
        } catch (InternalException e) {
            log.error("InternalException occured in thread " + Thread.currentThread().getName(), e);
        }
    }

    /**
     * Default public constructor.
     */
    public VisitingStatService() {
    }

    /**
     * Spring init-method.
     */
    public final void init() {
        CommonThreadFactory threadFactory = new CommonThreadFactory(true, VisitingStatService.class.getSimpleName() + "-");
        threadFactory.newThread(this).start();
    }

    private HashMap<String, RequestCountInfo> requestsInfo = new HashMap<String, RequestCountInfo>();
    private final HashMap<Long, Date> lastVisit = new HashMap<Long, Date>();

    @Override
    public final void run() {
        log.info(getClass().toString() + " thread " + Thread.currentThread().getName() + " started.");
        while (!Thread.interrupted()) {
            try {
                synchronized (this) {
                    try {
                        this.wait(AUTO_FIRE_TIME_IN_MILLIS);
                    } catch (InterruptedException e) {
                        log.warn("InterruptedException occured in thread " + Thread.currentThread().getName(), e);
                        return;
                    }
                }
                log.info(getClass().toString() + " thread " + Thread.currentThread().getName() + " auto fired.");

                saveChangesIntoDb();
            } catch (RuntimeException e) {
                log.error("RuntimeException occured in thread " + Thread.currentThread().getName(), e);
            } catch (InternalException e) {
                log.error("InternalException occured in thread " + Thread.currentThread().getName(), e);
            }
        }
    }

    private synchronized void saveChangesIntoDb() throws InternalException {
        log.info(getClass().toString() + ": saving last visit info into DB: " + lastVisit.size() + " entries");
        for (Map.Entry<Long, Date> entry : lastVisit.entrySet()) {
            userInfoService.updateLastVisit(entry);
        }
        lastVisit.clear();

        for (final String requestName : requestsInfo.keySet()) {
            RequestCountInfo info = requestsInfo.get(requestName);
            info.visit(new RequestCountInfo.IRequestCountInfoVisitor() {
                @Override
                public boolean accept(Date date, long userId, long count) {
                    try {
                        getUserRequestCountUpdateQuery(date, userId, count, requestName);
                        return true;
                    } catch (InternalException e) {
                        log.error("InternalException occured in thread " + Thread.currentThread().getName(), e);
                        return false;
                    }
                }
            });
        }
        requestsInfo.clear();
    }

    @Override
    public synchronized void updateVisitingStat(long userId, final String requestName) {
        lastVisit.put(userId, new Date());
        RequestCountInfo info = requestsInfo.get(requestName);
        if (info == null) {
            info = new RequestCountInfo();
            requestsInfo.put(requestName, info);
        }
        info.addUserRequest(userId);
    }

    /**
     * Returns a list of VisitingStatInfo for a specified day and specified user. Each element in the list tell about
     * one page. userId can be null - in this case, stats for summary of all users will be returned.
     *
     * @param date Date, for which to collect stats.
     * @param userId User id can be null - in this case, stats for summary of all users will be returned.
     * @return Returns statistics of service usage.
     * @throws InternalException Thrown if something goes wrong.
     */
    private List<VisitingStatInfo> getVisitingStatForDate(Date date, Long userId) throws InternalException {
        date = SqlUtil.getMidnightTimestamp(date);

        if (userId == null) {
            return getVisitingStatQuery(date);
        } else {
            return getVisitingStatForUserQuery(date, userId);
        }
    }

    @Override
    public List<VisitingStatInfo> getVisitingStatForLastWeek(Long userId) throws InternalException {
        Calendar cal = Calendar.getInstance();
        cal.setTime(SqlUtil.getMidnightTimestamp(null));

        List<VisitingStatInfo> result = new ArrayList<VisitingStatInfo>();

        for (int i = 0; i < 7; i++) {
            result.addAll(getVisitingStatForDate(cal.getTime(), userId));
            cal.add(Calendar.DAY_OF_MONTH, -1);
        }

        return result;
    }

    abstract protected void getUserRequestCountUpdateQuery(Date date, long userId, long count, String requestName) throws InternalException;

    abstract protected List<VisitingStatInfo> getVisitingStatForUserQuery(Date date, Long userId) throws InternalException;

    abstract protected List<VisitingStatInfo> getVisitingStatQuery(Date date) throws InternalException;

    abstract protected List<RequestType> getRequestTypeSelectQuery2() throws InternalException;

    @Required
    public void setUserInfoService(IUserInfoService userInfoService) {
        this.userInfoService = userInfoService;
    }
}
