package ru.yandex.wmtools.common.service;

import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.jdom.output.XMLOutputter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.wmtools.common.data.xmlsearch.XmlSearchRequest;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URISyntaxException;
import java.util.concurrent.TimeUnit;

public class XmlSearchService implements IService {
    private static final Logger log = LoggerFactory.getLogger(XmlSearchService.class);

    private static final String WMCVIEWER_USER_AGENT = "WMCViewer/1.0";

    private static final int DEFAULT_XMLSEARCH_CONNECTION_TIMEOUT = 7500;
    private static final int DEFAULT_XMLSEARCH_SOCKET_TIMEOUT = 15000;
    private static final int DEFAULT_MAX_TOTAL_CONNECTIONS = 50;
    private static final int CONNECTION_ATTEMPTS_COUNT = 3;

    private CloseableHttpClient httpClient;
    private String xmlSearchUrl = "https://yandex.ru/search/xml?waitall=da&relev=attr_limit%3D10000000&";

    private static final String COOKIE_HEADER_NAME = "Cookie";
    private static final String I_AM_NOT_A_HACKER_COOKIE = "i-m-not-a-hacker=ZJYnDvaNXYOmMgNiScSyQSGUDffwfSET";

    private int maxNumberOfConnections = DEFAULT_MAX_TOTAL_CONNECTIONS;
    private int connectionTimeout = DEFAULT_XMLSEARCH_CONNECTION_TIMEOUT;
    private int socketTimeout = DEFAULT_XMLSEARCH_SOCKET_TIMEOUT;

    private boolean logRawXmlResponse = false;

    protected CloseableHttpClient getHttpClient() {
        if (httpClient == null) {
            ConnectionConfig connectionConfig = ConnectionConfig.custom()
                    .setCharset(Charsets.UTF_8)
                    .build();

            RequestConfig requestConfig = RequestConfig.custom()
                    .setConnectionRequestTimeout(1)
                    .setConnectTimeout(connectionTimeout)
                    .setSocketTimeout(socketTimeout)
                    .setCookieSpec(CookieSpecs.IGNORE_COOKIES)
                    .build();

            httpClient = HttpClients.custom()
                    .setConnectionTimeToLive(30, TimeUnit.SECONDS)
                    .setDefaultConnectionConfig(connectionConfig)
                    .setDefaultRequestConfig(requestConfig)
                    .setMaxConnPerRoute(maxNumberOfConnections)
                    .setMaxConnTotal(maxNumberOfConnections)
                    .setUserAgent(WMCVIEWER_USER_AGENT)
                    .build();
        }
        return httpClient;
    }

    public void init() {
        getHttpClient();
    }

    public void destroy() {
        try {
            httpClient.close();
        } catch (IOException e) {
            log.error("Unable to close httpClient", e);
        }
    }

    public <T> T callXMLSearch(XmlSearchRequest request, XmlSearchResultParser<T> parser) throws InternalException {
        XMLOutputter outputer = new XMLOutputter();
        outputer.setFormat(outputer.getFormat().setEncoding("UTF-8"));
        String requestDocumentString = outputer.outputString(request.getDocument());

        return callXMLSearch(requestDocumentString, request.getUrlParams(), parser);
    }

    public <T> T callXMLSearch(String query, String urlParams, XmlSearchResultParser<T> parser) throws InternalException {
        try {
            log.debug("XmlSearchService.callXMLSearch: Start");
            byte[] bytes = httpPostWithRetry(getXmlSearchUrl() + urlParams, query, CONNECTION_ATTEMPTS_COUNT);
            if (logRawXmlResponse) {
                log.debug("Raw XML response: {}", new String(bytes, Charsets.UTF_8));
            }
            Reader reader = new InputStreamReader(new ByteArrayInputStream(bytes), Charsets.UTF_8);
            T result = parser.parseResult(reader);
            log.debug("XmlSearchService.callXMLSearch: Done");
            return result;
        } catch (URISyntaxException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Cannot execute query to XML Search", e);
        } catch (HttpException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Cannot execute query to XML Search", e);
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Cannot execute query to XML Search", e);
        }
    }

    private byte[] httpPostWithRetry(String urlAddress, String query, int attemptsLeft) throws URISyntaxException, IOException, HttpException {
        if (attemptsLeft > 1) {
            try {
                return httpPost(urlAddress, query);
            } catch (HttpException e) {
                log.warn("Failed to request XML search. Attempts left " + attemptsLeft, e);
                return httpPostWithRetry(urlAddress, query, attemptsLeft - 1);
            } catch (IOException e) {
                log.warn("Failed to request XML search. Attempts left " + attemptsLeft, e);
                return httpPostWithRetry(urlAddress, query, attemptsLeft - 1);
            }
        }
        return httpPost(urlAddress, query);
    }

    private byte[] httpPost(String urlAddress, String query)
            throws URISyntaxException, HttpException, IOException {
        HttpPost post = new HttpPost(urlAddress);
        post.setEntity(new StringEntity(query, "UTF-8"));
        post.addHeader(COOKIE_HEADER_NAME, I_AM_NOT_A_HACKER_COOKIE);

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (CloseableHttpResponse response = getHttpClient().execute(post)) {
            if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
                try {
                    response.getEntity().getContent().close();
                } catch (IOException e) {
                    // ignore
                }
                throw new IOException("Http connection returned status code " + response.getStatusLine().getStatusCode());
            }
            IOUtils.copy(response.getEntity().getContent(), baos);
        }
        return baos.toByteArray();
    }

    public void setXmlSearchUrl(String xmlSearchUrl) {
        this.xmlSearchUrl = xmlSearchUrl;
    }

    private String getXmlSearchUrl() {
        return xmlSearchUrl;
    }

    public void setMaxNumberOfConnections(int maxNumberOfConnections) {
        this.maxNumberOfConnections = maxNumberOfConnections;
    }

    public void setConnectionTimeout(int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    public void setSocketTimeout(int socketTimeout) {
        this.socketTimeout = socketTimeout;
    }

    public void setLogRawXmlResponse(boolean logRawXmlResponse) {
        this.logRawXmlResponse = logRawXmlResponse;
    }
}
