package ru.yandex.wmtools.common.sita;

import com.fasterxml.jackson.annotation.JsonFilter;
import com.fasterxml.jackson.annotation.JsonGetter;
import org.jetbrains.annotations.NotNull;

import ru.yandex.wmtools.common.util.enums.IntEnum;
import ru.yandex.wmtools.common.util.enums.IntEnumResolver;
import ru.yandex.wmtools.common.util.http.HttpUtils;

import java.util.List;

/**
 * Based on yweb/robot/sita/protos/ yweb/protos/spider.proto
 *
 * @author aherman
 */
@SuppressWarnings("PMD.VariableNamingConventions")
public class SitaJson {
    public static final String TURLFETCHINGRESULT_FILTER = "TUrlFetchingResult.Filter";

    // Request

    public static enum EActionType {
        AT_URL_EXTENSION,
        AT_URL_UNMIRRORING,
        AT_URL_INFO_READING,
        AT_URL_CANONIZATION,
        AT_URL_RICH_CONTENT_EXTRACTION,
        AT_URL_FETCHING,
        AT_MIRRORING,
        AT_URL_STATUS_CHECKING,
        AT_NEW_MIRRORING,
    }

    public static class TRequest {
        public TAuthInfo AuthInfo;
        public TActionData Data[];
        public TAction Actions[];
        public TSettings Settings;
    }

    public static class TAuthInfo {
        public final String Type = "BASIC";
        public String User;
    }

    public static class TActionData {
        public String Url;
    }

    public static enum EUrlValidator {
        NO_URL_VALIDATOR,
        ROBOT_URL_VALIDATOR
    }

    public static class TSettings {
        public Long MaxDataAge;
        public EUrlValidator UrlValidator;
        public TKiwiSettings Kiwi;
        public TZoraSettings Zora;
        public TZoraSettings OnlineZora;
    }

    public static enum WebmasterDatasourceName {
        wm_onlinezora_15s,
        wm_onlinezora_40s
    }

    public static class TZoraSettings {
        public int Timeout;
        public int Freshness;
        public Integer IfModifiedSince;
        public String UserAgent;
        public String User;
        public WebmasterDatasourceName DataSourceName;
    }

    public static class TKiwiSettings {

        // Branch name. E.g. "TRUNK" or "0".
        public String Branch;

        // Specifies whether to read data from several replicas and merge them.
        public Boolean MergeReplicas;

        // Specifies how many deltas to use during reading from Kiwi.
        // Kiwi doesn't know which delta contains requested attribute
        // so in practice only 0 (no deltas) and UINT32_MAX (all deltas) make sense.
        public Long UseDeltas;

        // Maximal allowed timestamp of attribute(s) == revision.
        public Long Timestamp;

        // Maximal allowed data age (in seconds).
        public Long MaxAge;

        // Max Kiwi-requests retries count.
        public Long MaxRetries;

        // Name of datasource, that will be used for request processing (from Sita's config).
        public String DataSourceName;
    }

    ;

    public static class TAction {
        public EActionType Type = EActionType.AT_URL_FETCHING;
        public TUrlFetchingData UrlFetchingData;
        public TUrlInfoReadingData UrlInfoReadingData;
        public TMirroringData MirroringData;
        public SitaUrlStatusCheckingJson.TUrlStatusCheckingData UrlStatusCheckingData;
        public TNewMirroringData NewMirroringData;
    }

    public static class TUrlFetchingData {
        public boolean CheckIfAllowed;
        public String DocumentFormat;
        public String RobotsTxtFormat;
    }

    public static class TUrlInfoReadingData {
        public TKiwiAttributeFilter KiwiAttributeFilters[];
        public String KiwiQuery;
        public String DataType;
    }

    public static class TNewMirroringData {
        public ENewMirroringAction Action;
        public List<String> Hosts;
    }

    public enum ENewMirroringAction {
        UNSTICK,
        MOVE,
    }

    public static class TNewMirroringResult{
        public ENewMirroringAction Action;
        public TNewMirroringResponse Response;
    }

    public static class TNewMirroringResponse {
        public ENewMirroringActionStatus Status;
        public List<TNewMirroringResponseResult> Results;
    }

    public static class TNewMirroringResponseResult {
        public ENewMirroringHostStatus Status;
        public String Host;
        public String NewMain;
        public String Target;
        public Integer HttpCode;
    }

    public enum ENewMirroringActionStatus {
        OK,
        ERROR_USER,
        ERROR_INTERNAL,
        ERROR_TIMEOUT,
    }

    public enum ENewMirroringHostStatus {
        OK,
        ERROR_BAD_REDIRECT,
        ERROR_BAD_HOST_DIRECTIVE,
        ERROR_NOT_MAIN,
        ERROR_DIFFERENT_CONTENT,
        ERROR_BAD_HTTP_CODE,
        ERROR_UNKNOWN,
        ERROR_HOST_NOT_PROCESSED,
    }

    public static class TMirroringData {
        public EMirroringAction Action;
        public String NewMainHost;      // FOR RERANK ONLY
        public String[] Hosts;          // FOR STICK, UNSTICK
    }

    public static enum EMirroringAction {
        RERANK,
        STICK,
        UNSTICK
    }

    public static class TKiwiAttributeFilter {
        public String Name;
        public Long Timestamp;
        public String Branch;
    }

    // Response

    public static class TResponse {
        public TActionResult[] Results;
        public TError[] Errors;
    }

    public static class TActionResult {
        public EActionType Type;
        public TActionData Data;

        public TError[] Errors;

        public TUrlFetchingResult UrlFetchingResult;
        public TUrlInfoReadingResult UrlInfoReadingResult;
        public TMirroringResult MirroringResult;
        public SitaUrlStatusCheckingJson.TUrlStatusCheckingResult UrlStatusCheckingResult;
        public TUrlExtensionResult UrlExtensionResult;
        public TNewMirroringResult NewMirroringResult;
    }

    public static class TMirroringResult {
        public EMirroringAction Action;
        public TMirroringResponse Response;
    }

    public static enum EMirroringActionStatus {
        OK,                           // запрос будет выполнен
        ERROR_INTERNAL,               // ошибка, то что попросил пользователь нельзя сделать
        ERROR_USER,                   // проблема на стороне зеркальщика
        ERROR_TIMEOUT                 // таймаут на стороне зеркальщика (60 сек на проверку всей группы)
    }

    public static class TMirroringResponse {
        public EMirroringActionStatus Status;
        public TMirroringHostResponse[] Results;
    }

    public static enum EMirroringHostStatus {
        OK,                                     // хост прошёл проверку
        ERROR,                                  // прочие проблемы
        ERROR_HOST_NOT_PROCESSED,               // для хоста не были получены результаты проверки
        ERROR_NOT_UNSTICKED,                    // у хоста, запрошенного на отклейку, не меняется главное
        ERROR_BAD_HOST_DIRECTIVE_AT_NEW_MAIN,   // у нового главного неправильная директива "host:" в robots.txt
        ERROR_BAD_REAL_MAIN,                    // для хоста указано неправильное главное зеркало
    }

    public static class TMirroringHostResponse {
        public EMirroringHostStatus Status;
        public String Host;
        public String NewMain;
    }

    public static class TUrlInfoReadingResult {
        public TKiwiRecord KiwiObject;
    }

    public static class TKiwiRecord {
        public String Key;
        public long Keytype;
        public TKiwiTuple Tuples[];
    }

    public static class TKiwiTuple {
        public long AttrId;
        public String AttrName;
        public long BranchId;
        public String BranchName;
        public long TimeStamp;
        public EKiwiAttrType Type;
        public String RawData;
        public String StringData;
        public long DataHash;
        public TKiwiTupleInfo Info;
    }

    public static enum EKiwiAttrType {
        AT_UNDEF,
        AT_I8,
        AT_UI8,
        AT_I16,
        AT_UI16,
        AT_I32,
        AT_UI32,
        AT_I64,
        AT_UI64,
        AT_FLOAT,
        AT_DOUBLE,
        AT_STRING,
        AT_BLOB,
        AT_BOOL,
        AT_TOP_K_BLOB,
        AT_GENERIC_TABLE,
        AT_TABLE_NOKEY_UI32,
        AT_TABLE_NOKEY_UI64
    }

    public static class TKiwiTupleInfo {
        public TKiwiStatus Status;
        public TKiwiSummary Summary;
    }

    public static class TKiwiStatus {
        public String Label;
        public EKiwiExecStatus ExecStatus;
        public String Info;
        public Double ExecTimeMilliSec;
    }

    public static class TKiwiSummary {
        public int Failures;
        public EKiwiExecStatus LastErrorStatus;
        public String LastErrorDescr;
    }

    public static enum EKiwiExecStatus {
        SUCCESS,
        UNKNOWN_FUNCTION,
        UDF_FAILED,                 //
        UDF_TIMEOUT,                // started but didn't finish
        NOT_ENOUGH_TIME,            // didn't even start
        PARAMETER_NOT_FOUND,        //
        UDF_EXCEPTION,              // exception caught
        UDF_DISABLED,               // udf is disabled
        UNSERIALIZABLE_VALUE,       // udf returned a value that cannot be serialized
        UDFLIB_VERSION_MISMATCH,    // the udf .so was built with the different

        // version of udflib.h
        UDF_CRASHED,
        UDF_NOT_LOADED,
        INVALID_TYPE,
        INVALID_DATA,
        CANNOT_CAST,
        DATA_NOT_FOUND,
        INVALID_NODE,
        DIVISION_BY_ZERO,
        DOMAIN_ERROR,
        INVALID_TABLE_TYPE,
        DATA_IS_STUB,
        CALCULATION_FAILED,
        RESULT_IS_TOO_LONG
    }

    public static enum EErrorSource {
        SITA,
        MSGBUS,
        KIWI,
        ZORA,
        MIRRORS_DB,
        QUOTA,
        GEMINI,
        STICKER
    }

    public static class TError {
        public EErrorSource Source;
        public String Description;

        public TSitaError SitaError;
        public TMsgBusError MsgBusError;
        public TKiwiError KiwiError;
        public TZoraError ZoraError;
        public TMirrorsDbError MirrorsDbError;
        public TQuotaError QuotaError;
        public TGeminiError GeminiError;
        public TStickerError StickerError;
    }

    public static enum ESitaErrorCode {
        UNKNOWN_REQUEST,        // Non-sita's request received.
        MALFORMED_REQUEST,      // Request contains no actions.
        AUTH_ERROR,             // Authentication error.
        UNSUPPORTED_ACTION,     // Current instance of Sita doesn't support action.
        INVALID_PIPE,           // Invalid pipe between actions.
        INVALID_URL,
        INVALID_USERNAME,
        UNKNOWN_DATA_TYPE,
        INVALID_QUERY,          // When KwHen returns INVALID_QUERY.

        METADATA_ERROR,
        ZORA_DATA_ERROR,

        URL_NOT_FOUND,
        INCOMPLETE_RESPONSE,

        QUOTA_ERROR,

        TIMEOUT_EXCEEDED,
        REDIRECTIONS_LOOP_FOUND,
    }

    public static class TSitaError {
        public ESitaErrorCode Code;
    }

    public static class TMsgBusError {
        public Integer MessageStatus;
    }

    public static enum EKiwiErrorType {
        READ_ERROR,
        DATA_ERROR,
        WRITE_ERROR,
        CALC_ERROR,
    }

    public static class TKiwiReadError {
        public Integer ReadStatus;
        public Integer ErrorCode;
    }

    public static enum EKiwiDataError {
        DE_NO_ENOUGH_ATTRIBUTES, // DEPRECATED
        DE_UNKNOWN_ATTRIBUTE,
        DE_CANT_UPDATE_METADATA,
        DE_UNKNOWN_BRANCH,
        DE_URL_NOT_FOUND,
        DE_METADATA_ERROR,
        DE_UNKNOWN_UDF,
        DE_DATA_NOT_FOUND,
    }

    public static class TKiwiWriteError {
        public Integer WriteError;
        public Integer ErrorCode;
    }

    public static class TKiwiCalcError {
        public Integer CalcStatus;
    }

    public static class TKiwiError {
        public EKiwiErrorType Type;
        public TKiwiReadError ReadError;
        public EKiwiDataError DataError;
        public TKiwiWriteError WriteError;
        public TKiwiCalcError CalcError;
    }

    public static enum EZoraErrorType {
        FETCH_ERROR,
        DATA_ERROR,
    }

    public static enum EZoraDataError {
        DE_INVALID_REDIRECT, // E.g. REDIR status and non-redirect HTTP code
        DE_URL_NOT_FOUND,
        DE_INVALID_LOGEL,
        DE_UNKNOWN_USER_AGENT, // User agent not from BotInfoArr in yweb/robot/robots_txt/robotstxtcfg.h.
        DE_UNKNOWN_LANG_REGION, // Lang region not from yweb/robot/spiderlib/types/langregion.h
    }

    enum EZoraFetchStatus {
        OK,              // got HTML content, Logel contains TDocLogel
        REJECT,          // fetcher did not attempt to crawl because queue is full, Logel is empty
        UPDATE,          // failed to crawl, Logel contains TUpdUrlLogel to explain why
        DISALLOW,        // failed due to robots disallow (or spam), Logel is empty
        REDIR,           // page is redirect, Logel contains TRedirLogel
        ROBOTS,          // response for http://www.host.ru/robots.txt, Logel contain TRobotsLogel
        FETCHERROR,      // http code clarifies error
        UNAVAILABLE,     // Server Unavailable
        ACCEPTED,        // Special query accepted (for example, host information has been renewed), Logel is empty
    }

    public static class TZoraError {
        public EZoraErrorType Type;
        public EZoraFetchStatus FetchStatus;
        public EZoraDataError DataError;
    }

    public static enum EMirrorsErrorCode {
        CANT_UPDATE_DB,
        URL_NOT_FOUND,
    }

    public static class TMirrorsDbError {
        public EMirrorsErrorCode Code;
    }

    // TReason in yweb/robot/quotalib/quota_common/distr_request_status.h
    public static enum EQuotaErrorCode implements IntEnum {
        UNKNOWN(-1),

        AllRight(0),
        SrcNameNotFound(1),
        ReqTypeNotFound(2),
        IpNotFound(3),
        ReqTypeQuotaExceeded(4),
        IpQuotaExceeded(5),
        ReplyingIsNotPermited(6),
        FirstPrecalcStatsIsNotReceived(7),;

        private final int value;

        private EQuotaErrorCode(int value) {
            this.value = value;
        }

        @Override
        public int value() {
            return value;
        }

        public static final IntEnumResolver<EQuotaErrorCode> R = IntEnumResolver.r(EQuotaErrorCode.class);
    }

    public static class TQuotaError {
        // See TReason in yweb/robot/quotalib/quota_common/distr_request_status.h
        public Integer Code;

        @JsonGetter
        public EQuotaErrorCode getReadableErrorCode() {
            if (Code == null) {
                return EQuotaErrorCode.UNKNOWN;
            }
            return EQuotaErrorCode.R.fromValueOrUnknown(Code);
        }
    }

    public static class TGeminiError {
        public static enum EErrorType {
            BAD_URL,
            URL_NOT_FOUND,
            URL_NOT_FOUND_ERROR,
            INNER_MB_ERROR,
            ALL_REPLICA_DOWN,
            QUOTA_EXCEEDED,
            TIMEOUT_EXCEEDED,
            ET_TRY_AGAIN,
            ET_RETRIES_EXCEEDED
        }

        public EErrorType Code;
    }

    public static class TStickerError {
        public EStickerErrorType Type;
        public EStickerDataError DataError;
        public EStickerHttpError HttpError;
    }

    public static enum EStickerErrorType {
        DATA_ERROR,
        HTTP_ERROR
    }

    public static enum EStickerDataError {
        DE_INVALID_JSON
    }

    public static enum EStickerHttpError {
        HE_RESOLVE_ERROR,
        HE_BAD_HTTP_CODE
    }

    ;

    public static class TDiagInfo {
        public TZoraDiagInfo ZoraInfo;
        public TSpiderDiagInfo SpiderInfo;
        public TLogicdocDiagInfo LogicdocInfo;
    }

    public static class TZoraDiagInfo {
        public Long GotRequest;
        public Long SentRequest;
        public Long GotReply;
        public Long SentReply;

        public String HostName;
    }

    public static class TSpiderDiagInfo {
        public Long GotRequest;
        public Long StartSend;
        public Long FinishSend;
        public Long StartReceive;
        public Long FinishReceive;
        public Long StartCalc;
        public Long FinishCalc;
        public Long SentReply;

        public String HostName;
        public String FetcherName;
    }

    public static class TLogicdocDiagInfo {
        public Long Received;
        public Long WorkStarted;
        public Long WorkFinished;
        public Long Sent;
    }

    @JsonFilter(TURLFETCHINGRESULT_FILTER)
    public static class TUrlFetchingResult {
        public Boolean IsUrlAllowed;
        public String Document;
        public String RobotsTxt;
        public Long Ip4;
        public Integer HttpCode;
        public String RedirTarget;
        public Integer MimeType;
        public Integer Encoding;
        public Integer Language;
        public TDiagInfo Times;

        @JsonGetter
        @NotNull
        public YandexCharset getReadableEncoding() {
            if (Encoding == null) {
                return YandexCharset.UNKNOWN;
            }
            return YandexCharset.R.fromValueOrUnknown(Encoding);
        }

        @JsonGetter
        @NotNull
        public YandexDocumentLanguage getReadableLanguage() {
            if (Language == null) {
                return YandexDocumentLanguage.UNKNOWN;
            }
            return YandexDocumentLanguage.R.fromValueOrUnknown(Language);
        }

        @JsonGetter
        @NotNull
        public YandexMimeType getReadableMimeType() {
            if (MimeType == null) {
                return YandexMimeType.UNKNOWN;
            }
            return YandexMimeType.R.fromValueOrUnknown(MimeType);
        }

        /**
         * Debug print http header for document
         */
        @JsonGetter
        public String getDocumentHttpHeader() {
            if (Document == null) {
                return null;
            }
            return HttpUtils.getHttpHeader(Document);
        }
    }

    public static class TUrlExtensionResult {
        public List<TUrlExtensionRedirection> Redirections;
    }

    public static class TUrlExtensionRedirection {
        public String Url;
        public Integer HttpCode;
    }
}
