package ru.yandex.wmtools.common.sita;

import java.net.URI;
import java.util.List;

/**
 * User: azakharov
 * Date: 18.03.14
 * Time: 13:01
 */
public class SitaMirroringRequest {
    private final URI uri;
    private final MirroringActionEnum action;
    private final URI newMainMirror;
    private final List<URI> hosts;

    /**
     * Не используется в клиентском коде.
     * См. фабричные методы createRerankRequest, createUnstickRequest, createStickRequest
     *
     * @param uri               хост, над которым выполняется действие
     * @param action            тип действия
     * @param newMainMirror     новое главное зеркало (только для RERANK)
     * @param hosts             список хостов группы  (для STICK, UNSTICK)
     */
    private SitaMirroringRequest(URI uri, MirroringActionEnum action, URI newMainMirror, List<URI> hosts) {
        this.uri = uri;
        this.action = action;
        this.newMainMirror = newMainMirror;
        this.hosts = hosts;
    }

    public URI getUri() {
        return uri;
    }

    public MirroringActionEnum getAction() {
        return action;
    }

    public URI getNewMainMirror() {
        return newMainMirror;
    }

    public List<URI> getHosts() {
        return hosts;
    }


    /**
     * Создать запрос на переранжирование
     *
     * @param currentBaseMainMirror     старое главное зеркало
     * @param newMainMirror             новое главное зеркало
     * @return                          объект запроса
     */
    public static SitaMirroringRequest createRerankRequest(URI currentBaseMainMirror, URI newMainMirror) {
        // Data->Url              - главное зеркало по текущей базе зеркал
        // MirroringData->NewMain - новое главное зеркало в группе
        // MirroringData->Hosts   - null

        return new SitaMirroringRequest(currentBaseMainMirror, MirroringActionEnum.RERANK, newMainMirror, null);
    }

    /**
     * Создать запрос на отклеивание хоста
     *
     * @param currentBaseMainMirror     старое главное зеркало
     * @param hosts                 хосты, которые необходимо отклеить
     * @return                      объект запроса
     */
    public static SitaMirroringRequest createUnstickRequest(URI currentBaseMainMirror, List<URI> hosts) {
        return new SitaMirroringRequest(currentBaseMainMirror, MirroringActionEnum.UNSTICK, null, hosts);
    }

    /**
     * Создать запрос на склейку хоста с группой
     *
     * @param currentBaseMainMirror     старое главное зеркало
     * @param hosts                 хосты, которые необходимо склеить
     * @return                      объект запроса
     */
    public static SitaMirroringRequest createStickRequest(URI currentBaseMainMirror, List<URI> hosts) {
        return new SitaMirroringRequest(currentBaseMainMirror, MirroringActionEnum.STICK, null, hosts);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        SitaMirroringRequest request = (SitaMirroringRequest) o;

        if (action != request.action) return false;
        if (hosts != null ? !hosts.equals(request.hosts) : request.hosts != null) return false;
        if (newMainMirror != null ? !newMainMirror.equals(request.newMainMirror) : request.newMainMirror != null)
            return false;
        if (!uri.equals(request.uri)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = uri.hashCode();
        result = 31 * result + action.hashCode();
        result = 31 * result + (newMainMirror != null ? newMainMirror.hashCode() : 0);
        result = 31 * result + (hosts != null ? hosts.hashCode() : 0);
        return result;
    }
}
