package ru.yandex.wmtools.common.sita;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.codahale.metrics.MetricRegistry;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.params.HttpConnectionParams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.util.collections.Cf;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.sita.newmirroring.SitaNewMirroringJsonRequest;
import ru.yandex.wmtools.common.sita.newmirroring.SitaNewMirroringJsonResponse;
import ru.yandex.wmtools.common.sita.newmirroring.SitaNewMirroringRequest;
import ru.yandex.wmtools.common.sita.newmirroring.SitaNewMirroringResponse;
import ru.yandex.wmtools.common.sita.url.status.checking.SitaUrlStatusCheckingJsonRequest;
import ru.yandex.wmtools.common.sita.url.status.checking.SitaUrlStatusCheckingJsonResponse;
import ru.yandex.wmtools.common.sita.url.status.checking.SitaUrlStatusCheckingRequest;
import ru.yandex.wmtools.common.sita.url.status.checking.SitaUrlStatusCheckingResponse;
import ru.yandex.wmtools.common.sita.urlextension.SitaUrlExtensionJsonRequest;
import ru.yandex.wmtools.common.sita.urlextension.SitaUrlExtensionJsonResponse;
import ru.yandex.wmtools.common.sita.urlextension.SitaUrlExtensionRequest;
import ru.yandex.wmtools.common.sita.urlextension.SitaUrlExtensitonResponse;

/**
 * @author aherman
 */
public class SitaService {
    private static final Logger log = LoggerFactory.getLogger(SitaService.class);

    public static final Charset ISO8859_1 = Charset.forName("ISO8859-1");
    public static final Charset UTF_8 = Charset.forName("UTF-8");

    private URI sitaUri;
    private String sitaProxyUserName;
    private Map<UserAgentEnum, String> userAgentToSitaUserName;
    private int connectionTTLSeconds = 20;
    private int socketTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(17);
    private int connectionTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(2);
    private int connectionRequestTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(1);

    private MetricRegistry metricRegistry;
    private CloseableHttpClient httpClient;

    public void init() {
        ConnectionConfig connectionConfig = ConnectionConfig.custom()
                .setCharset(UTF_8)
                .build();

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(connectionRequestTimeoutMillis)
                .setConnectTimeout(connectionTimeoutMillis)
                .setSocketTimeout(socketTimeoutMillis)
                .build();

        httpClient = HttpClients.custom()
                .setConnectionTimeToLive(connectionTTLSeconds, TimeUnit.SECONDS)
                .setDefaultConnectionConfig(connectionConfig)
                .setDefaultRequestConfig(requestConfig)
                .setMaxConnPerRoute(16)
                .setUserAgent(UserAgentEnum.WEBMASTER.getValue())
                .build();
    }

    public void destroy() {
        try {
            httpClient.close();
        } catch (IOException e) {
            log.error("Unable to stop httpClient", e);
        }
    }

    public SitaUrlFetchResponse request(SitaUrlFetchRequest urlFetchRequest) throws InternalException {
        return request(urlFetchRequest, false);
    }

    public SitaUrlFetchResponse request(SitaUrlFetchRequest urlFetchRequest, boolean retainJsonRequestResponse) throws InternalException {
        metricRegistry.meter("monitoring.sita.totalCalls").mark();
        String sitaUserName = getSitaUserName(urlFetchRequest);
        try {
            String jsonRequest = SitaUrlFetchJsonRequest.toJson(urlFetchRequest, sitaUserName, sitaProxyUserName);
            if (retainJsonRequestResponse) {
                urlFetchRequest.setRequestJson(jsonRequest);
            }

            HttpPost httpRequest = new HttpPost(sitaUri);
            httpRequest.setEntity(new StringEntity(jsonRequest));
            configureSocketTimeout(urlFetchRequest, httpRequest);
            HttpResponse httpResponse = httpClient.execute(httpRequest);
            log.info("Sita status code: {}", httpResponse.getStatusLine().getStatusCode());
            HttpEntity entity = httpResponse.getEntity();
            try (InputStreamReader reader = new InputStreamReader(entity.getContent(), ISO8859_1)) {
                SitaUrlFetchResponse result = SitaUrlFetchJsonResponse.parse(reader, retainJsonRequestResponse);
                metricRegistry.meter("monitoring.sita.successCalls").mark();
                return result;
            }
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Sita error", e);
        }
    }

    public SitaUrlFetchExtendedResponse requestExtended(SitaUrlFetchRequest urlFetchRequest, boolean retainJsonRequestResponse) throws InternalException {
        metricRegistry.meter("monitoring.sita.totalCalls").mark();
        String sitaUserName = getSitaUserName(urlFetchRequest);
        try {
            String jsonRequest = SitaUrlFetchJsonRequest.toJson(urlFetchRequest, sitaUserName, sitaProxyUserName);
            if (retainJsonRequestResponse) {
                urlFetchRequest.setRequestJson(jsonRequest);
            }

            HttpPost httpRequest = new HttpPost(sitaUri);
            httpRequest.setEntity(new StringEntity(jsonRequest));
            configureSocketTimeout(urlFetchRequest, httpRequest);
            HttpResponse httpResponse = httpClient.execute(httpRequest);
            log.info("Sita status code: {}", httpResponse.getStatusLine().getStatusCode());
            HttpEntity entity = httpResponse.getEntity();
            try (InputStreamReader reader = new InputStreamReader(entity.getContent(), ISO8859_1)) {
                SitaUrlFetchExtendedResponse result = SitaUrlFetchJsonResponse.parseExtended(reader, retainJsonRequestResponse);
                metricRegistry.meter("monitoring.sita.successCalls").mark();
                return result;
            }
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Sita error", e);
        }
    }

    public SitaUrlInfoReadingResponse request(SitaUrlInfoReadingRequest request) throws InternalException {
        metricRegistry.meter("monitoring.sita.totalCalls").mark();
        try {
            String jsonRequest = SitaUrlInfoReadingJsonRequest.toJson(request, sitaProxyUserName);

            HttpPost httpRequest = new HttpPost(sitaUri);
            httpRequest.setEntity(new StringEntity(jsonRequest));
            configureSocketTimeout(request, httpRequest);
            HttpResponse httpResponse = httpClient.execute(httpRequest);
            log.info("Sita status code: {}", httpResponse.getStatusLine().getStatusCode());
            HttpEntity entity = httpResponse.getEntity();
            try (InputStreamReader reader = new InputStreamReader(entity.getContent(), ISO8859_1)) {
                SitaUrlInfoReadingResponse result = SitaUrlInfoReadingJsonResponse.parse(reader);
                metricRegistry.meter("monitoring.sita.successCalls").mark();
                return result;
            }
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Sita error", e);
        }
    }

    public URI getMainMirrorCurrentBase(final URI uri) throws InternalException {

        final SitaUrlInfoReadingRequest query = new SitaUrlInfoReadingRequest(
                uri,
                Cf.list(new KiwiAttributeFilter("MirrorMain", null, "TRUNK"),
                        new KiwiAttributeFilter("IsMultiLang", null, "TRUNK"),
                        new KiwiAttributeFilter("MirrorMain", null, "RUS")),
                KiwiDataTypeEnum.DT_HOST,
                SitaRequestTimeout.DEFAULT);
        final SitaUrlInfoReadingResponse response;
        try {
            response = request(query);
        } catch (SitaKiwiKeyNotFoundException e) {
            return uri;
        }

        if (response.getTuples() == null || response.getTuples().size() != 3) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Empty attribute value");
        }

        String trunkMainMirror = null;
        String rusMainMirror = null;
        boolean isMultiLang = false;
        for (KiwiAttributeTuple t : response.getTuples()) {

            if ("MirrorMain".equals(t.getAttributeName())) {
                if ("TRUNK".equals(t.getBranchName())) {
                    trunkMainMirror = t.getRawData();
                } else if ("RUS".equals(t.getBranchName())) {
                    rusMainMirror = t.getRawData();
                } else {
                    log.debug("UNKNOWN branch name for attribute MirrorMain. Suppose it was TRUNK");
                    if (trunkMainMirror == null) {
                        trunkMainMirror = t.getRawData();
                    }
                }
            } else if ("IsMultiLang".equals(t.getAttributeName())) {
                if (!StringUtils.isEmpty(t.getRawData())) {
                    try {
                        isMultiLang = "\u0001".equals(t.getRawData());
                    } catch (NumberFormatException e) {
                        log.error("unable to parse IsMultiLang value: {}", t.getRawData());
                    }
                }
            }
        }

        String mainMirror = isMultiLang ? rusMainMirror : trunkMainMirror;

        if (StringUtils.isEmpty(mainMirror) || "null".equalsIgnoreCase(mainMirror)) {
            return uri;
        }

        try {
            final URI mirrorUri = new URI(mainMirror);
            return mirrorUri.normalize();
        } catch (URISyntaxException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Uri syntax error", e);
        }
    }

    public SitaMirroringResponse request(SitaMirroringRequest query) throws InternalException {
        metricRegistry.meter("monitoring.sita.totalCalls").mark();
        try {
            String jsonRequest = SitaMirroringJsonRequest.toJson(query, sitaProxyUserName);

            HttpPost httpRequest = new HttpPost(sitaUri);
            httpRequest.setEntity(new StringEntity(jsonRequest));

            HttpConnectionParams.setSoTimeout(httpRequest.getParams(), 180250);

            HttpResponse httpResponse = httpClient.execute(httpRequest);
            log.info("Sita status code: {}", httpResponse.getStatusLine().getStatusCode());
            HttpEntity entity = httpResponse.getEntity();
            try (InputStreamReader reader = new InputStreamReader(entity.getContent(), ISO8859_1)) {
                SitaMirroringResponse result = SitaMirroringJsonResponse.parse(reader);
                metricRegistry.meter("monitoring.sita.successCalls").mark();
                return result;
            }
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Sita error", e);
        }
    }

    public SitaUrlStatusCheckingResponse request(SitaUrlStatusCheckingRequest query) throws InternalException {
        metricRegistry.meter("monitoring.sita.totalCalls").mark();
        try {
            String jsonRequest = SitaUrlStatusCheckingJsonRequest.toJson(query, sitaProxyUserName);

            HttpPost httpRequest = new HttpPost(sitaUri);
            httpRequest.setEntity(new StringEntity(jsonRequest));

            HttpConnectionParams.setSoTimeout(httpRequest.getParams(), 180250);

            HttpResponse httpResponse = httpClient.execute(httpRequest);
            log.info("Sita status code: {}", httpResponse.getStatusLine().getStatusCode());
            HttpEntity entity = httpResponse.getEntity();
            try (InputStreamReader reader = new InputStreamReader(entity.getContent(), ISO8859_1)) {
                SitaUrlStatusCheckingResponse result = SitaUrlStatusCheckingJsonResponse.parse(reader);
                metricRegistry.meter("monitoring.sita.successCalls").mark();
                return result;
            }
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Sita error", e);
        }
    }

    public SitaUrlExtensitonResponse urlExtensionRequest(SitaUrlExtensionRequest query) throws InternalException {
        metricRegistry.meter("monitoring.sita.totalCalls").mark();
        try {
            String jsonRequest = SitaUrlExtensionJsonRequest.toJson(query, sitaProxyUserName);

            HttpPost httpRequest = new HttpPost(sitaUri);
            httpRequest.setEntity(new StringEntity(jsonRequest));

            HttpConnectionParams.setSoTimeout(httpRequest.getParams(), 180250);

            HttpResponse httpResponse = httpClient.execute(httpRequest);
            log.info("Sita status code: {}", httpResponse.getStatusLine().getStatusCode());
            HttpEntity entity = httpResponse.getEntity();
            try (InputStreamReader reader = new InputStreamReader(entity.getContent(), ISO8859_1)) {
                SitaUrlExtensitonResponse result = SitaUrlExtensionJsonResponse.parse(reader);
                metricRegistry.meter("monitoring.sita.successCalls").mark();
                return result;
            }
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Sita error", e);
        }
    }

    public SitaNewMirroringResponse newMirroringRequest(SitaNewMirroringRequest request) throws InternalException {
        metricRegistry.meter("monitoring.sita.totalCalls").mark();
        try {
            String jsonRequest = SitaNewMirroringJsonRequest.toJson(request, sitaProxyUserName);

            HttpPost httpRequest = new HttpPost(sitaUri);
            httpRequest.setEntity(new StringEntity(jsonRequest));

            HttpConnectionParams.setSoTimeout(httpRequest.getParams(), 180250);

            HttpResponse httpResponse = httpClient.execute(httpRequest);
            log.info("Sita status code: {}", httpResponse.getStatusLine().getStatusCode());
            HttpEntity entity = httpResponse.getEntity();
            try (InputStreamReader reader = new InputStreamReader(entity.getContent(), ISO8859_1)) {
                SitaNewMirroringResponse result = SitaNewMirroringJsonResponse.parse(reader);
                metricRegistry.meter("monitoring.sita.successCalls").mark();
                return result;
            }
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Sita error", e);
        }
    }

    private void configureSocketTimeout(SitaUrlFetchRequest urlFetchRequest, HttpPost httpRequest) {
        int timeoutMillis = 0;

        if (urlFetchRequest.getRequestTimeout() != SitaRequestTimeout.DEFAULT) {
            switch (urlFetchRequest.getRequestTimeout()) {
                case _15_SECONDS:
                    timeoutMillis = 15000;
                    break;
                case _40_SECONDS:
                    timeoutMillis = 40000;
                    break;
            }

            if (urlFetchRequest.getCheckAllowedInRobotsTxt()
                    || urlFetchRequest.getRobotsTxtFormat() != RobotsTxtFormatEnum.RF_NO_ROBOTS_TXT) {
                timeoutMillis = timeoutMillis * 2;
            }
        } else if (urlFetchRequest.getSitaSocketTimeoutMillis() > 0) {
            timeoutMillis = urlFetchRequest.getSitaSocketTimeoutMillis();
        }

        if (timeoutMillis > 0) {
            timeoutMillis = timeoutMillis + 2500; // 2500 millis for unexpected timeouts
            log.info("Custom socket timeout: " + timeoutMillis);
            HttpConnectionParams.setSoTimeout(httpRequest.getParams(), timeoutMillis);
        }
    }

    private void configureSocketTimeout(SitaUrlInfoReadingRequest urlInfoReadingRequest, HttpPost httpRequest) {
        int timeoutMillis = 0;

        if (urlInfoReadingRequest.getRequestTimeout() != SitaRequestTimeout.DEFAULT) {
            switch (urlInfoReadingRequest.getRequestTimeout()) {
                case _15_SECONDS:
                    timeoutMillis = 15000;
                    break;
                case _40_SECONDS:
                    timeoutMillis = 40000;
                    break;
            }

        } else if (urlInfoReadingRequest.getSitaSocketTimeoutMillis() > 0) {
            timeoutMillis = urlInfoReadingRequest.getSitaSocketTimeoutMillis();
        }

        if (timeoutMillis > 0) {
            timeoutMillis = timeoutMillis + 2500; // 2500 millis for unexpected timeouts
            log.info("Custom socket timeout: " + timeoutMillis);
            HttpConnectionParams.setSoTimeout(httpRequest.getParams(), timeoutMillis);
        }
    }

    private String getSitaUserName(SitaUrlFetchRequest request) {
        String sitaUserName = userAgentToSitaUserName.get(request.getUserAgent());
        if (sitaUserName != null) {
            return sitaUserName;
        }
        return userAgentToSitaUserName.get(UserAgentEnum.ROBOT);
    }

    @Required
    public void setConnectionTTLSeconds(int connectionTTLSeconds) {
        this.connectionTTLSeconds = connectionTTLSeconds;
    }

    @Required
    public void setSocketTimeoutMillis(int socketTimeoutMillis) {
        this.socketTimeoutMillis = socketTimeoutMillis;
    }

    @Required
    public void setConnectionTimeoutMillis(int connectionTimeoutMillis) {
        this.connectionTimeoutMillis = connectionTimeoutMillis;
    }

    @Required
    public void setSitaUri(URI sitaUri) {
        this.sitaUri = sitaUri;
    }

    @Required
    public void setSitaProxyUserName(String sitaProxyUserName) {
        this.sitaProxyUserName = sitaProxyUserName;
    }

    @Required
    public void setUserAgentToSitaUserName(Map<UserAgentEnum, String> userAgentToSitaUserName) {
        this.userAgentToSitaUserName = userAgentToSitaUserName;
    }

    @Required
    public void setMetricRegistry(MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }
}
