package ru.yandex.wmtools.common.sita;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.Charset;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.ReaderInputStream;
import org.apache.http.HttpResponse;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.util.http.HttpUtils;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * @author aherman
 */
public class SitaUrlFetchResponse {
    private final Boolean isAllowedInRobotsTxt;
    private final Long ip4;
    private final Integer sitaHttpStatusCode;
    private final String redirTarget;

    private final String rawDocument;
    private final String robotsTxtContent;
    private final String originalHttpHeader;
    private final HttpResponse parsedHttpHeaders;
    private final Integer serverHttpStatusCode;
    private final Long responseTime;
    private final YandexCharset charset;
    private final YandexDocumentLanguage language;
    private final YandexMimeType mimeType;

    private final boolean errorState;

    private SitaRedirectInfo sitaRedirectInfo = SitaRedirectInfo.NO_REDIRECTS;

    private String responseJson;

    public SitaUrlFetchResponse(Boolean allowedInRobotsTxt, Long ip4, Integer sitaHttpStatusCode, String redirTarget,
            String rawDocument, String robotsTxtContent, String originalHttpHeader, HttpResponse parsedHttpHeaders,
            Integer serverHttpStatusCode, boolean errorState, Long responseTime,
            @NotNull YandexCharset charset, @NotNull YandexDocumentLanguage language, @NotNull YandexMimeType mimeType)
    {
        this.isAllowedInRobotsTxt = allowedInRobotsTxt;
        this.ip4 = ip4;
        this.sitaHttpStatusCode = sitaHttpStatusCode;
        this.redirTarget = redirTarget;
        this.rawDocument = rawDocument;
        this.robotsTxtContent = robotsTxtContent;
        this.originalHttpHeader = originalHttpHeader;
        this.parsedHttpHeaders = parsedHttpHeaders;
        this.serverHttpStatusCode = serverHttpStatusCode;
        this.errorState = errorState;
        this.responseTime = responseTime;
        this.charset = charset;
        this.language = language;
        this.mimeType = mimeType;
    }

    public Boolean isAllowedInRobotsTxt() {
        return isAllowedInRobotsTxt;
    }

    public Long getIp4() {
        return ip4;
    }

    public YandexHttpStatus getSitaHttpStatus() {
        if (sitaHttpStatusCode == null) {
            return YandexHttpStatus.UNKNOWN;
        }

        return YandexHttpStatus.parseCode(sitaHttpStatusCode);
    }

    public String getRedirTarget() {
        return redirTarget;
    }

    public Integer getSitaHttpStatusCode() {
        return sitaHttpStatusCode;
    }

    public boolean hasDocument() {
        return rawDocument != null;
    }

    public HttpResponse getParsedHttpHeaders() {
        if (!hasDocument()) {
            throw new SitaException("Unable to read http document headers");
        }
        return parsedHttpHeaders;
    }

    public HttpResponse getDocument() {
        if (!hasDocument()) {
            throw new SitaException("Unable to read http document");
        }
        try {
            return HttpUtils.parse(
                    new ReaderInputStream(new StringReader(rawDocument), SitaService.ISO8859_1)
            );
        } catch (Exception e) {
            throw new SitaException("Unable to read http document", e);
        }
    }

    public String getRobotsTxtContent() {
        return robotsTxtContent;
    }

    public String getHttpHeader() {
        if (originalHttpHeader == null) {
            return null;
        }
        return HttpUtils.cleanHttpHeader(originalHttpHeader);
    }

    public Integer getServerHttpStatusCode() {
        return serverHttpStatusCode;
    }

    public boolean isErrorState() {
        return errorState;
    }

    public Long getResponseTime() {
        return responseTime;
    }

    @NotNull
    public YandexCharset getCharset() {
        return charset;
    }

    @NotNull
    public YandexDocumentLanguage getLanguage() {
        return language;
    }

    @NotNull
    public YandexMimeType getMimeType() {
        return mimeType;
    }

    @Nullable
    public Reader getDocumentContent() throws InternalException {
        if (rawDocument == null) {
            return null;
        }
        boolean gzipEncoded = HttpUtils.isGzipEncoded(getDocument());
        return HttpUtils.getResponseContent(getDocument(), gzipEncoded);
    }

    @Nullable
    public Reader getDocumentContent(Charset charset) throws InternalException {
        if (rawDocument == null) {
            return null;
        }
        boolean gzipEncoded = HttpUtils.isGzipEncoded(getDocument());
        return HttpUtils.getResponseContent(getDocument(), gzipEncoded, charset);
    }

    @Nullable
    public InputStream getDocumentContentStream() throws InternalException {
        if (rawDocument == null) {
            return null;
        }
        boolean gzipEncoded = HttpUtils.isGzipEncoded(getDocument());
        return HttpUtils.getResponseContentStream(getDocument(), gzipEncoded);
    }

    @Nullable
    public String getDocumentContentAsString() throws IOException, InternalException {
        Reader reader = getDocumentContent();
        if (reader == null) {
            return null;
        }
        return IOUtils.toString(reader);
    }

    @Nullable
    public String getDocumentContentAsString(Charset charset) throws IOException, InternalException {
        Reader reader = getDocumentContent(charset);
        if (reader == null) {
            return null;
        }
        return IOUtils.toString(reader);
    }

    public void setSitaRedirectInfo(SitaRedirectInfo sitaRedirectInfo) {
        this.sitaRedirectInfo = sitaRedirectInfo;
    }

    public SitaRedirectInfo getSitaRedirectInfo() {
        return sitaRedirectInfo;
    }

    public String getResponseJson() {
        return responseJson;
    }

    public void setResponseJson(String responseJson) {
        this.responseJson = responseJson;
    }
}
