package ru.yandex.wmtools.common.sita;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.ser.impl.SimpleBeanPropertyFilter;
import com.fasterxml.jackson.databind.ser.impl.SimpleFilterProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * User: azakharov
 * Date: 28.02.14
 * Time: 19:12
 */
public class SitaUrlInfoReadingJsonResponse {
    private static final Logger log = LoggerFactory.getLogger(SitaUrlInfoReadingJsonResponse.class);

    private static final ObjectMapper OM = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private static final ObjectWriter LOG_OBJECT_WRITER;

    static {
        SimpleFilterProvider filterProvider = new SimpleFilterProvider();
        filterProvider.addFilter(SitaJson.TURLFETCHINGRESULT_FILTER,
                SimpleBeanPropertyFilter.serializeAllExcept("Document", "RobotsTxt"));
        LOG_OBJECT_WRITER = OM.disable(SerializationFeature.WRITE_NULL_MAP_VALUES).writer(filterProvider);
    }

    public static SitaUrlInfoReadingResponse parse(Reader reader) throws IOException {
        SitaJson.TResponse response = OM.readValue(reader, SitaJson.TResponse.class);
        return parseJson(response);
    }

    private static SitaUrlInfoReadingResponse parseJson(SitaJson.TResponse response) throws JsonProcessingException {
        log.debug("Sita response: {}", LOG_OBJECT_WRITER.writeValueAsString(response));

        if (response.Results == null || response.Results.length == 0) {
            log.error("TResponse.Results is empty");
            throw new SitaException("Empty Sita results");
        }

        SitaJson.TActionResult result = null;
        for (SitaJson.TActionResult actionResult : response.Results) {
            if (actionResult.Type != SitaJson.EActionType.AT_URL_INFO_READING) {
                log.warn("Found unknown action result: {}", actionResult.Type);
            } else {
                result = actionResult;
                break;
            }
        }

        if (result != null && result.Errors != null && result.Errors.length > 0) {
            for (SitaJson.TError error : result.Errors) {
                if (SitaJson.EErrorSource.SITA == error.Source && error.SitaError != null) {
//                    if (SitaJson.ESitaErrorCode.INCOMPLETE_RESPONSE == error.SitaError.Code) {
//                        throw new SitaIncompleteResponseException("Incomplete response");
//                    }
                } else if (SitaJson.EErrorSource.KIWI == error.Source && error.KiwiError != null) {
                    if (SitaJson.EKiwiErrorType.DATA_ERROR == error.KiwiError.Type &&
                            SitaJson.EKiwiDataError.DE_URL_NOT_FOUND == error.KiwiError.DataError) {
                        throw new SitaKiwiKeyNotFoundException("Key not found");
                    }
                }
            }
        }

        if (response.Errors != null && response.Errors.length > 0) {
            for (SitaJson.TError error : response.Errors) {
                if (error.SitaError != null && error.SitaError.Code == SitaJson.ESitaErrorCode.TIMEOUT_EXCEEDED) {
                    throw new SitaIncompleteResponseException("Timeout exceeded");
                }
            }
        }

        if (result == null
                || result.UrlInfoReadingResult == null
                || result.UrlInfoReadingResult.KiwiObject == null
                || result.UrlInfoReadingResult.KiwiObject.Tuples == null) {
            log.error("Sita url info reading result not found");
            throw new SitaException("Sita url info reading result not found");
        } else {
            List<KiwiAttributeTuple> tuples = new ArrayList<>(result.UrlInfoReadingResult.KiwiObject.Tuples.length);


            for (SitaJson.TKiwiTuple t : result.UrlInfoReadingResult.KiwiObject.Tuples) {
                final String attrName = t.AttrName;
                final String rawData = t.RawData;
                final String stringData = t.StringData;
                final String branchName = t.BranchName;
                long timeStamp = t.TimeStamp;
                final String label;
                if (t.Info != null && t.Info.Status != null) {
                    label = t.Info.Status.Label;
                } else {
                    label = null;
                }
                tuples.add(new KiwiAttributeTuple(attrName, branchName, new Date(timeStamp), rawData, stringData, label));
            }

            return new SitaUrlInfoReadingResponse(tuples);
        }
    }
}
