package ru.yandex.wmtools.common.sita.newmirroring;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.ser.impl.SimpleBeanPropertyFilter;
import com.fasterxml.jackson.databind.ser.impl.SimpleFilterProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.wmtools.common.sita.SitaException;
import ru.yandex.wmtools.common.sita.SitaJson;
import ru.yandex.wmtools.common.sita.SitaMirroringActionStatusEnum;
import ru.yandex.wmtools.common.sita.SitaMirroringHostStatusEnum;
import ru.yandex.wmtools.common.sita.SitaMirroringResponse;

import java.io.IOException;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author avhaliullin
 */
public class SitaNewMirroringJsonResponse {
    private static final Logger log = LoggerFactory.getLogger(SitaNewMirroringJsonResponse.class);

    private static final ObjectMapper OM = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private static final ObjectWriter LOG_OBJECT_WRITER;

    static {
        SimpleFilterProvider filterProvider = new SimpleFilterProvider();
        filterProvider.addFilter(SitaJson.TURLFETCHINGRESULT_FILTER,
                SimpleBeanPropertyFilter.serializeAllExcept("Document", "RobotsTxt"));
        LOG_OBJECT_WRITER = OM.disable(SerializationFeature.WRITE_NULL_MAP_VALUES).writer(filterProvider);
    }

    public static SitaNewMirroringResponse parse(Reader reader) throws IOException {
        SitaJson.TResponse response = OM.readValue(reader, SitaJson.TResponse.class);
        return parseJson(response);
    }

    static SitaNewMirroringResponse parseJson(SitaJson.TResponse response) throws IOException {
        log.debug("Sita response: {}", LOG_OBJECT_WRITER.writeValueAsString(response));

        if (response.Errors != null && response.Errors.length > 0) {
            for (SitaJson.TError error : response.Errors) {
                if (error.SitaError != null) {
                    throw new SitaException("Sita Error " + error.SitaError.Code);
                }
            }
        }

        if (response.Results == null || response.Results.length == 0) {
            log.error("TResponse.Results is empty");
            throw new SitaException("Empty Sita results");
        }

        SitaJson.TActionResult result = null;
        for (SitaJson.TActionResult actionResult : response.Results) {
            if (actionResult.Type != SitaJson.EActionType.AT_NEW_MIRRORING) {
                log.warn("Found unknown action result: {}", actionResult.Type);
            } else {
                result = actionResult;
                break;
            }
        }
        if (result == null) {
            throw new SitaException("AT_NEW_MIRRORING Result not found");
        }
        if (result.NewMirroringResult == null) {
            throw new SitaException("Empty MirroringResult");
        }
        if (result.NewMirroringResult.Response == null) {
            throw new SitaException("Empty MirroringResult.Response");
        }

        SitaJson.TNewMirroringResponse mr = result.NewMirroringResult.Response;
        if (mr.Status == null) {
            throw new SitaException("Bad Sita Response: Status or Results field is empty");
        }

        List<SitaNewMirroringResponse.Result> hostResults = new ArrayList<>();
        if (mr.Results != null) {
            for (SitaJson.TNewMirroringResponseResult hr : mr.Results) {
                if (hr.Status == null || hr.Host == null) {
                    throw new SitaException("Sita mirroring result for host has null fields");
                }

                final SitaNewMirroringHostStatus hostStatus;
                switch (hr.Status) {
                    case OK:
                        hostStatus = SitaNewMirroringHostStatus.OK;
                        break;
                    case ERROR_BAD_HOST_DIRECTIVE:
                        hostStatus = SitaNewMirroringHostStatus.ERROR_BAD_HOST_DIRECTIVE;
                        break;
                    case ERROR_BAD_HTTP_CODE:
                        hostStatus = SitaNewMirroringHostStatus.ERROR_BAD_HTTP_CODE;
                        break;
                    case ERROR_BAD_REDIRECT:
                        hostStatus = SitaNewMirroringHostStatus.ERROR_BAD_REDIRECT;
                        break;
                    case ERROR_DIFFERENT_CONTENT:
                        hostStatus = SitaNewMirroringHostStatus.ERROR_DIFFERENT_CONTENT;
                        break;
                    case ERROR_HOST_NOT_PROCESSED:
                        hostStatus = SitaNewMirroringHostStatus.ERROR_HOST_NOT_PROCESSED;
                        break;
                    case ERROR_NOT_MAIN:
                        hostStatus = SitaNewMirroringHostStatus.ERROR_NOT_MAIN;
                        break;
                    case ERROR_UNKNOWN:
                        hostStatus = SitaNewMirroringHostStatus.ERROR_UNKNOWN;
                        break;
                    default:
                        throw new SitaException("Unknown sita new mirroring host status " + hr.Status);
                }

                hostResults.add(new SitaNewMirroringResponse.Result(hr.Host, hr.HttpCode, hostStatus));
            }
        }
        SitaNewMirroringActionStatus status = null;
        if (mr.Status != null) {
            switch (mr.Status) {
                case OK:
                    status = SitaNewMirroringActionStatus.OK;
                    break;
                case ERROR_TIMEOUT:
                    status = SitaNewMirroringActionStatus.ERROR_TIMEOUT;
                    break;
                case ERROR_USER:
                    status = SitaNewMirroringActionStatus.ERROR_USER;
                    break;
                case ERROR_INTERNAL:
                    status = SitaNewMirroringActionStatus.ERROR_INTERNAL;
                    break;
                default:
                    throw new SitaException("Unknown sita new mirroring action status "+mr.Status);
            }
        }
        return new SitaNewMirroringResponse(status, hostResults);
    }

}
