package ru.yandex.wmtools.common.sita.url.status.checking;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.wmtools.common.sita.SitaException;
import ru.yandex.wmtools.common.sita.SitaIncompleteResponseException;
import ru.yandex.wmtools.common.sita.SitaJson;
import ru.yandex.wmtools.common.sita.SitaUrlStatusCheckingJson;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

/**
 * User: azakharov
 * Date: 30.04.15
 * Time: 14:30
 */
public class SitaUrlStatusCheckingJsonResponse {
    private static final Logger log = LoggerFactory.getLogger(SitaUrlStatusCheckingJsonResponse.class);

    private static final ObjectMapper OM = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private static final ObjectWriter LOG_OBJECT_WRITER;

    static {
        LOG_OBJECT_WRITER = OM.disable(SerializationFeature.WRITE_NULL_MAP_VALUES).writer();
    }

    public static SitaUrlStatusCheckingResponse parse(Reader reader) throws IOException {
        SitaJson.TResponse response = OM.readValue(reader, SitaJson.TResponse.class);
        return parseJson(response);
    }

    private static SitaUrlStatusCheckingResponse parseJson(SitaJson.TResponse response) throws JsonProcessingException {
        log.debug("Sita response: {}", LOG_OBJECT_WRITER.writeValueAsString(response));
        if (response.Results == null || response.Results.length == 0) {
            log.error("TResponse.Results is empty");
            throw new SitaException("Empty Sita results");
        }

        SitaJson.TActionResult result = null;
        for (SitaJson.TActionResult actionResult: response.Results) {
            if (actionResult.Type != SitaJson.EActionType.AT_URL_STATUS_CHECKING) {
                log.warn("Found unknown action result: {}", actionResult.Type);
            } else {
                result = actionResult;
                break;
            }
        }

        if (response.Errors != null && response.Errors.length > 0) {
            for (SitaJson.TError error : response.Errors) {
                if (error.SitaError != null && error.SitaError.Code == SitaJson.ESitaErrorCode.TIMEOUT_EXCEEDED) {
                    throw new SitaIncompleteResponseException("Timeout exceeded");
                }
            }
        }

        if (result == null
                || result.UrlStatusCheckingResult == null
                || result.UrlStatusCheckingResult.UrlStatus == null) {

            log.error("Sita url status checking result not found");
            throw new SitaException("Sita url status checking result not found");
        } else {
            return new SitaUrlStatusCheckingResponse(
                    convertUrlStatus(result.UrlStatusCheckingResult.UrlStatus),
                    extractSubstatuses(result.UrlStatusCheckingResult),
                    result.UrlStatusCheckingResult.CanonizedUrl,
                    result.UrlStatusCheckingResult.HostIsMultiLang,
                    result.UrlStatusCheckingResult.UrlOwner,
                    result.UrlStatusCheckingResult.RedirTargetUrl,
                    result.UrlStatusCheckingResult.CanonizedRedirTargetUrl,
                    result.UrlStatusCheckingResult.RobotMainContentUrl
            );
        }
    }

    public static SitaUrlCheckingUrlStatusEnum convertUrlStatus(SitaUrlStatusCheckingJson.EUrlStatus status) {
        if (status == null) {
            return null;
        }
        return SitaUrlCheckingUrlStatusEnum.valueOf(status.name());
    }

    public static List<SitaUrlCheckingExtendedStatusEnum> extractSubstatuses(SitaUrlStatusCheckingJson.TUrlStatusCheckingResult result) {
        List<SitaUrlCheckingExtendedStatusEnum> substatuses = new ArrayList<>();
        if (result.CrawledExtendedStatus != null && result.CrawledExtendedStatus.Reason != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.CrawledExtendedStatus.Reason.name()));
        }
        if (result.DeadUrlExtendedStatus != null && result.DeadUrlExtendedStatus.Reason != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.DeadUrlExtendedStatus.Reason.name()));
        }
        if (result.ErrorExtendedStatus != null && result.ErrorExtendedStatus.Reason != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.ErrorExtendedStatus.Reason.name()));
        }
        if (result.FilteredExtendedStatus != null && result.FilteredExtendedStatus.Reasons != null) {
            for (SitaUrlStatusCheckingJson.EFilteredStatusReason r : result.FilteredExtendedStatus.Reasons) {
                substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(r.name()));
            }
        }
        if (result.NotCrawlableExtendedStatus != null && result.NotCrawlableExtendedStatus.Reason != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.NotCrawlableExtendedStatus.Reason.name()));
        }
        if (result.NotDiscoveredExtendedStatus != null && result.NotDiscoveredExtendedStatus.Reason != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.NotDiscoveredExtendedStatus.Reason.name()));
        }
        if (result.NotIndexedExtendedStatus != null && result.NotIndexedExtendedStatus.Reason != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.NotIndexedExtendedStatus.Reason.name()));
        }
        if (result.PostCrawlDeletedExtendedStatus != null && result.PostCrawlDeletedExtendedStatus.Reason != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.PostCrawlDeletedExtendedStatus.Reason.name()));
        }
        if (result.SpamExtendedStatus != null && result.SpamExtendedStatus.AntispamStatus != null) {
            substatuses.add(SitaUrlCheckingExtendedStatusEnum.valueOf(result.SpamExtendedStatus.AntispamStatus.name()));
        }

        return substatuses;
    }
}
