package ru.yandex.wmtools.common.util;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import ru.yandex.common.util.xml.XmlConvertable;

/**
 * Allows to convert to XML nameable collection, each element of which implements XmlConvertable.
 * <p/>
 * User: baton
 * Date: 21.03.2007
 * Time: 14:24:40
 */
public class XmlConvertableCollectionWrapper extends XmlDataWrapper<Iterable<? extends XmlConvertable>> {
    public XmlConvertableCollectionWrapper(
            Iterable<? extends XmlConvertable> collection,
            String name,
            String... attributes
    ) {
        super(collection, name, attributes);
    }

    public XmlConvertableCollectionWrapper(Iterable<? extends XmlConvertable> collection, String name) {
        super(collection, name);
    }

    public XmlConvertableCollectionWrapper(Iterable<? extends XmlConvertable> collection) {
        super(collection);
    }

    @Override
    protected void doToXml(StringBuilder result) {
        for (XmlConvertable element : data) {
            if (element != null) {
                element.toXml(result);
            }
        }
    }

    public static <E, W extends XmlDataWrapper<E>> XmlConvertableCollectionWrapper wrap(
            Iterable<E> collection,
            Class<W> wrapClass,
            String name,
            String... attributes
    ) {
        List<W> wrappedList = wrapCollection(collection, wrapClass);
        return new XmlConvertableCollectionWrapper(wrappedList, name, attributes);
    }

    public static <E> XmlConvertableCollectionWrapper wrap(
            Iterable<E> collection,
            WrapperFactory<E> wrapperFactory,
            String name,
            String... attributes
    ) {
        List<XmlConvertable> wrappedList = wrapCollection(collection, wrapperFactory);
        return new XmlConvertableCollectionWrapper(wrappedList, name, attributes);
    }

    public static <E> List<XmlConvertable> wrapCollection(
            Iterable<E> collection,
            WrapperFactory<E> wrapperFactory) {
        List<XmlConvertable> wrappedList = new ArrayList<XmlConvertable>();
        for (E element : collection) {
            wrappedList.add(wrapperFactory.wrap(element));
        }
        return wrappedList;
    }

    public static <E, W extends XmlDataWrapper<E>> List<W> wrapCollection(
            Iterable<E> collection,
            Class<W> wrapClass
    ) {
        List<W> wrappedList = new ArrayList<W>();
        for (E element : collection) {
            try {
                Constructor<?>[] constructors = wrapClass.getConstructors();
                Constructor<W> correctConstructor = null;
                for (Constructor<?> constructor : constructors) {
                    if (constructor.getParameterTypes().length == 1 && constructor.getParameterTypes()[0].isAssignableFrom(element.getClass())) {
                        correctConstructor = wrapClass.getConstructor(constructor.getParameterTypes()[0]);
                        break;
                    }
                }
                if (correctConstructor == null) {
                    throw new IllegalArgumentException("Cannot find constructor!");
                }
                W wrapper = correctConstructor.newInstance(element);
                wrappedList.add(wrapper);
            } catch (NoSuchMethodException e) {
                throw new IllegalArgumentException("Cannot find constructor!", e);
            } catch (IllegalAccessException e) {
                throw new IllegalArgumentException("Wrong arguments for constructor!", e);
            } catch (InvocationTargetException e) {
                throw new IllegalArgumentException(e);
            } catch (InstantiationException e) {
                throw new IllegalArgumentException(e);
            }
        }
        return wrappedList;
    }
}
