package ru.yandex.wmtools.common.util.geobase;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.WritableResource;
import org.springframework.util.xml.StaxUtils;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import ru.yandex.wmtools.common.data.info.RegionInfo;
import ru.yandex.wmtools.common.data.info.RegionName;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;

/**
 * @author aherman
 */
public class GeobaseRegionSaver {
    private static final Logger log = LoggerFactory.getLogger(GeobaseRegionSaver.class);

    private static final String ELEMENT_REGION = "region";

    private static final String ATTRIBUTE_ID = "id";
    private static final String ATTRIBUTE_PARENT = "parent";
    private static final String ATTRIBUTE_GEO_TYPE = "geo_type";
    private static final String ATTRIBUTE_HIDE = "hide";
    private static final String ATTRIBUTE_TYPE = "type";
    private static final String ATTRIBUTE_RUNAME = "runame";
    private static final String ATTRIBUTE_UKNAME = "ukname";
    private static final String ATTRIBUTE_ENNAME = "enname";
    private static final String ATTRIBUTE_TRNAME = "trname";
    private static final String ATTRIBUTE_KKNAME = "kkname";
    private static final String ATTRIBUTE_BYNAME = "byname";

    public static void save(Collection<RegionInfo> regions, WritableResource resource) throws InternalException {
        try {
            XMLEventWriter xmlEventWriter = XMLOutputFactory.newFactory().createXMLEventWriter(
                    new BufferedOutputStream(resource.getOutputStream()));
            XMLStreamWriter eventStreamWriter = StaxUtils.createEventStreamWriter(xmlEventWriter);

            eventStreamWriter.writeStartDocument();
            eventStreamWriter.writeStartElement("regions");
            for (RegionInfo regionInfo: regions) {
                saveRegion(regionInfo, eventStreamWriter);
            }
            eventStreamWriter.writeEndElement();
            eventStreamWriter.writeEndDocument();
            eventStreamWriter.flush();
            eventStreamWriter.close();
        } catch (IOException e) {
            log.error("Unable to save regions", e);
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Unable to save regions cache", e);
        } catch (XMLStreamException e) {
            log.error("Unable to save regions", e);
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Unable to save regions cache", e);
        }
    }

    private static void saveRegion(RegionInfo regionInfo, XMLStreamWriter eventStreamWriter) throws XMLStreamException {
        eventStreamWriter.writeStartElement(ELEMENT_REGION);
        eventStreamWriter.writeAttribute(ATTRIBUTE_ID, String.valueOf(regionInfo.getRealId()));
        eventStreamWriter.writeAttribute(ATTRIBUTE_PARENT, String.valueOf(regionInfo.getParent()));
        eventStreamWriter.writeAttribute(ATTRIBUTE_GEO_TYPE, String.valueOf(regionInfo.getGeotype()));
        eventStreamWriter.writeAttribute(ATTRIBUTE_HIDE, String.valueOf(regionInfo.getHide()));
        eventStreamWriter.writeAttribute(ATTRIBUTE_TYPE, String.valueOf(regionInfo.getType()));

        RegionName regionName = regionInfo.getRegionName();
        if (regionName != null) {
            if (regionName.getRuName() != null) {
                addName(eventStreamWriter, ATTRIBUTE_RUNAME, regionName.getRuName());
                addName(eventStreamWriter, ATTRIBUTE_UKNAME, regionName.getUkName());
                addName(eventStreamWriter, ATTRIBUTE_ENNAME, regionName.getEnName());
                addName(eventStreamWriter, ATTRIBUTE_TRNAME, regionName.getTrName());
            }
        }
        eventStreamWriter.writeEndElement();
    }

    private static void addName(XMLStreamWriter eventStreamWriter, String attributeName, String localizedRegionName) throws XMLStreamException {
        if (localizedRegionName != null) {
            eventStreamWriter.writeAttribute(attributeName, localizedRegionName);
        }
    }

    public static List<RegionInfo> load(WritableResource resource)
            throws ParserConfigurationException, SAXException, IOException
    {
        List<RegionInfo> regionInfos = new ArrayList<RegionInfo>();
        SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
        saxParserFactory.setNamespaceAware(true);
        saxParserFactory.setValidating(false);
        saxParserFactory.setXIncludeAware(false);

        SAXParser saxParser = saxParserFactory.newSAXParser();
        saxParser.parse(resource.getInputStream(), new Handler(regionInfos));

        return regionInfos;
    }

    private static class Handler extends DefaultHandler {
        private final List<RegionInfo> regions;

        private Handler(List<RegionInfo> regions) {
            this.regions = regions;
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes)
                throws SAXException
        {
            if (!ELEMENT_REGION.equals(localName)) {
                return;
            }
            String idStr = attributes.getValue(ATTRIBUTE_ID);

            String ruName = attributes.getValue(ATTRIBUTE_RUNAME);
            String ukName = attributes.getValue(ATTRIBUTE_UKNAME);
            String enName = attributes.getValue(ATTRIBUTE_ENNAME);
            String trName = attributes.getValue(ATTRIBUTE_TRNAME);
            String kkName = attributes.getValue(ATTRIBUTE_KKNAME);
            String byName = attributes.getValue(ATTRIBUTE_BYNAME);

            RegionName regionName = ruName != null ? new RegionName(ruName, ukName, enName, trName, kkName, byName) : null;

            String parentStr = attributes.getValue(ATTRIBUTE_PARENT);
            String geoTypeStr = attributes.getValue(ATTRIBUTE_GEO_TYPE);
            String hideStr = attributes.getValue(ATTRIBUTE_HIDE);
            String typeStr = attributes.getValue(ATTRIBUTE_TYPE);

            RegionInfo regionInfo = new RegionInfo(Integer.parseInt(idStr),
                    Byte.parseByte(typeStr),
                    Byte.parseByte(hideStr),
                    Integer.parseInt(parentStr),
                    Byte.parseByte(geoTypeStr)
            );
            regionInfo.setRegionName(regionName);

            regions.add(regionInfo);
        }
    }
}
