package ru.yandex.wmtools.common.util.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.zip.GZIPInputStream;
import java.util.zip.Inflater;
import java.util.zip.InflaterInputStream;

import org.apache.http.Consts;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.DefaultHttpResponseFactory;
import org.apache.http.impl.entity.EntityDeserializer;
import org.apache.http.impl.entity.LaxContentLengthStrategy;
import org.apache.http.impl.io.AbstractSessionInputBuffer;
import org.apache.http.impl.io.DefaultHttpResponseParser;
import org.apache.http.io.SessionInputBuffer;
import org.apache.http.message.BasicLineParser;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpParams;
import org.apache.http.params.HttpProtocolParams;
import org.apache.http.util.EntityUtils;

import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;

/**
 * @author aherman
 */
public class HttpUtils {
    private static final String X_YANDEX_HEADER_PREFIX = "X-Yandex-";

    private static final String CRLF = "\r\n";
    private static final String CRLFCRLF = "\r\n\r\n";

    private static final int GZIP_FIRST_MAGIC_BYTE = 0x1F;
    private static final int GZIP_SECOND_MAGIC_BYTE = 0x8B;

    private static final String GZIP_ENC = "gzip";
    private static final String GZIP_MIME= "application/x-gzip";
    private static final String DEFLATE_ENC = "deflate";

    private static final String UTF_8 = "UTF-8";

    public static HttpResponse parse(InputStream is) throws IOException, HttpException {
        HttpParams params = new BasicHttpParams();
        HttpProtocolParams.setHttpElementCharset(params, UTF_8);
        SessionInputBuffer buffer = new InputStreamBuffer(is, params);
        DefaultHttpResponseParser parser = new DefaultHttpResponseParser(buffer,
                BasicLineParser.DEFAULT,
                new DefaultHttpResponseFactory(),
                params);
        HttpResponse httpResponse = parser.parse();
        EntityDeserializer entityDeserializer = new EntityDeserializer(new LaxContentLengthStrategy());
        HttpEntity entity = entityDeserializer.deserialize(buffer, httpResponse);
        httpResponse.setEntity(entity);
        return httpResponse;
    }

    public static String getHttpHeader(String httpDocument) {
        int headerEnd = httpDocument.indexOf(CRLFCRLF);
        if (headerEnd == -1) {
            return null;
        }
        return httpDocument.substring(0, headerEnd);
    }

    public static String cleanHttpHeader(String httpHeader) {
        StringBuilder sb = new StringBuilder(httpHeader.length());
        int position = 0;
        int nextPosition;
        do {
            nextPosition = httpHeader.indexOf(CRLF, position);
            String line;
            if (nextPosition > 0) {
                line = httpHeader.substring(position, nextPosition);
                position = nextPosition + CRLF.length();
            } else {
                line = httpHeader.substring(position);
            }
            if (!line.startsWith(X_YANDEX_HEADER_PREFIX)) {
                sb.append(line).append(CRLF);
            }
        } while (nextPosition > 0);
        return sb.toString();
    }

    private static class InputStreamBuffer extends AbstractSessionInputBuffer {
        private InputStreamBuffer(InputStream is, HttpParams httpParams) {
            super();
            init(is, 128, httpParams);
        }

        @Override
        public boolean isDataAvailable(int timeout) throws IOException {
            return true;
        }
    }

    public static boolean isGzipEncoded(HttpResponse httpResponse) throws InternalException {
        HttpEntity entity = httpResponse.getEntity();
        if (entity == null) {
            return false;
        }

        Header contentType = entity.getContentType();
        Header contentEncoding = entity.getContentEncoding();
        boolean gzipContentEncoding = isGzipped(contentType, contentEncoding);

        if (!gzipContentEncoding) {
            try {
                InputStream content = entity.getContent();
                int firstByte = content.read();
                if (firstByte == GZIP_FIRST_MAGIC_BYTE) {
                    int secondByte = content.read();
                    if (secondByte == GZIP_SECOND_MAGIC_BYTE) {
                        gzipContentEncoding = true;
                    }
                }
                EntityUtils.consume(entity);
            } catch (IOException e) {
                throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable get http content", e);
            }
        }
        return gzipContentEncoding;
    }

    private static Charset guessCharset(HttpResponse httpResponse) {
        HttpEntity entity = httpResponse.getEntity();
        if (entity == null) {
            return Consts.ISO_8859_1;
        }
        ContentType contentType = ContentType.getOrDefault(entity);
        Charset charset = contentType.getCharset();
        if (charset != null) {
            return charset;
        }
        return Consts.ISO_8859_1;
    }

    public static Reader getResponseContent(HttpResponse httpResponse, boolean gzipEncoded) throws InternalException {
        return getResponseContent(httpResponse, gzipEncoded, guessCharset(httpResponse));
    }

    public static Reader getResponseContent(HttpResponse httpResponse, boolean gzipEncoded, Charset charset) throws InternalException {
        return new InputStreamReader(getResponseContentStream(httpResponse, gzipEncoded), charset);
    }

    public static InputStream getResponseContentStream(HttpResponse httpResponse, boolean gzipEncoded)
            throws InternalException
    {
        HttpEntity entity = httpResponse.getEntity();
        if (entity == null) {
            return null;
        }

        InputStream content;
        try {
            content = entity.getContent();
        } catch (IOException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable get http content", e);
        }

        if (gzipEncoded) {
            try {
                content = new GZIPInputStream(content);
            } catch (IOException e) {
                throw new InternalException(InternalProblem.PROCESSING_ERROR, "Unable to read http entity content", e);
            }
        } else {
            // check for deflate encoding
            Header header = entity.getContentEncoding();
            if (header != null && DEFLATE_ENC.equals(header.getValue())) {
                content = new InflaterInputStream(content, new Inflater(true));
            }
        }

        return content;
    }

    public static boolean isGzipped(Header contentType, Header contentEncoding) {
        return isGzipped(
                contentType != null ? contentType.getValue() : null,
                contentEncoding != null ? contentEncoding.getValue() : null
        );
    }

    public static boolean isGzipped(String contentType, String contentEncoding) {
        return (contentType != null && contentType.contains(GZIP_MIME))
                || (contentEncoding != null && contentEncoding.contains(GZIP_ENC));
    }

}
