package ru.yandex.wmtools.common.util.uri;

import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.net.IDN;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Objects;
import java.util.Set;

/**
 * @author aherman
 */
public class UriUtils {
    static final String HTTP_SCHEMA = "http";
    static final String HTTPS_SCHEMA = "https";

    public static URI2 toUri(@NotNull String string, UriFeature... uriFeatures) {
        Set<UriFeature> featureSet = Collections.emptySet();
        if (uriFeatures != null && uriFeatures.length > 0) {
            featureSet = EnumSet.noneOf(UriFeature.class);
            for (UriFeature f : uriFeatures) {
                featureSet.add(f);
            }
        }
        return toUri(string, featureSet);
    }

    public static URI2 toUri(@NotNull String string, Set<UriFeature> features) {
        if (StringUtils.isEmpty(string)) {
            throw new IllegalArgumentException("Url must not be empty");
        }

        if (features.contains(UriFeature.DEFAULT_SCHEME_HTTP) && !string.contains("://")) {
            string = HTTP_SCHEMA + "://" + string;
        }

        UriComponentsBuilder builder = UriComponentsBuilder.fromUriString(string);
        boolean encoded = isEncoded(builder, builder.build(false));

        if (features.contains(UriFeature.CLEAN_AUTHORITY)) {
            builder.userInfo(null);
        }
        if (features.contains(UriFeature.CLEAN_FRAGMENT)) {
            builder.fragment(null);
        }
        if (features.contains(UriFeature.USE_PUNYCODED_HOSTNAME)) {
            String hostname = builder.build(false).getHost();
            builder.host(IDN.toASCII(hostname));
            verifyASCIIDomain(hostname);
        }

        if (features.contains(UriFeature.NORMALIZE_PLUS_IN_QUERY)) {
            normalizePlusInQuery(builder, builder.build(false), encoded);
        }

        if (!features.contains(UriFeature.DO_NOT_NORMALIZE)) {
            normalizeUri(builder, builder.build(false), encoded);
        }
        UriComponents build = builder.build(encoded);
        if (!features.contains(UriFeature.DO_NOT_NORMALIZE)) {
            build = build.normalize();
        }
        return new URI2(build);
    }

    static boolean isEncoded(UriComponentsBuilder builder, UriComponents components) {
        String oldQuery = components.getQuery();
        String newQuery = StringUtils.replace(oldQuery, "+", "%20");
        boolean replacedQuery = !Objects.equals(oldQuery, newQuery);
        if (replacedQuery) {
            builder.replaceQuery(newQuery);
        }
        boolean encoded = true;
        try {
            builder.build(true);
        } catch (IllegalArgumentException e) {
            encoded = false;
        }
        if (replacedQuery) {
            builder.replaceQuery(oldQuery);
        }
        return encoded;
    }

    static void normalizeUri(UriComponentsBuilder builder, UriComponents components, boolean encoded) {
        components = normalizeSchema(builder, components);
        components = normalizePort(builder, components);
        components = normalizeHost(builder, components);
        if (encoded) {
            components = capitalizeUrlEncode(builder, components);
        }
    }

    static UriComponents normalizeSchema(UriComponentsBuilder builder, UriComponents components) {
        if (components.getScheme() == null) {
            return components;
        }
        builder.scheme(components.getScheme().toLowerCase());
        return builder.build(false);
    }

    static UriComponents normalizePort(UriComponentsBuilder builder, UriComponents components) {
        if (components.getPort() < 0) {
            return components;
        }

        if (components.getPort() == 80 && HTTP_SCHEMA.equals(components.getScheme())) {
            builder.port(-1);
            return builder.build(false);
        }

        if (components.getPort() == 443 && HTTPS_SCHEMA.equals(components.getScheme())) {
            builder.port(-1);
            return builder.build(false);
        }

        return components;
    }

    static UriComponents normalizeHost(UriComponentsBuilder builder, UriComponents components) {
        if (components.getHost() == null) {
            return components;
        }
        builder.host(components.getHost().toLowerCase());
        return builder.build(false);
    }

    static UriComponents normalizePlusInQuery(UriComponentsBuilder builder, UriComponents components, boolean encoded) {
        String query = components.getQuery();
        if (query == null) {
            return components;
        }

        if (query.indexOf('+') >= 0) {
            if (encoded) {
                query = StringUtils.replace(query, "+", "%20");
            } else {
                query = query.replace('+', ' ');
            }
        }
        builder.replaceQuery(query);
        return builder.build(false);
    }

    static UriComponents capitalizeUrlEncode(UriComponentsBuilder builder, UriComponents components) {
        boolean changed = false;
        if (components.getPath() != null && components.getPath().indexOf('%') >= 0) {
            builder.replacePath(capitalizeUrlEncode(components.getPath()));
            changed = true;
        }

        if (components.getQuery() != null && components.getQuery().indexOf('%') >= 0) {
            builder.replaceQuery(capitalizeUrlEncode(components.getQuery()));
            changed = true;
        }

        if (changed) {
            return builder.build(true);
        } else {
            return components;
        }
    }

    static String capitalizeUrlEncode(String value) {
        StringBuilder sb = new StringBuilder(value.length());
        int inEncodedPart = 0;

        for (int i = 0; i < value.length(); i++) {
            char ch = value.charAt(i);

            if (inEncodedPart == 0) {
                if (ch == '%') {
                    inEncodedPart = 2;
                }
            } else {
                inEncodedPart--;
                if (ch >= 'a' && ch <= 'z') {
                    ch = Character.toUpperCase(ch);
                }
            }
            sb.append(ch);
        }
        return sb.toString();
    }

    static UriComponentsBuilder toBuilder(URI2 uri) {
        return toBuilder(uri.getUriComponents());
    }

    static UriComponentsBuilder toBuilder(UriComponents uriComponents) {
        return UriComponentsBuilder.newInstance()
                .scheme(uriComponents.getScheme())
                .userInfo(uriComponents.getUserInfo())
                .host(uriComponents.getHost())
                .port(uriComponents.getPort())
                .path(uriComponents.getPath())
                .query(uriComponents.getQuery())
                .fragment(uriComponents.getFragment());
    }

    public static URI2 resolveUri(URI2 baseURI, URI2 referenceURI) {
        if (referenceURI.isAbsolute()) {
            if (baseURI.getFragment() == null) {
                return referenceURI;
            }
            UriComponentsBuilder builder = toBuilder(referenceURI);
            builder.fragment(baseURI.getFragment());
            return new URI2(builder.build());
        }

        UriComponentsBuilder uriComponentsBuilder = UriComponentsBuilder.newInstance();
        uriComponentsBuilder.fragment(referenceURI.getFragment());
        uriComponentsBuilder.scheme(baseURI.getScheme());

        if (referenceURI.getHost() != null) {
            uriComponentsBuilder.userInfo(referenceURI.getUserInfo());
            uriComponentsBuilder.host(referenceURI.getHost());
            uriComponentsBuilder.port(referenceURI.getPort());
            uriComponentsBuilder.query(referenceURI.getQuery());
            return new URI2(uriComponentsBuilder.build());
        }
        uriComponentsBuilder.userInfo(baseURI.getUserInfo());
        uriComponentsBuilder.host(baseURI.getHost());
        uriComponentsBuilder.port(baseURI.getPort());

        if (StringUtils.isEmpty(referenceURI.getPath())) {
            uriComponentsBuilder.path(baseURI.getPath());
            if (StringUtils.isEmpty(referenceURI.getQuery())) {
                uriComponentsBuilder.query(baseURI.getQuery());
            } else {
                uriComponentsBuilder.query(referenceURI.getQuery());
            }

            return new URI2(uriComponentsBuilder.build());
        }
        uriComponentsBuilder.query(referenceURI.getQuery());

        if (referenceURI.getPath().startsWith("/")) {
            uriComponentsBuilder.path(referenceURI.getPath());
            return new URI2(uriComponentsBuilder.build());
        }

        uriComponentsBuilder.path(mergePath(baseURI, referenceURI));
        return new URI2(uriComponentsBuilder.build().normalize());
    }

    private static String mergePath(URI2 baseURI, URI2 referenceURI) {
        if (StringUtils.isEmpty(baseURI.getHost())) {
            return "/" + StringUtils.defaultString(referenceURI.getPath());
        }

        String basePath = StringUtils.defaultString(baseURI.getUriComponents().getPath());
        if (!basePath.isEmpty()) {
            int lastSlash = basePath.lastIndexOf('/');
            if (lastSlash < 0) {
                basePath = "";
            } else {
                basePath = basePath.substring(0, lastSlash);
            }
        }
        return basePath + "/" + StringUtils.defaultString(referenceURI.getPath());
    }

    public static URI2 punycodeToUnicode(URI2 uri) {
        UriComponentsBuilder builder = toBuilder(uri);
        builder.host(IDN.toUnicode(uri.getHost()));
        return new URI2(builder.build());
    }

    public static enum UriFeature {
        CLEAN_FRAGMENT,
        CLEAN_AUTHORITY,

        DEFAULT_SCHEME_HTTP,
        USE_PUNYCODED_HOSTNAME,

        DO_NOT_NORMALIZE,
        NORMALIZE_PLUS_IN_QUERY
    }

    public static void verifyASCIIDomain(String s) {
        if (s.endsWith(".")){
            throw new IllegalArgumentException("Domain should not end with point");
        }
        int dotsCount = 0;

        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            if (isCharDisallowedInHost(c)) {
                throw new IllegalArgumentException("Character " + c + "(" + ((int) c) + ") is not allowed in hostname");
            }
            if (c == '.') {
                dotsCount++;
            }
        }
        if (dotsCount == 0) {
            throw new IllegalArgumentException("Domain doesn't contain dots at all: '" + s + "'");
        }
    }

    private static boolean isCharDisallowedInHost(int ch) {
        return (0x0000 <= ch && ch <= 0x002C) ||
                (ch == 0x002F) ||
                (0x003A <= ch && ch <= 0x0040) ||
                (0x005B <= ch && ch <= 0x005E) ||
                (ch == 0x0060) ||
                (0x007B <= ch && ch <= 0x007F);
    }
}
