package ru.yandex.wmtools.common.util.uri;

import java.net.IDN;
import java.net.URI;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;

import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.ExtraTagNameEnum;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;

/**
 * @author aherman
 */
public class WebmasterUriUtils {
    private static final int MAX_URL_LENGTH = 1024;
    private static final int MAX_HOST_LENGTH = 255;

    private static final String HTTP_SCHEMA = "http";
    private static final String HTTPS_SCHEMA = "https";

    private static final Pattern IPV4_PATTERN = Pattern.compile("\\d+\\.\\d+\\.\\d+\\.\\d+");


    public static URI toOldUri(String uriString) throws UserException {
        uriString = StringUtils.trimToEmpty(uriString);
        if (StringUtils.isEmpty(uriString)) {
            throw new UserException(UserProblem.INVALID_URL, "Empty url");
        }

        if (uriString.length() > MAX_URL_LENGTH) {
            throw new UserException(UserProblem.URL_TOO_LONG, "Url too long: " + uriString);
        }

        try {
            URI2 uri2 = UriUtils.toUri(uriString,
                    UriUtils.UriFeature.CLEAN_AUTHORITY,
                    UriUtils.UriFeature.CLEAN_FRAGMENT,
                    UriUtils.UriFeature.DEFAULT_SCHEME_HTTP,
                    UriUtils.UriFeature.USE_PUNYCODED_HOSTNAME,
                    UriUtils.UriFeature.NORMALIZE_PLUS_IN_QUERY);

            if (!HTTP_SCHEMA.equals(uri2.getScheme()) && !HTTPS_SCHEMA.equals(uri2.getScheme())) {
                throw new UserException(UserProblem.WRONG_PROTOCOL_USED,
                        "Unexpected protocol was found in url: " + uriString,
                        new ExtraTagInfo(ExtraTagNameEnum.GIVEN, uri2.getScheme()));
            }

            validateHostname(uri2.getHost(), uriString);

            return uri2.toUri();
        } catch (IllegalArgumentException e) {
            throw new UserException(UserProblem.INVALID_URL, "Invalid url: " + uriString, e);
        }
    }

    static void validateHostname(String hostname, String uriString) throws UserException {
        if (hostname == null) {
            throw new UserException(UserProblem.INVALID_URL, "Host name is empty: " + uriString,
                    new ExtraTagInfo(ExtraTagNameEnum.URL, uriString));
        }
        if (hostname.length() > MAX_HOST_LENGTH) {
            throw new UserException(UserProblem.HOST_NAME_TOO_LONG, "Hostname too long: " + IDN.toUnicode(hostname));
        }

        // IPv4
        if (IPV4_PATTERN.matcher(hostname).matches()) {
            throw new UserException(UserProblem.IP_ADDRESSES_FORBIDDEN,
                    "IP address was used instead of hostname: " + hostname);
        }
        //IPv6
        if (hostname.startsWith("[")) {
            throw new UserException(UserProblem.IP_ADDRESSES_FORBIDDEN,
                    "IP address was used instead of hostname: " + hostname);
        }
    }
}
