#pragma once

#ifndef _WMC_APPLICATION_H_INCLUDED_
#define _WMC_APPLICATION_H_INCLUDED_

#include <iostream>
#include <vector>
#include <string>
#include <functional>

#include <Poco/Util/Application.h>
#include <Poco/Util/Option.h>
#include <Poco/Util/OptionSet.h>
#include <Poco/Util/HelpFormatter.h>
#include <Poco/Util/AbstractConfiguration.h>
#include <Poco/LoggingFactory.h>
#include <Poco/Formatter.h>
#include <Poco/Instantiator.h>

#include "wmconsole/legacy/util/logutil.h"
#include "wmconsole/legacy/util/pattern_formatter.h"
#include "wmconsole/legacy/util/string_utils.h"

using Poco::Util::Application;
using Poco::Util::Option;
using Poco::Util::OptionSet;
using Poco::Util::HelpFormatter;
using Poco::Util::AbstractConfiguration;
using Poco::Util::OptionCallback;
using Poco::LoggingFactory;
using Poco::Formatter;
using Poco::Instantiator;

namespace NWebmaster {

class WMCApplication : public Application {
private:
//    static const std::string OP_MYSQL_URL;
//    static const std::string OP_MYSQL_HOST_URLS;
//    static const std::string OP_MYSQL_MOVED_TABLES;
protected:
    static const std::string OP_USER_DB_URL;
    static const std::string OP_HOST_DB_URLS;
public:
    static const std::string OP_HELP;
    static const std::string OP_CONFIG_FILE;
    static const std::string OP_LOG;

    WMCApplication() : fShowHelp(false) {
        LoggingFactory::defaultFactory().registerFormatterClass(
            "WMCPatternFormatter",
            new Instantiator<WMCPatternFormatter, Formatter>);
    }

    ~WMCApplication() override {}

    void print_keys() const {
        std::vector<std::string> keys;
        config().keys(keys);
        for (std::vector<std::string>::const_iterator iter = keys.begin(); iter != keys.end(); ++iter) {
            WMCLOG_INFO(logger(), *iter << " = " << config().getString(*iter, ""));
        }
    }

protected:

    void initialize(Application &self) override {
        loadConfiguration();
        Application::initialize(self);
    }

    void defineHostDbOptions(OptionSet &options) {
        options.addOption(
            Option(OP_HOST_DB_URLS, "hd",
                   "MySQL host databases URLs\n"
                   "  Format: <user>[:<password>]@<host>[:<port>]/<dbname>{,<user>[:<password>]@<host>[:<port>]/<dbname>}\n")
            .required(true)
            .repeatable(false)
            .argument("<url-list>")
            .binding(OP_HOST_DB_URLS)
        );
    }

    void defineUserDbOptions(OptionSet &options) {

        options.addOption(
            Option(OP_USER_DB_URL, "ud",
                   "MySQL user database URL\n  Format: <user>[:<password>]@<host>[:<port>]/<dbname>\n")
            .required(false)
            .repeatable(false)
            .argument("<url>")
            .binding(OP_USER_DB_URL)
        );

    }

    void defineDbOptions(OptionSet &options) {
        defineUserDbOptions(options);
        defineHostDbOptions(options);
    }


    void defineOptions(OptionSet &options) override {
        Application::defineOptions(options);

        options.addOption(
            Option(OP_HELP, "h", "Display help information on command line arguments\n")
            .required(false)
            .repeatable(false)
            .callback(OptionCallback<WMCApplication>(
                          this, &WMCApplication::handleHelp)));

        options.addOption(
            Option(OP_CONFIG_FILE, "f", "Load configuration data from a file\n")
            .required(false)
            .repeatable(true)
            .argument("file")
            .callback(OptionCallback<WMCApplication>(
                          this, &WMCApplication::handleConfig)));

        /*
                options.addOption(
                    Option("log", "l",
                            "Log file path or '-' for stdout\n"
                            "Use FILE::LEVEL for logging level. Available levels are:\n"
                            "- fatal critical error warning notice information debug trace -\n"
                            "FILE or ::LEVEL (but not both) can be omitted\n"
                    )
                    .required(false)
                    .repeatable(false)
                    .argument("<FILE::LEVEL>")
                    .binding("log")
                    .callback(OptionCallback<WMCApplication>(this, &WMCApplication::optionLogCallback))
                );
        */
    }


    void optionLogCallback(const std::string & /* name */, const std::string &value) {
        std::string logPath = value;

        size_t n = logPath.find("::");
        if (n != string::npos) {
            string level(logPath.begin() + n+2, logPath.end()); // Poco will perform validity check
            logPath.erase(n);
            config().setString("logging.loggers.root.level",  level);
        }

        if (logPath == "-") {
            config().setString("logging.loggers.root.channel",  "stdout");
        } else if (!logPath.empty()) {
            config().setString("logging.loggers.root.channel",  "logfile");
            config().setString("logging.channels.logfile.path",  logPath);
        }

    }

    void handleHelp(const std::string & /*name*/, const std::string & /*value*/) {
        fShowHelp = true;
        displayHelp();
        stopOptionsProcessing();
    }

    void handleConfig(const std::string & /*name*/, const std::string &value) {
        loadConfiguration(value);
    }

    virtual void displayHelp() {
        HelpFormatter helpFormatter(options());
        helpFormatter.setCommand(commandName());
        helpFormatter.setHeader( helpHeader() );
        helpFormatter.setUsage("OPTIONS");
        helpFormatter.format(std::cout);
    }

    virtual std::string helpHeader() const {
        return "";
    }

    std::string getUserDbUrl() const {
        return config().getString(OP_USER_DB_URL);
    }

    std::vector<std::string> getHostDbUrls() const {
        std::vector<std::string> hostUrls;
        if(config().hasOption(OP_HOST_DB_URLS)) {
            std::string hu = config().getString(OP_HOST_DB_URLS);
            parseString(hu.c_str(), ',', hostUrls);
        }
        return hostUrls;
    }

    int main(const std::vector<std::string> &args) override {
        int result = Application::EXIT_OK;

        if (!fShowHelp) {
            WMCLOG_INFO(logger(), "Started");
            result = doMain(args);
            WMCLOG_INFO(logger(), "Finished");
        }

        return result;
    }

protected:
    virtual int doMain(const std::vector<std::string> &args) = 0;

protected:
    bool fShowHelp;
};

} // namespace NWebmaster

#endif
