#pragma once

#ifndef WMCDBOPTIONS_H_INCLUDED_
#define WMCDBOPTIONS_H_INCLUDED_

//! \file db_options.h

#include <iostream>
#include <fstream>
#include <sstream>
#include <stdexcept>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include <Poco/Util/AbstractConfiguration.h>
#include <Poco/Util/Option.h>
#include <Poco/Util/OptionSet.h>
#include <Poco/Util/OptionCallback.h>

#include "wmconsole/legacy/util/string_utils.h"


namespace NWebmaster {
/*!
 * \defgroup WMCApplicationOptions
 * \brief The set of template mixin classes provides orthogonal implementation of typical cmdline options: logging, debug, DB, etc
 * \par Mixin definition from GoF
 * A mixin class is a class that's intended to provide an optional interface or
 * functionality to other classes. It's similar to an abstract class in that it's not intended to be instantiated.
 * Mixin classes require multiple inheritance
 *
 */

/*! \par Rationale:
 * Option strings below declared as static const instead of static const class members
 * to make their actual values visible for a user of the class - no need to open .cpp file
 * Additionally, design of these clases does not assume implementation files at all (headers only)
 */
static const char *const OP_USER_DB_URL = "user-db-url";
static const char *const OP_HOST_DB_URLS = "host-db-urls";



/*! \brief Mix-in (policy) class defines command-line options "user-db-url" and "host-db-urls".
 * \ingroup WMCApplicationOptions
 * Example of usage:
 * \code
 *  class MyApplication
 *      : public WMCBaseApplication
 *      , public DBOptions<MyApplication>
 *      , public LoggingOptions<MyApplication>
 *  {
 *  protected:
 *      void defineOptions(OptionSet& options) {
 *
 *          WMCBaseApplication::defineOptions(options);
 *          DBOptions<MyApplication>::defineOptions(options);
 *          LoggingOptions<MyApplication>::defineOptions(options);
 *
 *      // Add your specific options here
 *      //  options.addOption( ... );
 *      }
 *
 *      virtual int doMain(const std::vector<std::string>& args) {
 *          // Host DB
 *          vector<string> const& connVec = getConnectionStrings(); // bind to const ref
 *          for (vector<string>::const_iterator p = connVec.begin(); p != connVec.end(); ++p) {
 *              try {
 *                  log_debug << "Processing database " << *p;
 *                  Session session(Poco::Data::MySQL::Connector::KEY, *p);
 *                  // do whatever you want with session
 *              }
 *              catch // and so on
 *          }
 *
 *          // User DB:
 *          Session userSession(Poco::Data::MySQL::Connector::KEY, getUserConnectionString());
 *          // and so on
 *      }
 *
 *  };
 * \endcode
 */
template <class App>
class DBOptions {
protected:
    // non-virtual, the class is not intended for polymorphic usage
    void defineOptions(Poco::Util::OptionSet &options) {
        options.addOption(
            Poco::Util::Option(OP_USER_DB_URL, "ud",
                               "MySQL user database URL\n  Format: <user>[:<password>]@<host>[:<port>]/<dbname>\n")
            .required(false)
            .repeatable(false)
            .argument("<url>")
            .binding(OP_USER_DB_URL)
        );

        options.addOption(
            Option(OP_HOST_DB_URLS, "hd",
                   "MySQL host databases URLs\n"
                   "  Format: <user>[:<password>]@<host>[:<port>]/<dbname>{,<user>[:<password>]@<host>[:<port>]/<dbname>}\n")
            .required(false)
            .repeatable(false)
            .argument("<url-list>")
            .binding(OP_HOST_DB_URLS)
            //  .callback(OptionCallback<DBOptions>(this, &DBOptions::optionLogCallback))
        );
    }
public:
    typedef std::vector<std::string> DBUrls;

    //! @return vector of connection strings which can be used as Poco::Data::Session() parameter
    const DBUrls &hostConnectionStrings() const;
    const DBUrls &getConnectionStrings() const  {
        return hostConnectionStrings();
    }

    //! @return vector of strings which contains mysql command line options
    const DBUrls &hostCommandlineStrings() const;
    const DBUrls &getCommandlineStrings() const {
        return hostCommandlineStrings();
    }

    //! @return UserDB connection string which can be used as Poco::Data::Session() parameter
    const std::string &userConnectionString() const;
    const std::string &getUserConnectionString() const {
        return userConnectionString();
    }

    //! @return UserDB command line string
    const std::string &userCommandlineString() const;
    const std::string &getUserCommandlineString() const {
        return userCommandlineString();
    }

protected:  // mix-in class; shall not be instantiated.
    DBOptions() {}
    ~DBOptions() {}

private:
    enum { fUserDB = false, fHostDB = true };

    class DBUrlHelper : public DBUrls {
    public:
        DBUrlHelper(const App *app, bool asHostDB, bool asCommandLine) {

            DBUrls dbUrls;
            const string &opt = asHostDB ? OP_HOST_DB_URLS : OP_USER_DB_URL;
            if (app->config().hasOption(opt)) {
                std::string hu = app->config().getString(opt);
                if (hu.empty()) {
                    throw std::invalid_argument( std::string("Required argument '" + opt + "' missed"));
                }
                parseString(hu.c_str(), ',', dbUrls);
            } else {
                throw std::invalid_argument( std::string("Required argument '" + opt + "' missed"));
            }

            for (DBUrls::const_iterator p=dbUrls.begin(); p!=dbUrls.end(); ++p) {
                const string &url = *p;

                std::string::size_type slashPos = url.find_last_of('/');

                if (slashPos == std::string::npos || slashPos >= url.size() - 1) {
                    throw std::invalid_argument("Invalid MySQL url: database name missed");
                }

                std::string fName = url.substr(slashPos + 1);
                std::string::size_type atPos = url.find_last_of('@');

                if (atPos > slashPos) {
                    throw std::invalid_argument("Invalid MySQL url: host[:port] part missed");
                }

                if (atPos == 0) {
                    throw std::invalid_argument("Invalid MySQL url: user[:password] missed");
                }

                std::string hostAndPort = url.substr(atPos + 1, slashPos - atPos - 1);
                std::string::size_type colonPos = hostAndPort.find_first_of(':');

                std::string fHost = hostAndPort;
                unsigned int fPort = 3306;
                if (colonPos != std::string::npos) {
                    fHost = hostAndPort.substr(0, colonPos);
                    std::istringstream tmp(hostAndPort.substr(colonPos + 1));
                    if ( !(tmp >> fPort) ) {
                        throw std::invalid_argument("Invalid MySQL url: illegal port value");
                    }
                }

                std::string userAndPassword = url.substr(0, atPos);

                colonPos = userAndPassword.find_first_of(':');

                std::string fUser = userAndPassword;
                std::string fPassword = "";
                if (colonPos != std::string::npos) {
                    fUser = userAndPassword.substr(0, colonPos);
                    fPassword = userAndPassword.substr(colonPos + 1);
                }

                std::ostringstream os;
                if (asCommandLine) {
                    os << "-h " << fHost << " -P " << fPort << " -A " << fName << " -u " << fUser;
                    if (!fPassword.empty()) {
                        os << " -p " << fPassword;
                    }
                } else {
                    os  << "user=" << fUser << ";"
                        << "password=" << fPassword << ";"
                        << "db=" << fName << ";"
                        << "port=" << fPort << ";"
                        << "host=" << fHost << ";"
                        << "auto-reconnect=true";
                }

                push_back(os.str());
            }
        }
    };

};



template <class App>
inline std::vector<std::string> const &
DBOptions<App>::hostConnectionStrings() const {
    static const DBUrlHelper s_(static_cast<const App *>(this), fHostDB, false);
    return s_;
}

template <class App>
inline std::vector<std::string> const &
DBOptions<App>::hostCommandlineStrings() const {
    static const DBUrlHelper s_(static_cast<const App *>(this), fHostDB, true);
    return s_;
}

template <class App>
inline std::string const &
DBOptions<App>::userConnectionString() const {
    static const DBUrlHelper s_(static_cast<const App *>(this), fUserDB, false);
    static const std::string e = "";
    return s_.empty() ? e : s_[0];
}

template <class App>
inline std::string const &
DBOptions<App>::userCommandlineString() const {
    static const DBUrlHelper s_(static_cast<const App *>(this), fUserDB, true);
    static const std::string e = "";
    return s_.empty() ? e : s_[0];
}


} // namespace NWebmaster

#endif // WMCDBOPTIONS_H_INCLUDED_

