#pragma once

#ifndef EVENT_H_F2918131_13EC_4f01_A797_55061EE3EE64
#define EVENT_H_F2918131_13EC_4f01_A797_55061EE3EE64

#include <list>
#include <algorithm>
#include <type_traits>

#include <pthread.h>
#include <errno.h>

#include <util/generic/noncopyable.h>
#include <util/system/mutex.h>

#ifndef __linux__
#include <machine/atomic.h>
#endif
#include "timespec.h"
#include "wmc_assert.h"

namespace NWebmaster {
namespace threads {

namespace detail {

class event_hoder_interface {
public:
    virtual void reset_signal(int event_num) = 0;
    virtual void set_signal(int event_num) = 0;
};

} //detail

class event : public TNonCopyable {
    template <int t_size>
    friend class event_holder;
public:
    event() : flag(false) {}
    event(bool flag) : flag(flag) {}
    inline ~event();

    inline void set();
    inline void reset();
    inline bool is_setted() const;

private:
    inline void add_event_holder(int num, detail::event_hoder_interface *hldr, volatile bool &state);
    inline void remove_event_holder(int num, detail::event_hoder_interface *hldr);

    volatile bool flag;

    mutable TMutex lock_mutex;

    typedef std::pair<int, detail::event_hoder_interface *> hldr_desc_t;
    typedef std::list<hldr_desc_t> hldr_arr_t;
    hldr_arr_t holders;
};

template <int t_size>
class event_holder : public detail::event_hoder_interface, public TNonCopyable {
public:
    static const int timeout_result = -1;

    inline event_holder();
    inline ~event_holder();

    template <int t_num>
    inline void set_event(event &ev);
    inline void set_event(int num, event &ev);
    inline int wait_any() const;
    inline int wait_any(unsigned long time_ms) const;
    inline int wait_all() const;
    inline int wait_all(unsigned long time_ms) const;

private:
    inline bool check_all() const;
    inline int check_any() const;
    inline void wait_condition() const;
    inline void wait_condition(const timespec &dur) const;
    void set_signal(int event_num) override;
    void reset_signal(int event_num) override;

    event *arr[t_size];
    volatile bool state_arr[t_size];

    mutable TMutex lock_mutex;

    mutable pthread_cond_t cond;
};

event::~event() {
    WMC_CHECK(holders.size() == 0, "Logic error! All event_holders for event must be destroyed before event will be destroyed.");
}

void event::set() {
    TGuard<TMutex> guard(lock_mutex);
    if (flag) {
        return;
    }
    flag = true;
    for (hldr_desc_t &hldr : holders) {
        hldr.second->set_signal(hldr.first);
    }
}

void event::reset() {
    TGuard<TMutex> guard(lock_mutex);
    if (!flag) {
        return;
    }
    flag = false;
    for (hldr_desc_t &hldr : holders) {
        hldr.second->reset_signal(hldr.first);
    }
}

bool event::is_setted() const {
    TGuard<TMutex> guard(lock_mutex);
    return flag;
}

void event::add_event_holder(int num, detail::event_hoder_interface *hldr, volatile bool &state) {
    TGuard<TMutex> guard(lock_mutex);
    hldr_desc_t hldr_desk(num, hldr);
    holders.push_back(hldr_desk);
    state = flag;
}

void event::remove_event_holder(int num, detail::event_hoder_interface *hldr) {
    TGuard<TMutex> guard(lock_mutex);
    hldr_desc_t hldr_desk(num, hldr);
    holders.erase(std::find(holders.begin(), holders.end(), hldr_desk));
}

template <int t_size>
event_holder<t_size>::event_holder() {
    memset(&arr[0], 0, sizeof(const event *) * t_size);
    for (volatile bool &state : state_arr) {
        state = false;
    }
    WMC_CHECK(!pthread_cond_init(&cond, nullptr), "pthread_cond_init fail " + ToString(errno) + " " + strerror(errno));
}

template <int t_size>
event_holder<t_size>::~event_holder() {
    for (int num = 0; num < t_size; ++num) {
        if (arr[num]) {
            arr[num]->remove_event_holder(num, this);
        }
    }
    WMC_CHECK(!pthread_cond_destroy(&cond), "pthread_cond_destroy fail " + ToString(errno) + " " + strerror(errno));
}

template <int t_size>
template <int t_num>
void event_holder<t_size>::set_event(event &ev) {
    static_assert((t_num < t_size) && (t_num >= 0), "incorrect condition");

    if (&ev == arr[t_num]) {
        return;
    }
    TGuard<TMutex> guard(lock_mutex);
    arr[t_num] = &ev;
    ev.add_event_holder(t_num, this, state_arr[t_num]);
}

template <int t_size>
void event_holder<t_size>::set_event(int num, event &ev) {
    WMC_CHECK(((num < t_size) && (num >= 0)), "called set_event with incorrect num " + ToString(num) + " " + ToString(t_size));
    if (&ev == arr[num]) {
        return;
    }
    TGuard<TMutex> guard(lock_mutex);
    arr[num] = &ev;
    ev.add_event_holder(num, this, state_arr[num]);
}

template <int t_size>
int event_holder<t_size>::wait_any() const {
    TGuard<TMutex> guard(lock_mutex);
    int i = check_any();
    while (i == t_size) {
        wait_condition();
        i = check_any();
    }
    return i;
}

template <int t_size>
int event_holder<t_size>::wait_any(unsigned long time_ms) const {
    TGuard<TMutex> guard(lock_mutex);

    int i = check_any();
    if (i < t_size) {
        return i;
    }

    timespec start_time;
    WMC_CHECK(!clock_gettime(CLOCK_MONOTONIC, &start_time), " clock_gettime fail" + ToString(errno) + " " + strerror(errno));
    timespec full_timer = get_alarm_timer_from_ms(time_ms);

    do {
        timespec cur_time;
        WMC_CHECK(!clock_gettime(CLOCK_MONOTONIC, &cur_time), " clock_gettime fail" + ToString(errno) + " " + strerror(errno));

        cur_time = cur_time - start_time;

        if (full_timer < cur_time) {
            return timeout_result;
        }

        timespec timer = full_timer - cur_time;
        wait_condition(timer);
    } while ((i = check_any()) == t_size);
    return i;
}

template <int t_size>
int event_holder<t_size>::wait_all() const {
    TGuard<TMutex> guard(lock_mutex);
    while (!check_all()) {
        wait_condition();
    }
    return 0;
}

template <int t_size>
int event_holder<t_size>::wait_all(unsigned long time_ms) const {
    TGuard<TMutex> guard(lock_mutex);

    if (check_all()) {
        return 0;
    }

    timespec start_time;
    WMC_CHECK(!clock_gettime(CLOCK_MONOTONIC, &start_time), " clock_gettime fail" + ToString(errno) + " " + strerror(errno));

    timespec full_timer = get_alarm_timer_from_ms(time_ms);

    do {
        timespec cur_time;
        WMC_CHECK(!clock_gettime(CLOCK_MONOTONIC, &cur_time), " clock_gettime fail" + ToString(errno) + " " + strerror(errno));

        cur_time = cur_time - start_time;

        if (full_timer < cur_time) {
            return timeout_result;
        }

        timespec timer = full_timer - cur_time;
        wait_condition(timer);
    } while (!check_all());
    return 0;
}

template <int t_size>
bool event_holder<t_size>::check_all() const {
    for (bool state : state_arr) {
        if (!state) {
            return false;
        }
    }
    return true;
}

template <int t_size>
int event_holder<t_size>::check_any() const {
    for (int i = 0; i < t_size; ++i) {
        if (state_arr[i]) {
            return i;
        }
    }
    return t_size;
}

template <int t_size>
void event_holder<t_size>::wait_condition() const {
    pthread_mutex_t *native_id = static_cast<pthread_mutex_t *>(lock_mutex.Handle());
    WMC_CHECK(!pthread_cond_wait(&cond, native_id), "pthread_cond_wait fail " + ToString(errno) + " " + strerror(errno));
}

template <int t_size>
void event_holder<t_size>::wait_condition(const timespec &dur) const {
    timespec cur_time;
    WMC_CHECK(!clock_gettime(CLOCK_REALTIME, &cur_time), " clock_gettime fail " + ToString(errno) + " " + strerror(errno));
    cur_time = cur_time + dur;
    pthread_mutex_t *native_id = static_cast<pthread_mutex_t *>(lock_mutex.Handle());
    const int res = pthread_cond_timedwait(&cond, native_id, &cur_time);
    WMC_CHECK(((0 == res)||(ETIMEDOUT == res)), "pthread_cond_timedwait fail " + ToString(errno) + " " + strerror(errno));
}

template <int t_size>
void event_holder<t_size>::set_signal(int event_num) {
    TGuard<TMutex> guard(lock_mutex);
    state_arr[event_num] = true;
    WMC_CHECK(!pthread_cond_broadcast(&cond), "pthread_cond_broadcast fail " + ToString(errno) + " " + strerror(errno));
}

template <int t_size>
void event_holder<t_size>::reset_signal(int event_num) {
    TGuard<TMutex> guard(lock_mutex);
    state_arr[event_num] = false;
}

} //threads
} //namespace NWebmaster

#endif //EVENT_H_F2918131_13EC_4f01_A797_55061EE3EE64
