#include "pattern_formatter.h"

#include <Poco/Message.h>
#include <Poco/DateTimeFormat.h>
#include <Poco/DateTimeFormatter.h>
#include <Poco/DateTime.h>
#include <Poco/Timestamp.h>
#include <Poco/Timezone.h>
#include <Poco/NumberFormatter.h>
#include <Poco/Environment.h>

using Poco::Message;
using Poco::DateTimeFormat;
using Poco::DateTimeFormatter;
using Poco::DateTime;
using Poco::Timestamp;
using Poco::Timezone;
using Poco::NumberFormatter;
using Poco::Environment;

namespace NWebmaster {

const string WMCPatternFormatter::PROP_PATTERN = "pattern";
const string WMCPatternFormatter::PROP_TIMES   = "times";


WMCPatternFormatter::WMCPatternFormatter():
    _localTime(true) {
}


WMCPatternFormatter::WMCPatternFormatter(const string &format):
    _localTime(true),
    _pattern(format) {
}


WMCPatternFormatter::~WMCPatternFormatter() {
}


void WMCPatternFormatter::format(const Message &msg, string &text) {
    Timestamp timestamp = msg.getTime();

    if (_localTime) {
        timestamp += Timezone::utcOffset() * Timestamp::resolution();
        timestamp += Timezone::dst() * Timestamp::resolution();
    }

    DateTime dateTime = timestamp;
    string::const_iterator it  = _pattern.begin();
    string::const_iterator end = _pattern.end();

    while (it != end) {
        if (*it == '%') {
            if (++it != end) {
                switch (*it) {
                case 's':
                    text.append(msg.getSource());
                    break;
                case 't':
                    text.append(msg.getText());
                    break;
                case 'l':
                    text.append(NumberFormatter::format((int) msg.getPriority()));
                    break;
                case 'p':
                    text.append(getPriorityName((int) msg.getPriority()));
                    break;
                case 'q':
                    text += getPriorityName((int) msg.getPriority()).at(0);
                    break;
                case 'P':
                    text.append(NumberFormatter::format(msg.getPid()));
                    break;
                case 'T':
                    text.append(msg.getThread());
                    break;
                case 'I':
                    text.append(NumberFormatter::format(msg.getTid()));
                    break;
                case 'N':
                    text.append(Environment::nodeName());
                    break;
                case 'w':
                    text.append(DateTimeFormat::WEEKDAY_NAMES[dateTime.dayOfWeek()], 0, 3);
                    break;
                case 'W':
                    text.append(DateTimeFormat::WEEKDAY_NAMES[dateTime.dayOfWeek()]);
                    break;
                case 'b':
                    text.append(DateTimeFormat::MONTH_NAMES[dateTime.month() - 1], 0, 3);
                    break;
                case 'B':
                    text.append(DateTimeFormat::MONTH_NAMES[dateTime.month() - 1]);
                    break;
                case 'd':
                    text.append(NumberFormatter::format0(dateTime.day(), 2));
                    break;
                case 'e':
                    text.append(NumberFormatter::format(dateTime.day()));
                    break;
                case 'f':
                    text.append(NumberFormatter::format(dateTime.day(), 2));
                    break;
                case 'm':
                    text.append(NumberFormatter::format0(dateTime.month(), 2));
                    break;
                case 'n':
                    text.append(NumberFormatter::format(dateTime.month()));
                    break;
                case 'o':
                    text.append(NumberFormatter::format(dateTime.month(), 2));
                    break;
                case 'y':
                    text.append(NumberFormatter::format0(dateTime.year() % 100, 2));
                    break;
                case 'Y':
                    text.append(NumberFormatter::format0(dateTime.year(), 4));
                    break;
                case 'H':
                    text.append(NumberFormatter::format0(dateTime.hour(), 2));
                    break;
                case 'h':
                    text.append(NumberFormatter::format0(dateTime.hourAMPM(), 2));
                    break;
                case 'a':
                    text.append(dateTime.isAM() ? "am" : "pm");
                    break;
                case 'A':
                    text.append(dateTime.isAM() ? "AM" : "PM");
                    break;
                case 'M':
                    text.append(NumberFormatter::format0(dateTime.minute(), 2));
                    break;
                case 'S':
                    text.append(NumberFormatter::format0(dateTime.second(), 2));
                    break;
                case 'i':
                    text.append(NumberFormatter::format0(dateTime.millisecond(), 3));
                    break;
                case 'c':
                    text.append(NumberFormatter::format(dateTime.millisecond()/100));
                    break;
                case 'z':
                    text.append(DateTimeFormatter::tzdISO(_localTime ? Timezone::tzd() : DateTimeFormatter::UTC));
                    break;
                case 'Z':
                    text.append(DateTimeFormatter::tzdRFC(_localTime ? Timezone::tzd() : DateTimeFormatter::UTC));
                    break;
                case '[': {
                    ++it;
                    string prop;
                    while (it != end && *it != ']') {
                        prop += *it++;
                    }
                    if (it != end) {
                        ++it;
                    }
                    try {
                        text.append(msg[prop]);
                    } catch (...) {
                    }
                    break;
                }
                default:
                    text += *it;
                }
                ++it;
            }
        } else {
            text += *it++;
        }
    }
}


void WMCPatternFormatter::setProperty(const string &name, const string &value) {
    if (name == PROP_PATTERN) {
        _pattern = value;
    } else if (name == PROP_TIMES) {
        _localTime = (value == "local");
    } else {
        Formatter::setProperty(name, value);
    }
}


string WMCPatternFormatter::getProperty(const string &name) const {
    if (name == PROP_PATTERN) {
        return _pattern;
    } else if (name == PROP_TIMES) {
        return _localTime ? "local" : "UTC";
    } else {
        return Formatter::getProperty(name);
    }
}

const string &WMCPatternFormatter::getPriorityName(int priority) {
    static string priorities[] = {
        "",
        "FATAL",
        "CRITICAL",
        "ERROR",
        "WARN",
        "NOTICE",
        "INFO",
        "DEBUG",
        "TRACE"
    };

    poco_assert (1 <= priority && priority <= 8);
    return priorities[priority];
}

} // namespace NWebmaster
