#pragma once

#ifndef SITEMAPS_LOADING_INNER_23059F0E_2EA9_4870_B546_132083F83A53
#define SITEMAPS_LOADING_INNER_23059F0E_2EA9_4870_B546_132083F83A53

#include <algorithm>

#include <library/cpp/http/fetch/exthttpcodes.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <robot/library/sitemap/common/reciterators.h>
#include <robot/library/sitemap/processor/parser.h>

#include <wmconsole/legacy/util/algo.h>
#include <wmconsole/legacy/util/url_name.h>

namespace NWebmaster {
struct t_sitemap_info {
    const urlid_t &id() const {
        return _id;
    }

    friend bool operator<(const t_sitemap_info &info1, const t_sitemap_info &info2) {
        return info1.id() < info2.id();
    }

    void copy_from(const TSitemapStatRecNew &info, bool loader) {
        _id = info;
        HttpCode = info.HttpCode;
        Added = info.Added;
        LastAccess = info.LastAccess;
        LastSuccessAccess = info.LastSuccessAccess;
        if (loader) {
            UrlsCount = info.UrlsCount;
        } else {
            if ((HttpCode == HTTP_OK) || (HttpCode == 0)) {
                UrlsCount = info.UrlsCount;
            } else {
                UrlsCount = 0;
            }
        }
        Format = info.Format;
        Type = info.Type;
    }

    ui8 from_flags;

    //from TSitemapStatRecNew;
    urlid_t _id;
    ui32    HttpCode;
    TDataworkDate Added;
    TDataworkDate LastAccess;
    TDataworkDate LastSuccessAccess;
    ui32    UrlsCount;
    ui8     Format;
    ui8     Type;
};

typedef std::map<t_url_name, t_sitemap_info> t_stms_dat_map;
typedef std::list<Poco::Int64> t_stms_db_id_array; //TODO: after Java changes we can do the array as set!!!
typedef std::map<urlid_t, t_stms_db_id_array > t_smtms_indexes_map; //sitemap robot id to array of sitemaps db-id

namespace sitemaps_inner {

//----------------------------------------------------------------------
static const ui8 sitemap_source_mask = TSitemapSourceRec::ROBOTS |
                                       TSitemapSourceRec::SITEMAPINDEX |
                                       TSitemapSourceRec::WEBMASTER |
                                       TSitemapSourceRec::REDIRECT;

//----------------------------------------------------------------------
struct t_temp_sitemap_info {
    bool is_redir;
    urlid_t id;
    urlid_t id_redir;
    ui8 type;
};

//----------------------------------------------------------------------
template <typename t_dat_files_holder>
const THostRec *host_by_id(t_dat_files_holder &dat_files, ui32 hostId) {
    if (!dat_files.fhost.is_open()) {
        return nullptr;
    }

    if ((hostId & (1 << 31)) == 0) {
        // Usual host
        hostid_t id (hostId);
        const THostRec *res = dat_files.fhost().FindRecord(&id);
        if ((res) && (res->HostId == hostId)) {
            return res;
        }
    } else {
        // host from other segment
    }

    return nullptr;
}

struct normalizer_url {
    normalizer_url() {
        clear();
    }

    template <typename t_dat_files_holder>
    const t_url_name &normalize_url(t_dat_files_holder &dat_files, const TSitemapStatRecNew *rec) {
        if ((rec) && (rec->Name[0])) {
            if (last_host_id != rec->HostId) {
                istring res = get_host_name(dat_files, rec->HostId);
                if (!res.empty()) {
                    res = normalize_host(res.c_str());
                }
                last_name.clear();
                last_name.set_host(res);
                last_host_id = rec->HostId;
            }

            if (last_name.get_host().empty()) {
                return last_name;
            }

            last_name.set_url(rec->Name);
            return last_name;
        }
        return empty_name;
    }
private:
    static istring normalize_host(const char *host) {
        istring res;
        if (strstr(host, "://") == nullptr) {
            res = "http://";    // default scheme
        }
        res += host;

        THttpURL parsed;
        int code = parsed.ParseAbs(res.c_str(),
                                   THttpURL::FeaturesRobot | THttpURL::FeatureUnescapeStandard | THttpURL::FeatureEscSpace | THttpURL::FeatureEscapeUnescaped);

        if (code) {
            return res;    // can not parse -- do nothing
        }

        res = parsed.PrintS(THttpURL::FlagScheme | THttpURL::FlagHostPort);
        return res;
    }

    //----------------------------------------------------------------------
    template <typename t_dat_files_holder>
    static istring get_host_name(t_dat_files_holder &dat_files, ui32 host_id) {
        const THostRec *recHost = host_by_id(dat_files, host_id);
        if ((recHost) && (recHost->Name[0])) {
            return recHost->Name;
        }
        return istring();
    }

    void clear() {
        last_host_id = -1;
        last_name.clear();
    }
    ui32 last_host_id;
    t_url_name last_name;
    t_url_name empty_name;
};

//----------------------------------------------------------------------
template <typename t_dat_files_holder>
void load_sitemaps(t_stms_dat_map &res, ui32 host_id, t_dat_files_holder &dat_files, bool loader) {
    if ((!dat_files.fstmp_sources.is_open()) ||
            (!dat_files.fstmp_stat.is_open()) ||
            (!dat_files.fhost.is_open())
       ) {
        return;
    }

    std::map<urlid_t, t_temp_sitemap_info> sitemaps;

    const hostid_t temp_host_id(host_id);
    const urlid_t temp_url_id(host_id, 0);
    t_temp_sitemap_info temp_info;

    typename t_dat_files_holder::t_fstmp_sources &fstmp_sources = dat_files.fstmp_sources();
    typename t_dat_files_holder::t_fstmp_stat &fstmp_stat = dat_files.fstmp_stat();

    //load links sitemaps
    std::set<urlid_t> redir_urls;
    temp_info.is_redir = true;
    if (dat_files.fstmp_links.is_open()) {
        typename t_dat_files_holder::t_fstmp_links &fstmp_links = dat_files.fstmp_links();
        for (const TNamedLink *rec = fstmp_links.FindRecord(&temp_url_id);
                ((rec) && (rec->SrcHostId == host_id));
                rec = fstmp_links.Next()
            ) {
            urlid_t id(host_id, rec->SrcUrlId);
            const TSitemapSourceRec *rec_sources = fstmp_sources.FindRecord(&id);
            if ((rec_sources) && (host_id == rec_sources->HostId) && (rec->SrcUrlId == rec_sources->UrlId)) {
                temp_info.id = id;
                temp_info.id_redir = urlid_t(rec->DstHostId, rec->DstUrlId);
                temp_info.type = ((rec_sources->SitemapSources | TSitemapSourceRec::REDIRECT) & sitemap_source_mask);
                sitemaps.insert(std::make_pair(id, temp_info));
                redir_urls.insert(temp_info.id_redir);
            }
        }
    }

    //load allowed sitemaps
    temp_info.is_redir = false;
    if (dat_files.fstmp_allowed.is_open()) {
        typename t_dat_files_holder::t_fstmp_allowed &fstmp_allowed = dat_files.fstmp_allowed();
        for (const TWebmasterAllowedSitemapRec *rec = fstmp_allowed.FindRecord(&temp_host_id);
                ((rec) && (rec->ValidatedHostId == host_id));
                rec = fstmp_allowed.Next()
            ) {
            urlid_t id(rec->HostId, rec->UrlId);
            std::map<urlid_t, t_temp_sitemap_info>::iterator iter = sitemaps.find(id);
            if (iter != sitemaps.end()) {
                continue;
            }
            if (redir_urls.end() != redir_urls.find(id)) {
                continue;
            }

            const TSitemapSourceRec *rec_sources = fstmp_sources.FindRecord(&id);
            if ((rec_sources) && (id == urlid_t(*rec_sources))) {
                temp_info.id = id;
                temp_info.type = ((rec_sources->SitemapSources | TSitemapSourceRec::WEBMASTER) & sitemap_source_mask);
                sitemaps.insert(std::make_pair(id, temp_info));
            }
        }
    }

    //if (!sitemaps.empty())
    //{
    //    typedef std::pair<urlid_t, t_temp_sitemap_info> t_pair_info;
    //    for (const t_pair_info& val : sitemaps)
    //    {
    //        log_info << " sitemap: " << val.first.HostId << " " << val.first.UrlId << " " << val.second.id.HostId << " " << val.second.id.UrlId;
    //    }
    //}

    if (!sitemaps.empty()) {
        t_sitemap_info temp_info;
        t_url_name temp_url;
        normalizer_url norm;

        //translate sitemaps to TSitemapStatRecNew
        for (const TSitemapStatRecNew *rec = fstmp_stat.FindRecord(&temp_url_id);
                ((rec) && (rec->HostId == host_id));
                rec = fstmp_stat.Next()
            ) {
            std::map<urlid_t, t_temp_sitemap_info>::iterator iter = sitemaps.find(*rec);
            if ((sitemaps.end() != iter) && (iter->second.is_redir == false)) {
                temp_url = norm.normalize_url(dat_files, rec);
                if (!temp_url.empty()) {
                    temp_info.from_flags = iter->second.type;
                    temp_info.copy_from(*rec, loader);
                    res[temp_url] = temp_info;
                }
            }
        }

        typedef std::pair<urlid_t, t_temp_sitemap_info> t_pair_info;
        const TSitemapStatRecNew *rec = nullptr;
        for (const t_pair_info &val : sitemaps) {
            if (true == val.second.is_redir) {
                rec = fstmp_stat.FindRecord(&(val.second.id));
                if ((!rec) || (!(val.second.id == (*rec)))) {
                    continue;
                }
                temp_url = norm.normalize_url(dat_files, rec);
                if (temp_url.empty()) {
                    continue;
                }
                temp_info.copy_from(*rec, loader);
                temp_info.from_flags = val.second.type;
                res[temp_url] = temp_info;

                rec = fstmp_stat.FindRecord(&(val.second.id_redir));
                if ((!rec) || (!(val.second.id_redir == (*rec)))) {
                    continue;
                }
                temp_url = norm.normalize_url(dat_files, rec);
                if (temp_url.empty()) {
                    continue;
                }
                temp_info.copy_from(*rec, loader);
                temp_info.from_flags = val.second.type;
                res[temp_url] = temp_info;
            } else if (val.second.id.HostId != host_id) {
                rec = fstmp_stat.FindRecord(&(val.second.id));
                if ((!rec) || (!(val.second.id == (*rec)))) {
                    continue;
                }
                temp_url = norm.normalize_url(dat_files, rec);
                if (temp_url.empty()) {
                    continue;
                }
                temp_info.from_flags = val.second.type;
                temp_info.copy_from(*rec, loader);
                res[temp_url] = temp_info;
            }
        }
    }
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder>
void load_sitemap_indexes(const urlid_t &id,
                          t_dat_files_holder &hldr,
                          Poco::Int64 &db_id,
                          t_smtms_indexes_map &indexes_map) {
    using namespace NRobotSitemaps;

    if (hldr.fstmp_indexes.is_open()) {
        typename t_dat_files_holder::t_fstmp_indexes &fstmp_indexes = hldr.fstmp_indexes();
        const TSitemapUrlsRec *rec = fstmp_indexes.FindRecord(&id);
        if ((!rec) || (!(id == (*rec)))) {
            return;
        }
        for (TUrlsIterator url_iter(rec); url_iter.Current() != nullptr ; url_iter.Next()) {
            urlid_t url_id(url_iter.Current()->HostId, url_iter.Current()->UrlId);

            t_smtms_indexes_map::iterator iter = indexes_map.find(url_id);

            if (indexes_map.end() == iter) {
                t_stms_db_id_array array;
                array.push_back(db_id);
                indexes_map.insert(std::make_pair(url_id, array));
            } else {
                //TODO: remove the code after t_stms_db_id_array will change to set
                t_stms_db_id_array &array = iter->second;
                bool find = false;
                for (const Poco::Int64 &x : array) {
                    if (x == db_id) {
                        find = true;
                        break;
                    }
                }
                if (find) {
                    continue;
                }
                if (iter->first.HostId == id.HostId) {
                    iter->second.push_front(db_id);
                } else {
                    iter->second.push_back(db_id);
                }
            }
        }
    }
}

//----------------------------------------------------------------------
template <typename t_dict>
int encode_error(int err, const t_dict &tbl_dic_sitemap_error_type) {
    using namespace NRobotSitemaps;

    int code = -1;

    switch (err) {
    case WRN_UNKNOWN_TAG:
        code = tbl_dic_sitemap_error_type["unknown tag"];
        break;
    case WRN_TAG_CANNOT_BE_EMPTY:
        code = tbl_dic_sitemap_error_type["tag cannot be empty"];
        break;
    case WRN_TAG_CANNOT_BE_COMPLEX:
        code = tbl_dic_sitemap_error_type["tag can not be complex"];
        break;
    case WRN_TAG_DATA_TOO_LONG:
        code = tbl_dic_sitemap_error_type["tag data too long"];
        break;
    case WRN_TAG_EMPTY_DATA:
        code = tbl_dic_sitemap_error_type["tag empty data"];
        break;
    case WRN_PRIORITY_INVALID:
        code = tbl_dic_sitemap_error_type["bad priority format"];
        break;
    case WRN_LASTMOD_INVALID:
        code = tbl_dic_sitemap_error_type["lastmod invalid"];
        break;
    case WRN_TOO_MANY_WARNINGS:
        code = tbl_dic_sitemap_error_type["too much warnings"];
        break;

    case ERR_URL_TOO_LONG:
        code = tbl_dic_sitemap_error_type["too long url"];
        break;
    case ERR_URL_INVALID:
        code = tbl_dic_sitemap_error_type["not a url"];
        break;
    case ERR_UNKNOWN_TAG:
        code = tbl_dic_sitemap_error_type["err unknown tag"];
        break;
    case ERR_TAG_CANNOT_BE_EMPTY:
        code = tbl_dic_sitemap_error_type["impotant tag cannot be empty"];
        break;
    case ERR_TAG_APPEARS_TWICE:
        code = tbl_dic_sitemap_error_type["tag founded twice"];
        break;
    case ERR_TAG_CANNOT_BE_COMPLEX:
        code = tbl_dic_sitemap_error_type["impotant tag can not be complex"];
        break;
    case ERR_TAG_DATA_TOO_LONG:
        code = tbl_dic_sitemap_error_type["impotant tag data too long"];
        break;
    case ERR_TAG_EMPTY_DATA:
        code = tbl_dic_sitemap_error_type["impotant tag empty data"];
        break;
    case ERR_INCOMPLETE_URL_DATA:
        code = tbl_dic_sitemap_error_type["tag not found"];
        break;
    case ERR_HOST_NOT_CONFIRM:
        code = tbl_dic_sitemap_error_type["bad url base"];
        break;

    case ERR_INVALID_TEXT_FORMAT:
        code = tbl_dic_sitemap_error_type["bad format"];
        break;
    case ERR_INVALID_XML:
        code = tbl_dic_sitemap_error_type["incorrect xml"];
        break;
    case ERR_INVALID_MAIN_TAG:
        code = tbl_dic_sitemap_error_type["invalid main tag"];
        break;
    case ERR_TOO_MANY_URLS:
        code = tbl_dic_sitemap_error_type["too much urls"];
        break;
    }

    return code;
}

//----------------------------------------------------------------------
class TErrorsStmIterator {
public:
    TErrorsStmIterator(const TSitemapErrorsRec *rec) : Errors(rec), Readed(0) {}

    const TSitemapErrorsRec::Record *Current() const {
        if ((!Errors) || (Errors->Errors.Size  <= sizeof(Errors->Errors.Size) + Readed)) {
            return nullptr;
        }
        return (const TSitemapErrorsRec::Record *)(&Errors->Errors.Data[0] + Readed);
    }

    const TSitemapErrorsRec::Record *Next() {
        const TSitemapErrorsRec::Record *rec = Current();
        if (!rec) {
            return nullptr;
        }
        Readed += rec->SizeOf();
        return Current();
    }

private:
    const TSitemapErrorsRec *Errors;
    size_t Readed;
};


//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_dict>
void load_sitemap_errors(const urlid_t &id,
                         t_dat_files_holder &hldr,
                         const t_dict &tbl_dic_sitemap_error_type,
                         std::list< Poco::Int32 > &errors,
                         std::list< Poco::Tuple<Poco::Int32, Poco::Int32, std::string> > &parse_errors
                        ) {
    using namespace NRobotSitemaps;

    if (hldr.fstmp_errs.is_open()) {
        typename t_dat_files_holder::t_fstmp_errs &fstmp_errs = hldr.fstmp_errs();
        const TSitemapErrorsRec *rec = fstmp_errs.FindRecord(&id);
        if ((!rec) || (!(id == (*rec)))) {
            return;
        }
        for (TErrorsStmIterator iter(rec) ; iter.Current() != nullptr ; iter.Next()) {
            int code = encode_error(iter.Current()->Type, tbl_dic_sitemap_error_type);
            if (code > 0) {
                errors.push_back(code);
                parse_errors.push_back(Poco::Tuple<Poco::Int32, Poco::Int32, std::string>(iter.Current()->Line, iter.Current()->Column, iter.Current()->Text));
            }
        }
    }
}

//----------------------------------------------------------------------
template <typename t_dict>
void define_http_error(ui32 http_code,
                       const t_dict &tbl_dic_sitemap_error_type,
                       std::list< Poco::Int32 > &errors,
                       std::list< Poco::Tuple<Poco::Int32, Poco::Int32, std::string> > &parse_errors) {
    int code = tbl_dic_sitemap_error_type["can not download"];

    if (http_code < 1000) { //normal http-code
        code = tbl_dic_sitemap_error_type["http error"];
    } else {
        // specil cases
        switch (http_code) {
        case EXT_HTTP_HUGEDOC:
        case HTTP_BODY_TOO_LARGE:
            code = tbl_dic_sitemap_error_type["too big file"];
            break;
        case EXT_HTTP_PARSERROR:
            code = tbl_dic_sitemap_error_type["incorrect encoding"];
            break;
        case HTTP_BAD_URL:
            code = tbl_dic_sitemap_error_type["bad url"];
            break;

        // connect/download errors:
        case HTTP_CONNECTION_LOST:
            code = tbl_dic_sitemap_error_type["connection lost"];
            break;
        case HTTP_DNS_FAILURE:
            code = tbl_dic_sitemap_error_type["dns failure"];
            break;
        case HTTP_CONNECT_FAILED:
            code = tbl_dic_sitemap_error_type["connection failed"];
            break;
        case HTTP_ROBOTS_TXT_DISALLOW:
            code = tbl_dic_sitemap_error_type["robots txt disallow"];
            break;

        // connect ok, incorrect http:
        case HTTP_BAD_STATUS_CODE:
        case HTTP_BAD_HEADER_STRING:
        case HTTP_BAD_CHUNK:
        case HTTP_BAD_CONTENT_LENGTH:
        case HTTP_BAD_ENCODING:
        case HTTP_LENGTH_UNKNOWN:
        case HTTP_HEADER_EOF:
        case HTTP_MESSAGE_EOF:
        case HTTP_CHUNK_EOF:
        case HTTP_PAST_EOF:
        case HTTP_HEADER_TOO_LARGE:
        case HTTP_URL_TOO_LARGE:
        case HTTP_BAD_CONTENT_ENCODING:
        case HTTP_CHUNK_TOO_LARGE:
        case EXT_HTTP_EMPTY_RESPONSE:
            code = tbl_dic_sitemap_error_type["incorrect http"];
            break;

        // http ok, fatal problem with content:
        case EXT_HTTP_EMPTYDOC:
            code = tbl_dic_sitemap_error_type["empty doc"];
            break;
        case EXT_HTTP_GZIPERROR:
            code = tbl_dic_sitemap_error_type["gzip error"];
            break;
        }
    }
    errors.push_back(code);
    std::ostringstream buf;
    buf << code;
    parse_errors.push_back(Poco::Tuple<Poco::Int32, Poco::Int32, std::string>(0, 0, buf.str()));
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_dict>
void load_sitemap_errors(ui32 http_code,
                         const urlid_t &id,
                         t_dat_files_holder &hldr,
                         const t_dict &tbl_dic_sitemap_error_type,
                         std::list< Poco::Int32 > &errors,
                         std::list< Poco::Tuple<Poco::Int32, Poco::Int32, std::string> > &parse_errors) {
    errors.clear();
    parse_errors.clear();
    if (0 == http_code) {
        return;
    }

    if (HTTP_OK == http_code) {
        load_sitemap_errors(id, hldr, tbl_dic_sitemap_error_type, errors, parse_errors);
    } else {
        define_http_error(http_code, tbl_dic_sitemap_error_type, errors, parse_errors);
    }
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_stms_db, typename t_dict>
Poco::Int64 insert_sitemap(const t_url_name &url,
                           const t_sitemap_info &val,
                           t_dat_files_holder &dat_files,
                           t_stms_db &stm_db,
                           const t_dict &tbl_dic_sitemap_error_type,
                           const t_stms_db_id_array *indexes = nullptr) {
    std::list< Poco::Int32 > errors;
    std::list< Poco::Tuple<Poco::Int32, Poco::Int32, std::string> > parse_errors;
    load_sitemap_errors(val.HttpCode, val.id(), dat_files, tbl_dic_sitemap_error_type, errors, parse_errors);
    return stm_db.insert_sitemap(url, val, errors, parse_errors, indexes);
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_stms_db, typename t_dict>
Poco::Int64 insert_sitemap(const t_url_name &url,
                           const t_sitemap_info &val,
                           t_dat_files_holder &dat_files,
                           t_stms_db &stm_db,
                           const t_dict &tbl_dic_sitemap_error_type,
                           const t_smtms_indexes_map &index_map) {
    const t_stms_db_id_array *indexes = nullptr;
    t_smtms_indexes_map::const_iterator iter = index_map.find(val.id());
    if (iter != index_map.end()) {
        indexes = &(iter->second);
    }
    return insert_sitemap(url, val, dat_files, stm_db, tbl_dic_sitemap_error_type, indexes);
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_stms_db, typename t_dict>
Poco::Int64 insert_sitemap_index(const t_url_name &url,
                                 const t_sitemap_info &val,
                                 t_dat_files_holder &dat_files,
                                 t_stms_db &stm_db,
                                 const t_dict &tbl_dic_sitemap_error_type,
                                 t_smtms_indexes_map &index_map) {
    Poco::Int64 res = insert_sitemap(url, val, dat_files, stm_db, tbl_dic_sitemap_error_type);
    if (-1 == res) {
        return res;
    }
    load_sitemap_indexes(val.id(), dat_files, res, index_map);
    return res;
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_stms_db, typename t_db_coll, typename t_dict>
Poco::Int64 update_sitemap(const t_url_name &url,
                           const t_sitemap_info &val,
                           t_dat_files_holder &dat_files,
                           t_stms_db &stm_db,
                           const t_dict &tbl_dic_sitemap_error_type,
                           t_db_coll &db_coll,
                           const t_stms_db_id_array *indexes = nullptr) {
    typename t_db_coll::iterator db_iter = db_coll.find(url);
    if (db_iter == db_coll.end()) {
        throw std::logic_error("db_iter isn't defined for update");
    }
    std::list< Poco::Int32 > errors;
    std::list< Poco::Tuple<Poco::Int32, Poco::Int32, std::string> > parse_errors;
    load_sitemap_errors(val.HttpCode, val.id(), dat_files, tbl_dic_sitemap_error_type, errors, parse_errors);
    Poco::Int64 res = stm_db.update_sitemap(db_iter->second, val, errors, parse_errors, indexes);
    db_coll.erase(db_iter);
    return res;
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_stms_db, typename t_db_coll, typename t_dict>
Poco::Int64 update_sitemap(const t_url_name &url,
                           const t_sitemap_info &val,
                           t_dat_files_holder &dat_files,
                           t_stms_db &stm_db,
                           const t_dict &tbl_dic_sitemap_error_type,
                           t_db_coll &db_coll,
                           const t_smtms_indexes_map &index_map) {
    const t_stms_db_id_array *indexes = nullptr;
    t_smtms_indexes_map::const_iterator iter = index_map.find(val.id());
    if (iter != index_map.end()) {
        indexes = &(iter->second);
    }
    return update_sitemap(url, val, dat_files, stm_db, tbl_dic_sitemap_error_type, db_coll, indexes);
}

//----------------------------------------------------------------------
template <typename t_dat_files_holder, typename t_stms_db, typename t_db_coll, typename t_dict>
Poco::Int64 update_sitemap_index(const t_url_name &url,
                                 const t_sitemap_info &val,
                                 t_dat_files_holder &dat_files,
                                 t_stms_db &stm_db,
                                 const t_dict &tbl_dic_sitemap_error_type,
                                 t_db_coll &db_coll,
                                 t_smtms_indexes_map &index_map) {
    Poco::Int64 res = update_sitemap(url, val, dat_files, stm_db, tbl_dic_sitemap_error_type, db_coll);
    if (-1 == res) {
        return res;
    }
    load_sitemap_indexes(val.id(), dat_files, res, index_map);
    return res;
}

//----------------------------------------------------------------------
template <typename t_coll>
struct t_key_pred {
    t_key_pred(const t_coll &s, bool not_flag = false) :
        s(s),
        not_flag(not_flag) {
    };

    template <typename t_type1, typename t_type2>
    bool operator()(const std::pair<t_type1, t_type2> &p) {
        if (s.find(p.first) == s.end()) {
            return not_flag;
        }
        return (!not_flag);
    }

private:
    const t_coll &s;
    bool not_flag;
};

//----------------------------------------------------------------------
template <typename t_coll>
struct t_copy_id_act {
    t_copy_id_act(t_coll &s) : s(s) {};

    template <typename t_type1, typename t_type2>
    void operator()(const std::pair<t_type1, t_type2> &p) {
        s.insert(p.second.id());
    }
private:
    t_coll &s;
};

//----------------------------------------------------------------------
template <typename t_coll>
struct t_index_stms_pred {
    template <typename t_type1>
    bool operator()(const std::pair<t_type1, t_sitemap_info> &p) {
        return (TSitemapStatRecNew::SITEMAPINDEX == p.second.Type);
    }
};

//----------------------------------------------------------------------
template <typename t_coll>
struct t_copy_map_act {
    t_copy_map_act(t_coll &s) : s(s) {};

    template <typename t_type1, typename t_type2>
    void operator()(const std::pair<t_type1, t_type2> &p) {
        s.insert(p);
    }
private:
    t_coll &s;
};


} //sitemaps_inner
} //namespace NWebmaster

#endif
