package ru.yandex.webmaster.common.addurl;

import java.net.URI;
import java.net.URL;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.codahale.metrics.MetricRegistry;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.util.concurrent.CommonThreadFactory;
import ru.yandex.misc.ip.Ipv4Address;
import ru.yandex.webmaster.common.addurl.dao.TblAddUrlHostingsDao;
import ru.yandex.webmaster.common.addurl.dao.TblAddUrlPopularHostsDao;
import ru.yandex.webmaster.common.addurl.dao.TblAddUrlPopularIpDao;
import ru.yandex.wmconsole.data.AddUrlRequest;
import ru.yandex.wmconsole.service.AddUrlService;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmtools.common.error.UserException;

/**
 * @author aherman
 */
public class RateLimitedAddUrlService extends AddUrlService {
    private static final Logger log = LoggerFactory.getLogger(RateLimitedAddUrlService.class);

    private TblAddUrlHostingsDao tblAddUrlHostingsDao;
    private TblAddUrlPopularHostsDao tblAddUrlPopularHostsDao;
    private TblAddUrlPopularIpDao tblAddUrlPopularIpDao;

    private MetricRegistry metricRegistry;

    private volatile AddUrlRateLimiter addUrlRateLimiter;
    private int syncPopularHostsPeriodSeconds = 300;

    private String backendHostName;
    private int minRequestToSync = 20;

    private int maxRequestsForIp = 20000;
    private int maxRequestsForHost = 2000;

    private float maxCountError = 0.1f;

    private volatile LocalDate lastLimitCleanupDate = new LocalDate();


    public void init() {
        addUrlRateLimiter = new AddUrlRateLimiter();

        ScheduledExecutorService scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(
                new CommonThreadFactory(true, this.getClass().getSimpleName() + "-")
        );
        scheduledExecutorService.scheduleAtFixedRate(getSyncTask(), 5, syncPopularHostsPeriodSeconds, TimeUnit.SECONDS);
    }

    @Override
    protected void markAndCheckRequestIp(AddUrlRequest addUrlRequest) throws UserException {
        if (addUrlRequest.getUserIp() == null) {
            return;
        }

        Ipv4Address ipv4Address = null;
        try {
            ipv4Address = Ipv4Address.valueOf(addUrlRequest.getUserIp());
        } catch (Exception e) {
            log.error("Unable to parse IP address", e);
        }

        if (ipv4Address == null) {
            return;
        }

        if(addUrlRateLimiter.markAndCheckIpAllowed(ipv4Address.intValue(), maxRequestsForIp, maxCountError)) {
            log.info("AddUrl rate limit: ALLOW IP " + addUrlRequest.getUrl() + " " + addUrlRequest.getUserIp());
        } else {
            addUrlRequest.setAllowed(false);
            log.info("AddUrl rate limit: FORBID IP " + addUrlRequest.getUrl() + " " + addUrlRequest.getUserIp());

            metricRegistry.meter("monitoring.addUrl.forbidden.ip").mark();
            throw new UserException(WMCUserProblem.ADD_URL_RATE_LIMIT, "Rate limit exceeded");
        }
    }

    @Override
    protected void markAndCheckRequestUrl(AddUrlRequest addUrlRequest) throws UserException {
        if (!addUrlRequest.isAllowed()) {
            return;
        }

        if(addUrlRateLimiter.markAndCheckUrlAllowed(addUrlRequest.getUrl().getHost(), maxRequestsForHost, maxCountError)) {
            log.info("AddUrl rate limit: ALLOW URL " + addUrlRequest.getUrl());
        } else {
            addUrlRequest.setAllowed(false);
            log.info("AddUrl rate limit: FORBID URL " + addUrlRequest.getUrl());

            metricRegistry.meter("monitoring.addUrl.forbidden.url").mark();
            throw new UserException(WMCUserProblem.ADD_URL_RATE_LIMIT, "Rate limit exceeded");
        }
    }

    @Override
    protected void markAndCheckRedirectUrl(URI uri, AddUrlRequest addUrlRequest) throws UserException {
        if (!addUrlRequest.isAllowed()) {
            return;
        }

        if (StringUtils.equalsIgnoreCase(getHost(uri), getHost(addUrlRequest.getUrl()))) {
            return;
        }

        if(addUrlRateLimiter.markAndCheckUrlAllowed(getHost(uri), maxRequestsForHost, maxCountError)) {
            log.info("AddUrl rate limit: ALLOW REDIRECT-URL " + uri + " from " + addUrlRequest.getUrl());
        } else {
            addUrlRequest.setAllowed(false);
            log.info("AddUrl rate limit: FORBID REDIRECT-URL " + uri + " from " + addUrlRequest.getUrl());

            metricRegistry.meter("monitoring.addUrl.forbidden.redirectTargetUrl").mark();
            throw new UserException(WMCUserProblem.ADD_URL_RATE_LIMIT, "Rate limit exceeded");
        }
    }

    @Override
    protected void checkRedirectUrl(URI uri, AddUrlRequest addUrlRequest) throws UserException {
        if (!addUrlRequest.isAllowed()) {
            return;
        }

        if(addUrlRateLimiter.checkUrlAllowed(getHost(uri), maxRequestsForHost, maxCountError)) {
            log.info("AddUrl rate limit: ALLOW REDIRECT-URL " + uri + " from " + addUrlRequest.getUrl());
        } else {
            addUrlRequest.setAllowed(false);
            log.info("AddUrl rate limit: FORBID REDIRECT-URL " + uri + " from " + addUrlRequest.getUrl());

            metricRegistry.meter("monitoring.addUrl.forbidden.redirectUrl").mark();
            throw new UserException(WMCUserProblem.ADD_URL_RATE_LIMIT, "Rate limit exceeded");
        }
    }

    protected Runnable getSyncTask() {
        return new Runnable() {
            @Override
            public void run() {
                log.info("Sync addUrl shared data");
                cleanupDailyLimits();

                // Take into account only regularly updated counters
                // It effectively resets host limits on new day start
                DateTime limitsNewerThan = DateTime.now().minusSeconds(syncPopularHostsPeriodSeconds * 2);

                try {
                    List<String> hostings = tblAddUrlHostingsDao.getHostings();
                    addUrlRateLimiter.setHostings(hostings);
                } catch (Exception e) {
                    log.error("Unable to update hostings", e);
                }

                try {

                    List<PopularHostInfo> popularHosts = tblAddUrlPopularHostsDao.list(limitsNewerThan);
                    popularHosts = cleanupHosts(popularHosts);
                    Map<String, Integer> sharedHostCounter = new HashMap<>();
                    for (PopularHostInfo popularHost : popularHosts) {
                        Integer count = sharedHostCounter.get(popularHost.getHostname());
                        count = count == null ? 0 : count;
                        count += popularHost.getRequestCount();
                        sharedHostCounter.put(popularHost.getHostname(), count);
                    }
                    log.info("Popular hosts loaded: " + sharedHostCounter.size());
                    addUrlRateLimiter.setSharedHostnameCounter(sharedHostCounter);
                } catch (Exception e) {
                    log.error("Unable to sync popular hosts", e);
                }

                try {
                    List<PopularIpInfo> popularIps = tblAddUrlPopularIpDao.list(limitsNewerThan);
                    popularIps = cleanupIps(popularIps);
                    Map<Integer, Integer> sharedIpCounter = new HashMap<>();
                    for (PopularIpInfo popularIp : popularIps) {
                        Integer count = sharedIpCounter.get(popularIp.getIp());
                        count = count == null ? 0 : count;
                        count += popularIp.getRequestCount();
                        sharedIpCounter.put(popularIp.getIp(), count);
                    }
                    log.info("Popular ip loaded: " + sharedIpCounter.size());
                    addUrlRateLimiter.setSharedIpCounter(sharedIpCounter);
                } catch (Exception e) {
                    log.error("Unable to sync popular IPs", e);
                }

                List<PopularHostInfo> popularHosts = Collections.emptyList();
                List<PopularIpInfo> popularIp = Collections.emptyList();
                try {
                    popularHosts = addUrlRateLimiter.getPopularHosts(backendHostName, minRequestToSync);
                    popularIp = addUrlRateLimiter.getPopularIp(backendHostName, minRequestToSync);
                } catch (Exception e) {
                    log.error("Strange exception in counter", e);
                }

                log.info("Popular hosts to save: " + popularHosts.size());
                log.info("Popular IP to save: " + popularIp.size());
                try {
                    tblAddUrlPopularHostsDao.update(popularHosts);
                    tblAddUrlPopularIpDao.update(popularIp);
                } catch (Exception e) {
                    log.error("Unable to save addUrl shared counters", e);
                }
                log.info("AddUrl shared data synced");
            }

            private List<PopularHostInfo> cleanupHosts(List<PopularHostInfo> popularHosts) {
                List<PopularHostInfo> result = new LinkedList<>();
                for (PopularHostInfo popularHost : popularHosts) {
                    if (backendHostName.equals(popularHost.getBackendHost())) {
                        continue;
                    }
                    result.add(popularHost);
                }
                return result;
            }

            private List<PopularIpInfo> cleanupIps(List<PopularIpInfo> popularIps) {
                List<PopularIpInfo> result = new LinkedList<>();
                for (PopularIpInfo popularIp : popularIps) {
                    if (backendHostName.equals(popularIp.getBackendHost())) {
                        continue;
                    }
                    result.add(popularIp);
                }
                return result;
            }
        };
    }

    private void cleanupDailyLimits() {
        LocalDate today = new LocalDate();
        if (today.equals(lastLimitCleanupDate)) {
            return;
        }
        log.info("Reset local limits");
        addUrlRateLimiter = new AddUrlRateLimiter();
        lastLimitCleanupDate = today;

        LocalDate olderThan = lastLimitCleanupDate.minusDays(3);
        try {
            tblAddUrlPopularIpDao.deleteOldIps(olderThan);
        } catch (Exception e) {
            log.error("Unable to cleanup old IPs", e);
        }

        try {
            tblAddUrlPopularHostsDao.deleteOldHosts(olderThan);
        } catch (Exception e) {
            log.error("Unable to cleanup old hosts", e);
        }
    }

    private static String getHost(URI uri) {
        String host = uri.getHost();
        if (host == null) {
            host = uri.getAuthority();
        }
        return StringUtils.defaultString(host);
    }

    private static String getHost(URL url) {
        String host = url.getHost();
        if (host == null) {
            host = url.getAuthority();
        }
        return StringUtils.defaultString(host);
    }

    @Required
    public void setTblAddUrlHostingsDao(TblAddUrlHostingsDao tblAddUrlHostingsDao) {
        this.tblAddUrlHostingsDao = tblAddUrlHostingsDao;
    }

    @Required
    public void setTblAddUrlPopularHostsDao(TblAddUrlPopularHostsDao tblAddUrlPopularHostsDao) {
        this.tblAddUrlPopularHostsDao = tblAddUrlPopularHostsDao;
    }

    @Required
    public void setTblAddUrlPopularIpDao(TblAddUrlPopularIpDao tblAddUrlPopularIpDao) {
        this.tblAddUrlPopularIpDao = tblAddUrlPopularIpDao;
    }

    @Required
    public void setMetricRegistry(MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }

    @Required
    public void setBackendHostName(String backendHostName) {
        this.backendHostName = backendHostName;
    }

    @Required
    public void setMaxRequestsForIp(int maxRequestsForIp) {
        this.maxRequestsForIp = maxRequestsForIp;
    }

    @Required
    public void setMaxRequestsForHost(int maxRequestsForHost) {
        this.maxRequestsForHost = maxRequestsForHost;
    }

    @Required
    public void setMinRequestToSync(int minRequestToSync) {
        this.minRequestToSync = minRequestToSync;
    }

    public void setSyncPopularHostsPeriodSeconds(int syncPopularHostsPeriodSeconds) {
        this.syncPopularHostsPeriodSeconds = syncPopularHostsPeriodSeconds;
    }
}
