package ru.yandex.webmaster.common.host.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.misc.ip.Ipv4Address;
import ru.yandex.webmaster.common.host.HostNodeInfo;
import ru.yandex.wmconsole.data.HostInfoStatusEnum;
import ru.yandex.wmconsole.data.UpdateStateEnum;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.verification.VerificationTypeEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.SqlUtil;

/**
 * User: azakharov
 * Date: 15.01.14
 * Time: 14:48
 */
public class HostListDao extends AbstractDbService {
    private static final String FIELD_HOST_ID = "host_id";
    private static final String FIELD_HOSTNAME = "host_name";
    private static final String FIELD_VERIFICATION_STATE = "verified";
    private static final String FIELD_VERIFICATION_TYPE = "verification_type";
    private static final String FIELD_MIRROR_HOST_ID = "mirror_host_id";
    private static final String FIELD_MIRROR_HOST_NAME = "mirror_host_name";

    private static final String FIELD_URL_COUNT = "url_count";
    private static final String FIELD_UPDATE_STATE = "update_state";
    private static final String FIELD_PRECISE_INDEX_COUNT = "precise_index_count";
    private static final String FIELD_INDEX_COUNT = "index_count";
    private static final String FIELD_INDEX_COUNT_TIME = "index_count_time";
    private static final String FIELD_VIRUSED_STATE = "virused_state";
    private static final String FIELD_TCY = "tcy";
    private static final String FIELD_HOST_INFO_STATUS = "status";
    private static final String FIELD_HOST_INFO_STATUS_DATE = "status_date";
    private static final String FIELD_HOST_INFO_STATUS_HTARC = "status_htarc";
    private static final String FIELD_HOST_INFO_PENALTY_HTARC = "penalty_htarc";
    private static final String FIELD_HOST_INFO_STATUS_YA = "status_ya";
    private static final String FIELD_HOST_INFO_PENALTY_YA = "penalty_ya";
    private static final String FIELD_SPIDER_IP = "spider_ip";
    private static final String FIELD_LAST_ACCESS = "last_access";
    private static final String FIELD_HTARC_IN_INDEX = "htarc_in_index";
    private static final String FIELD_PREV_IN_INDEX = "prev_in_index";
    private static final String FIELD_BANNED_COUNT = "banned_count";

    public List<HostNodeInfo> getHostNodes(final long userId) throws InternalException {
        final String query =
                "SELECT " +
                        "    h.host_id AS " + FIELD_HOST_ID + ", " +
                        "    h.name AS " + FIELD_HOSTNAME + ", " +
                        "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                        "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                        "    m.host_id AS " + FIELD_MIRROR_HOST_ID + ", " +
                        "    m.name AS " + FIELD_MIRROR_HOST_NAME + " " +
                        "FROM " +
                        "    tbl_users_hosts uh " +
                        "LEFT JOIN " +
                        "    tbl_hosts h ON (uh.host_id = h.host_id) " +
                        "LEFT JOIN " +
                        "    tbl_hosts m ON (h.mirror_id = m.host_id) " +
                        "LEFT JOIN " +
                        "    tbl_users_hosts uhm ON (uhm.host_id = m.host_id AND uhm.user_id = uh.user_id)" +
                        "WHERE " +
                        "    uh.user_id = ? ";

        return getJdbcTemplate(WMCPartition.nullPartition()).query(query, userDbMapper, userId);
    }

    public void updateHostNodesWithHostDbInfo(final List<HostNodeInfo> nodes, final int dbIndex) throws InternalException {
        final String SELECT_HOST_DB_HOST_LIST_QUERY =
                "SELECT " +
                        "    h.host_id AS " + FIELD_HOST_ID + ", " +
                        "    h.name AS " + FIELD_HOSTNAME + ", " +
                        "    lc.index_count AS " + FIELD_INDEX_COUNT + ", " +
                        "    lc.index_count_time AS " + FIELD_INDEX_COUNT_TIME + ", " +
                        "    tc.tcy AS " + FIELD_TCY + ", " +
                        "    rdbi.state AS " + FIELD_UPDATE_STATE + ", " +
                        "    rdbi.index_count AS " + FIELD_PRECISE_INDEX_COUNT + ", " +
                        "    rdbi.banned_count AS " + FIELD_BANNED_COUNT + ", " +
                        "    rdbi.urls AS " + FIELD_URL_COUNT + ", " +
                        "    rdbi.htarc_in_index AS " + FIELD_HTARC_IN_INDEX + ", " +
                        "    rdbi.prev_in_index AS " + FIELD_PREV_IN_INDEX + ", " +
                        "    COALESCE(pi.last_access_ya, rdbi.last_access) AS " + FIELD_LAST_ACCESS + ", " +
                        "    si.spider_ip AS " + FIELD_SPIDER_IP + ", " +
                        "    ih.state AS " + FIELD_VIRUSED_STATE + ", " +
                        "    hi.status AS " + FIELD_HOST_INFO_STATUS + ", " +
                        "    hi.status_date AS " + FIELD_HOST_INFO_STATUS_DATE + "," +
                        "    pi.status_htarc AS " + FIELD_HOST_INFO_STATUS_HTARC + ", " +
                        "    pi.penalty_htarc AS " + FIELD_HOST_INFO_PENALTY_HTARC + ", " +
                        "    pi.status_ya AS " + FIELD_HOST_INFO_STATUS_YA + ", " +
                        "    pi.penalty_ya AS " + FIELD_HOST_INFO_PENALTY_YA + " " +
                        "FROM ((((((" +
                        "    tbl_hosts h " +
                        "LEFT JOIN " +
                        "    tbl_robotdb_info rdbi ON (h.host_id = rdbi.host_id)) " +
                        "LEFT JOIN " +
                        "    tbl_links_cache lc ON (h.host_id = lc.host_id)) " +
                        "LEFT JOIN " +
                        "    tbl_tcy_cache tc ON (h.host_id = tc.host_id)) " +
                        "LEFT JOIN " +
                        "    tbl_infected_hosts ih ON (h.host_id = ih.host_id)) " +
                        "LEFT JOIN " +
                        "    tbl_host_info hi ON (h.host_id = hi.host_id)) " +
                        "LEFT JOIN " +
                        "    tbl_penalty_info pi ON (h.host_id = pi.host_id)) " +
                        "LEFT JOIN " +
                        "    ( " +
                        "       SELECT " +
                        "           host_id, " +
                        "           MAX(spider_ip) AS spider_ip " +
                        "       FROM " +
                        "           ( " +
                        "               SELECT " +
                        "                   host_id, " +
                        "                   MAX(date) AS date " +
                        "               FROM " +
                        "                   tbl_spider_ips tmp1 " +
                        "               NATURAL JOIN " +
                        "                   tbl_hosts h2 " +
                        "               WHERE " +
                        "                   h2.name IN (%1$s) " +
                        "               GROUP BY host_id " +
                        "           ) tmp2 " +
                        "       NATURAL JOIN " +
                        "           tbl_spider_ips tmp3 " +
                        "       GROUP BY host_id" +
                        "    ) si " +
                        "ON (h.host_id = si.host_id ) " +
                        "WHERE " +
                        "    h.name IN (%1$s) ";


        StringBuilder commaSeparatedHosts = new StringBuilder();
        String separator = "";
        for (HostNodeInfo node: nodes) {
            commaSeparatedHosts.append(separator);
            commaSeparatedHosts.append("'");
            commaSeparatedHosts.append(node.getHostName().toLowerCase());
            commaSeparatedHosts.append("'");
            separator = ", ";
        }

        Map<String, HostNodeInfo> map = new HashMap<>();
        for (HostNodeInfo info : nodes) {
            map.put(info.getHostName().toLowerCase(), info);
        }

        // достаём всё, что можно из базы
        getJdbcTemplate(new WMCPartition(dbIndex)).query(
                String.format(SELECT_HOST_DB_HOST_LIST_QUERY, commaSeparatedHosts), new HostDbHostListinfoMapper(map));

    }

    public void updateMirrorNodesWithVirusInfo(final List<HostNodeInfo> nodes, final int dbIndex) throws InternalException {

        final String SELECT_VIRUS_INFO_QUERY =
                "SELECT h.name AS " + FIELD_HOSTNAME + ", ih.state AS " + FIELD_VIRUSED_STATE + " " +
                        "FROM tbl_hosts h LEFT JOIN tbl_infected_hosts ih ON (h.host_id = ih.host_id) " +
                        "WHERE h.name in (%1$s)";

        StringBuilder commaSeparatedHosts = new StringBuilder();
        String separator = "";
        for (HostNodeInfo node: nodes) {
            commaSeparatedHosts.append(separator);
            commaSeparatedHosts.append("'");
            commaSeparatedHosts.append(node.getHostName().toLowerCase());
            commaSeparatedHosts.append("'");
            separator = ", ";
        }

        Map<String, HostNodeInfo> map = new HashMap<>();
        for (HostNodeInfo info : nodes) {
            map.put(info.getHostName().toLowerCase(), info);
        }

        // достаём всё, что можно из базы
        getJdbcTemplate(new WMCPartition(dbIndex)).query(
                String.format(SELECT_VIRUS_INFO_QUERY, commaSeparatedHosts), new HostDbVirusInfoMapper(map));

    }

    private static final ParameterizedRowMapper<HostNodeInfo> userDbMapper = new ParameterizedRowMapper<HostNodeInfo>() {
        @Override
        public HostNodeInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            return new HostNodeInfo(
                    rs.getLong(FIELD_HOST_ID), rs.getString(FIELD_HOSTNAME),
                    SqlUtil.getLongNullable(rs, FIELD_MIRROR_HOST_ID), rs.getString(FIELD_MIRROR_HOST_NAME),
                    VerificationStateEnum.R.fromValueOrNull(rs.getInt(FIELD_VERIFICATION_STATE)),
                    VerificationTypeEnum.R.fromValueOrNull(rs.getInt(FIELD_VERIFICATION_TYPE)));
        }
    };

    public static class HostDbHostListinfoMapper implements ParameterizedRowMapper<HostNodeInfo> {
        private final Map<String, HostNodeInfo> hostNameToNodeInfo;

        public HostDbHostListinfoMapper(Map<String, HostNodeInfo> hostNameToNodeInfo) {
            this.hostNameToNodeInfo = hostNameToNodeInfo;
        }

        @Override
        public HostNodeInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            Long hostDbHostId = rs.getLong(FIELD_HOST_ID);
            String hostName = rs.getString(FIELD_HOSTNAME);
            HostDbHostInfo hostDbHostInfo = new HostDbHostInfo(hostDbHostId, hostName);

            Integer updateStateInt = rs.getInt(FIELD_UPDATE_STATE);
            UpdateStateEnum updateState = UpdateStateEnum.R.fromValueOrNull(updateStateInt);

            Long urlCount = SqlUtil.getLongNullable(rs, FIELD_URL_COUNT);

            Integer tcy = SqlUtil.getIntNullable(rs, FIELD_TCY);

            // Пытаемся получить точное число страниц в индексе
            Long indexCount = SqlUtil.getLongNullable(rs, FIELD_PRECISE_INDEX_COUNT);
            Long bannedCount = SqlUtil.getLongNullable(rs, FIELD_BANNED_COUNT);
            Date indexCountUpdateTime = null;
            // Если точное число страниц отсутствует, берем из кэша xml-поиска
            if (indexCount == null) {
                indexCount = SqlUtil.getLongNullable(rs, FIELD_INDEX_COUNT);
                indexCountUpdateTime = SqlUtil.safeGetTimestamp(rs, FIELD_INDEX_COUNT_TIME);
            } else if (bannedCount != null && bannedCount > 0) {
                indexCount = Math.max(0, indexCount - bannedCount);
            }

            boolean virusedState = rs.getObject(FIELD_VIRUSED_STATE) != null;

            HostInfoStatusEnum hostInfoStatus = HostInfoStatusEnum.R.fromValueOrNull(rs.getInt(FIELD_HOST_INFO_STATUS));
            if (rs.wasNull()) {
                hostInfoStatus = null;
            }
            final Date hostInfoStatusDate = SqlUtil.safeGetTimestamp(rs, FIELD_HOST_INFO_STATUS_DATE);

            HostInfoStatusEnum hostInfoStatusHtarc = HostInfoStatusEnum.R.fromValueOrNull(rs.getInt(FIELD_HOST_INFO_STATUS_HTARC));
            Integer penaltyHtarc = null;
            if (rs.wasNull()) {
                hostInfoStatusHtarc = null;
            } else {
                penaltyHtarc = SqlUtil.getIntNullable(rs, FIELD_HOST_INFO_PENALTY_HTARC);
            }

            HostInfoStatusEnum hostInfoStatusYa = HostInfoStatusEnum.R.fromValueOrNull(rs.getInt(FIELD_HOST_INFO_STATUS_YA));
            Integer penaltyYa = null;
            if (rs.wasNull()) {
                hostInfoStatusYa = null;
            } else {
                penaltyYa = SqlUtil.getIntNullable(rs, FIELD_HOST_INFO_PENALTY_YA);
            }

            Boolean htarcInIndex = SqlUtil.getBooleanNullable(rs, FIELD_HTARC_IN_INDEX);
            Boolean prevInIndex = SqlUtil.getBooleanNullable(rs, FIELD_PREV_IN_INDEX);

            Long ip = SqlUtil.getLongNullable(rs, FIELD_SPIDER_IP);
            String stringIp = ip != null ? Ipv4Address.valueOf(ip.intValue()).toString() : null;

            Date lastAccess = SqlUtil.safeGetTimestamp(rs, FIELD_LAST_ACCESS);

            HostNodeInfo res = hostNameToNodeInfo.get(hostName.toLowerCase());
            res.setHostDbFields(
                    hostDbHostInfo,
                    updateState,
                    urlCount,
                    virusedState,
                    tcy,
                    indexCount,
                    indexCountUpdateTime,
                    hostInfoStatus,
                    hostInfoStatusDate,
                    stringIp,
                    hostInfoStatusHtarc,
                    penaltyHtarc,
                    hostInfoStatusYa,
                    penaltyYa,
                    lastAccess,
                    htarcInIndex,
                    prevInIndex);

            return res;
        }
    }

    public static class HostDbVirusInfoMapper implements ParameterizedRowMapper<HostNodeInfo> {
        private final Map<String, HostNodeInfo> hostNameToNodeInfo;

        public HostDbVirusInfoMapper(Map<String, HostNodeInfo> hostNameToNodeInfo) {
            this.hostNameToNodeInfo = hostNameToNodeInfo;
        }

        @Override
        public HostNodeInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            String hostName = rs.getString(FIELD_HOSTNAME);

            boolean virusedState = rs.getObject(FIELD_VIRUSED_STATE) != null;

            HostNodeInfo res = hostNameToNodeInfo.get(hostName.toLowerCase());
            res.setVirused(virusedState);

            return res;
        }
    }
}
