package ru.yandex.webmaster.common.host.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.joda.time.DateTime;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.webmaster.common.host.HostEvent;
import ru.yandex.webmaster.common.host.HostEventType;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;

/**
 * CREATE TABLE IF NOT EXISTS tbl_host_events (
 *     event_id INT NOT NULL AUTO_INCREMENT PRIMARY KEY
 *   , host_id BIGINT NOT NULL
 *   , date TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
 *   , event_type SMALLINT NOT NULL
 *   , event_by_user_id BIGINT NOT NULL
 *   , event_for_user_id BIGINT NOT NULL
 *   , event_data VARCHAR(256) NULL
 *
 *   , FOREIGN KEY (host_id) REFERENCES tbl_hosts (host_id) ON DELETE CASCADE
 *   , KEY (event_type, event_id)
 * ) ENGINE=InnoDB;
 *
 * @author aherman
 */
public class TblHostEventDao extends AbstractDbService {
    public void save(HostEvent hostEvent) throws InternalException {
        String q = "INSERT INTO tbl_host_events(host_id, event_type, event_by_user_id, event_for_user_id, event_data)" +
                " VALUES (?, ?, ?, ?, ?)";
        getJdbcTemplate(WMCPartition.nullPartition()).update(q,
                hostEvent.getHostId(), hostEvent.getEventType().value(), hostEvent.getEventByUserId(),
                hostEvent.getEventForUserId(), hostEvent.getRawData()
        );
    }

    public List<HostEvent> getEvents(SqlCondition condition, SqlLimits limits, SqlOrder order) throws InternalException {
        String q = "SELECT * FROM tbl_host_events WHERE " + condition.sql() + " " + order.toSql() + " " + limits.toMysqlLimits();
        return getJdbcTemplate(WMCPartition.nullPartition()).query(q, getEventMapper(), condition.args().toArray());
    }

    public HostEvent getFirst(SqlCondition condition) throws InternalException {
        String q = "SELECT * FROM tbl_host_events WHERE " + condition.sql() + " LIMIT 1";
        List<HostEvent> result = getJdbcTemplate(WMCPartition.nullPartition())
                .query(q, getEventMapper(), condition.args().toArray());
        if (result.isEmpty()) {
            return null;
        }
        return result.get(0);
    }

    public int countEvents(SqlCondition condition) throws InternalException {
        String q = "SELECT count(*) FROM tbl_host_events WHERE " + condition.sql();
        return getJdbcTemplate(WMCPartition.nullPartition()).queryForInt(q, condition.args().toArray());
    }

    private ParameterizedRowMapper<HostEvent> getEventMapper() {
        return new ParameterizedRowMapper<HostEvent>() {
            @Override
            public HostEvent mapRow(ResultSet rs, int rowNum) throws SQLException {
                int eventId = rs.getInt("event_id");
                long hostId = rs.getLong("host_id");
                DateTime date = new DateTime(rs.getTimestamp("date"));
                HostEventType eventType = HostEventType.R.fromValueOrUnknown(rs.getInt("event_type"));
                long eventByUserId = rs.getLong("event_by_user_id");
                long eventForUserId = rs.getLong("event_for_user_id");
                String eventData = rs.getString("event_data");
                return new HostEvent(eventId, hostId, date, eventType, eventByUserId, eventForUserId, eventData);
            }
        };
    }

    public static final SqlOrder ORDER_BY_EVENT_ID = SqlOrder.orderByColumn("event_id");
}
