package ru.yandex.webmaster.common.host.dao;

import org.jetbrains.annotations.Nullable;
import org.springframework.jdbc.core.PreparedStatementCreator;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.support.GeneratedKeyHolder;
import ru.yandex.common.util.db.LongRowMapper;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.SqlUtil;

import java.sql.*;
import java.util.List;

/**
 * @author aherman
 */
public class TblHostsMainDao extends AbstractDbService {
    public BriefHostInfo addHostInfo(final String hostname, final Long mainMirrorHostId) throws InternalException {
        final String q = "INSERT INTO tbl_hosts (name, mirror_id) VALUES (?, ?)";
        GeneratedKeyHolder keyHolder = new GeneratedKeyHolder();
        PreparedStatementCreator statementCreator = new PreparedStatementCreator() {
            @Override
            public PreparedStatement createPreparedStatement(Connection connection) throws SQLException {
                PreparedStatement statement = connection.prepareStatement(q, Statement.RETURN_GENERATED_KEYS);
                statement.setString(1, hostname);
                if (mainMirrorHostId != null) {
                    statement.setLong(2, mainMirrorHostId);
                } else {
                    statement.setNull(2, Types.BIGINT);
                }
                return statement;
            }
        };
        getJdbcTemplate(WMCPartition.nullPartition()).getJdbcOperations().update(statementCreator, keyHolder);
        long hostId = keyHolder.getKey().longValue();
        return new BriefHostInfo(hostId, hostname, mainMirrorHostId);
    }

    public List<Long> getHostsMirrors(final List<Long> hostIds) throws InternalException {
        final String q = String.format(
                "SELECT host_id FROM tbl_hosts WHERE mirror_id IN (%s)", SqlUtil.getCommaSeparatedList(hostIds));
        return getJdbcTemplate(WMCPartition.nullPartition()).query(q, new LongRowMapper());
    }

    public List<BriefHostInfo> getHostMirrors(final long hostId) throws InternalException {
        final String q = "SELECT host_id, name, mirror_id FROM tbl_hosts WHERE mirror_id = ?";
        ParameterizedRowMapper<BriefHostInfo> mapper = getBriefHostInfoMapper();
        return getJdbcTemplate(WMCPartition.nullPartition()).query(q, mapper, hostId);
    }

    public BriefHostInfo getHostIdByHostname(String hostname) throws InternalException {
        String q = "SELECT host_id, name, mirror_id FROM tbl_hosts WHERE name = ?";
        List<BriefHostInfo> results = getJdbcTemplate(WMCPartition.nullPartition()).query(q, getBriefHostInfoMapper(), hostname);

        return results.isEmpty() ? null : results.iterator().next();
    }

    public BriefHostInfo getBriefHostInfoByHostId(long hostId) throws InternalException {
        String q = "SELECT host_id, name, mirror_id FROM tbl_hosts WHERE host_id = ?";
        List<BriefHostInfo> results = getJdbcTemplate(WMCPartition.nullPartition()).query(q, getBriefHostInfoMapper(), hostId);

        return results.isEmpty() ? null : results.iterator().next();
    }

    public void updateMirror(final BriefHostInfo hostInfo, @Nullable final BriefHostInfo newMainMirror) throws InternalException {
        String q = "UPDATE tbl_hosts SET mirror_id = ? WHERE host_id = ? ";
        Long newMirrorId = newMainMirror == null ? null : newMainMirror.getId();
        getJdbcTemplate(WMCPartition.nullPartition()).update(q, newMirrorId, hostInfo.getId());
    }

    public List<BriefHostInfo> listVerifiedHosts(long fromHostId, int count) throws InternalException {
        String query = "SELECT h.host_id, h.name, h.mirror_id FROM tbl_users_hosts uh JOIN tbl_hosts h USING(host_id) " +
                "WHERE uh.state IN (" + VerificationStateEnum.getCommaSeparatedListForVerifiedStates() + ") AND host_id > ? " +
                "GROUP BY host_id " +
                "ORDER BY host_id ASC " +
                "LIMIT ?";
        return getJdbcTemplate(WMCPartition.nullPartition()).query(query, getBriefHostInfoMapper(), fromHostId, count);
    }

    public static ParameterizedRowMapper<BriefHostInfo> getBriefHostInfoMapper() {
        return new ParameterizedRowMapper<BriefHostInfo>() {
            @Override
            public BriefHostInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
                return new BriefHostInfo(
                        rs.getLong("host_id"),
                        rs.getString("name"),
                        SqlUtil.getLongNullable(rs, "mirror_id")
                );
            }
        };
    }
}
