package ru.yandex.webmaster.common.metrika.dao;

import org.joda.time.DateTime;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import ru.yandex.webmaster.common.metrika.MetrikaProblemInfo;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;

/**
 * @author avhaliullin
 */
public class TblMetrikaProblemsDao extends AbstractDbService {
    private static final String F_HOST_ID = "host_id";
    private static final String F_COUNTER_ID = "counter_id";
    private static final String F_LAST_UPDATE = "last_update";
    private static final String F_LAST_CHECK = "last_check";
    private static final String F_RECHECK_REQUESTED = "recheck_requested";

    private static final String TABLE_NAME = "tbl_host_metrika_problem";
    private static final ParameterizedRowMapper<MetrikaProblemInfo> MAPPER = new ParameterizedRowMapper<MetrikaProblemInfo>() {
        @Override
        public MetrikaProblemInfo mapRow(ResultSet rs, int i) throws SQLException {
            DateTime recheckRequested = new DateTime(rs.getTimestamp(F_RECHECK_REQUESTED));
            if (rs.wasNull()) {
                recheckRequested = null;
            }
            return new MetrikaProblemInfo(
                    rs.getLong(F_HOST_ID),
                    new DateTime(rs.getTimestamp(F_LAST_CHECK)),
                    rs.getLong(F_COUNTER_ID),
                    recheckRequested);
        }
    };

    public void requestRecheck(BriefHostInfo hostInfo) throws InternalException {
        String q = "UPDATE " + TABLE_NAME + " SET recheck_requested = now() WHERE host_id = ?";
        getJdbcTemplate(WMCPartition.nullPartition())
                .update(q, hostInfo.getId());
    }

    public MetrikaProblemInfo getInfoForHost(BriefHostInfo hostInfo) throws InternalException {
        String q = "SELECT host_id, counter_id, last_update, last_check, recheck_requested FROM " + TABLE_NAME + " WHERE host_id = ?";
        return getJdbcTemplate(WMCPartition.nullPartition()).safeQueryForObject(q, MAPPER, hostInfo.getId());
    }

    public void insertValues(List<MetrikaProblemInfo> newValues) throws InternalException {
        if (newValues.isEmpty()) {
            return;
        }
        StringBuilder sb = new StringBuilder("INSERT INTO " + TABLE_NAME + " (host_id, counter_id, last_check, last_update) VALUES ");

        Object[] params = new Object[newValues.size() * 3];
        int index = 0;
        for (MetrikaProblemInfo problemInfo : newValues) {
            if (index != 0) {
                sb.append(",");
            }
            params[index++] = problemInfo.getHostId();
            params[index++] = problemInfo.getCounter();
            params[index++] = new Timestamp(problemInfo.getLastChecked().toDate().getTime());
            sb.append("(?,?,?,now())");
        }
        sb.append(" ON DUPLICATE KEY UPDATE counter_id = VALUES(counter_id), last_check = VALUES(last_check), last_update = NOW()");
        getJdbcTemplate(WMCPartition.nullPartition()).update(sb.toString(), params);
    }

    public void deleteOldRecords(DateTime cleanBefore) throws InternalException {
        String query = "DELETE FROM " + TABLE_NAME + " WHERE last_update < ?";
        getJdbcTemplate(WMCPartition.nullPartition()).update(query, new Timestamp(cleanBefore.toDate().getTime()));
    }
}
