package ru.yandex.webmaster.common.security;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Properties;

/**
 * @author avhaliullin
 */
public class SecretsLoadUtil {
    private static final Logger log = LoggerFactory.getLogger(SecretsLoadUtil.class);

    private static final String SECRET_PROPERTIES_PREFIX = "secrets.";

    public static Properties loadProperties(String dirName) {
        Properties properties = new Properties();
        File f = new File(dirName);
        if (f.isDirectory()) {
            handlePerFileProperties(properties, f);
        } else if (f.isFile()) {
            handleJavaProperties(properties, f);
        }
        return properties;
    }

    private static void handleJavaProperties(Properties properties, File file) {
        try (BufferedReader reader = new BufferedReader(new FileReader(file))) {
            String line;
            while ((line = reader.readLine()) != null) {
                line = line.trim();
                if (!line.startsWith("#")) {
                    String[] parts = line.split("=", 2);
                    properties.setProperty(SECRET_PROPERTIES_PREFIX + parts[0], parts[1]);
                }
            }
        } catch (IOException e) {
            throw new RuntimeException("Failed to read secret properties from file " + file, e);
        }
    }

    private static void handlePerFileProperties(Properties properties, File file) {
        if (file.isDirectory()) {
            File[] children = file.listFiles();
            if (children != null) {
                for (File child : children) {
                    handlePerFileProperties(properties, child);
                }
            }
        } else {
            try (InputStream in = new FileInputStream(file)) {
                String value = IOUtils.toString(in, StandardCharsets.UTF_8);
                String propertyName = SECRET_PROPERTIES_PREFIX + file.getName();
                properties.setProperty(propertyName, value);
                log.info("Found secret property {} from file {}", propertyName, file);
            } catch (IOException e) {
                throw new RuntimeException("Failed to read secret properties file " + file, e);
            }
        }
    }
}
