package ru.yandex.webmaster.common.seolinks;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import org.apache.commons.io.IOUtils;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.WritableResource;
import ru.yandex.kungfu.application.ApplicationResources;
import ru.yandex.webmaster.common.seolinks.dao.TblSeoLinksDao;
import ru.yandex.webmaster.common.seolinks.dao.TblSeoLinksHistoryDao;
import ru.yandex.webmaster.common.seolinks.info.SeoLinksInfo;
import ru.yandex.wmconsole.error.WMCExtraTagNameEnum;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.util.RsyncUtil;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.Iterator;
import java.util.zip.GZIPOutputStream;

/**
 * User: azakharov
 * Date: 30.05.14
 * Time: 19:44
 */
public class SeoLinksSettingsService {

    private static final int MIN_DAYS_TO_MODIFY = 31;

    private static final Logger log = LoggerFactory.getLogger(SeoLinksSettingsService.class);

    private TblSeoLinksDao tblSeoLinksDao;
    private TblSeoLinksHistoryDao tblSeoLinksHistoryDao;
    private ApplicationResources applicationResources;

    private String rsyncShareUri;
    private String rsyncFlags;
    private String rsyncPassword;

    /**
     * Inserts record for each host of iterator if it doesn't exist
     *
     * @param hostIterator
     */
    public void refreshHosts(final Iterator<String> hostIterator) {
        tblSeoLinksDao.insertIgnoreBatch(hostIterator);
    }

    public String exportSettings() {
        String exportFileName = "webmaster.seo_links_settings.json.gz";
        OutputStream os = null;
        try {
            WritableResource exportResource = applicationResources.getResourceInData(exportFileName);
            log.info("Save to file: {}", exportResource.getFile().getAbsolutePath());
            os = new GZIPOutputStream(exportResource.getOutputStream());
            JsonFactory factory = new JsonFactory();
            final JsonGenerator generator = factory.createGenerator(os, JsonEncoding.UTF8);
            SeoLinksSettingsHandler handler = new SeoLinksSettingsHandler() {
                @Override
                public void handleRecord(String hostName, boolean useSeoLinks, long modifiedOn) {
                    try {
                        generator.writeStartObject();
                        generator.writeStringField("Host", hostName);
                        generator.writeBooleanField("UseSeoLinks", useSeoLinks);
                        generator.writeNumberField("ModifiedOn", modifiedOn);
                        generator.writeEndObject();
                    } catch (IOException e) {
                        log.error("error in serialization", e);
                    }
                }
            };
            generator.writeStartObject();
            generator.writeArrayFieldStart("Settings");
            tblSeoLinksDao.getSeoLinks(handler);
            generator.writeEndArray();
            generator.writeEndObject();
            generator.flush();
            generator.close();
        } catch (IOException e) {
            log.error("Exception while exporting seo links settings to file");
        } finally {
            IOUtils.closeQuietly(os);
        }

        final String fromName;
        try {
            fromName = applicationResources.getData().getFile().getCanonicalPath() + "/" + exportFileName;
        } catch (IOException e) {
            final String message = "Exception while gettings file path for seo links settings file";
            log.error(message, e);
            return message;
        }

        try {
            RsyncUtil.rsync(fromName, rsyncShareUri, rsyncFlags, rsyncPassword);
        } catch (InternalException e) {
            final String message = "Exception while rsyncing seo links settings file";
            log.error(message, e);
            return message;
        }

        return "Seo links settings saved";
    }

    public @NotNull SeoLinksInfo getSeoLinksInfo(final String hostName) {
        SeoLinksInfo info = tblSeoLinksDao.getUseSeoLinksOptions(hostName);
        if (info == null) {
            return new SeoLinksInfo(true, null, null, true);
        }

        Date modifiedOn = info.getModifiedOn();
        Date modificationDiallowedUntil = null;
        DateTime now = new DateTime();
        boolean modificationAllowed = true;
        if (modifiedOn != null) {
            DateTime modifyDateTime = new DateTime(modifiedOn.getTime());
            DateTime modificationDisallowedDeadline = modifyDateTime.plusDays(MIN_DAYS_TO_MODIFY);
            modificationDiallowedUntil = modificationDisallowedDeadline.toDate();
            if (modificationDisallowedDeadline.isAfter(now)) {
                modificationAllowed = false;
            }
        }

        return new SeoLinksInfo(info.isUseSeoLinks(), info.getModifiedOn(), modificationDiallowedUntil, modificationAllowed);
    }

    public void saveSeoLinksOptions(final String hostName, final boolean useSeoLinks, final long userId) throws UserException {
        final SeoLinksInfo oldInfo = getSeoLinksInfo(hostName);
        final Date modifiedOn = oldInfo.getModifiedOn();
        final DateTime now = new DateTime();
        if (modifiedOn != null) {
            DateTime modifyDateTime = new DateTime(modifiedOn.getTime());
            if (modifyDateTime.plusDays(MIN_DAYS_TO_MODIFY).isAfter(now)) {
                // trying to modify use_seo_links field earlier than 1 month after last change
                throw new UserException(WMCUserProblem.SEO_LINKS_MODIFICATION_NOT_ALLOWED,
                        "seo links option modification is not allowed",
                        new ExtraTagInfo(WMCExtraTagNameEnum.LAST_MODIFIED, modifiedOn.toString()));
            }
        }
        if (oldInfo.isUseSeoLinks() != useSeoLinks) {
            tblSeoLinksHistoryDao.addHistoryRecord(hostName, useSeoLinks, userId);
            tblSeoLinksDao.saveSeoLinksOptions(hostName, useSeoLinks, now.toDate());
        }
    }

    @Required
    public void setTblSeoLinksDao(TblSeoLinksDao tblSeoLinksDao) {
        this.tblSeoLinksDao = tblSeoLinksDao;
    }

    @Required
    public void setApplicationResources(ApplicationResources applicationResources) {
        this.applicationResources = applicationResources;
    }

    @Required
    public void setRsyncShareUri(String rsyncShareUri) {
        this.rsyncShareUri = rsyncShareUri;
    }

    @Required
    public void setRsyncFlags(String rsyncFlags) {
        this.rsyncFlags = rsyncFlags;
    }

    @Required
    public void setRsyncPassword(String rsyncPassword) {
        this.rsyncPassword = rsyncPassword;
    }

    @Required
    public void setTblSeoLinksHistoryDao(TblSeoLinksHistoryDao tblSeoLinksHistoryDao) {
        this.tblSeoLinksHistoryDao = tblSeoLinksHistoryDao;
    }
}

