package ru.yandex.webmaster.common.seolinks.dao;

import com.datastax.driver.core.*;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster.common.cassandra.CassandraClient;
import ru.yandex.webmaster.common.cassandra.ClientFactory;
import ru.yandex.webmaster.common.seolinks.SeoLinksSettingsHandler;
import ru.yandex.webmaster.common.seolinks.info.SeoLinksInfo;

import java.util.Date;
import java.util.Iterator;

/**
 * User: azakharov
 * Date: 30.05.14
 * Time: 18:39
 */
public class TblSeoLinksDao {
    private ClientFactory cassandraClientFactory;

    public void getSeoLinks(final SeoLinksSettingsHandler handler) {
        CassandraClient client = cassandraClientFactory.getClientInstance();
        Session s = client.getSession();
        String query = "SELECT hostname, use_seo_links, modified_on FROM webmaster.tbl_seo_links";
        ResultSet rs = s.execute(query);
        for (Row row : rs) {
            Date modifiedOn = row.getTimestamp("modified_on");
            long modifiedTimestamp = modifiedOn != null ? modifiedOn.getTime() : 0l;
            handler.handleRecord(row.getString("hostname"), row.getBool("use_seo_links"), modifiedTimestamp);
        }
    }

    /**
     * Get seo links options
     *
     * if options not found in database, return null
     *
     * @param hostName  host name in robot format, in lowercase, punycode-encoded
     * @return seo links info object or null
     */
    public @Nullable
    SeoLinksInfo getUseSeoLinksOptions(final String hostName) {
        final CassandraClient client = cassandraClientFactory.getClientInstance();
        final Session s = client.getSession();
        final ResultSet rs = s.execute("SELECT use_seo_links, modified_on FROM webmaster.tbl_seo_links WHERE hostname = ?", hostName);
        final Iterator<Row> it = rs.iterator();
        if (!it.hasNext()) {
            return null;
        }
        final Row r = it.next();
        return new SeoLinksInfo(r.getBool("use_seo_links"), r.getTimestamp("modified_on"), null, false);
    }

    public void saveSeoLinksOptions(final String hostName, boolean useSeoLinks, Date modifiedOn) {
        CassandraClient client = cassandraClientFactory.getClientInstance();
        Session s = client.getSession();
        s.execute("INSERT INTO webmaster.tbl_seo_links (hostname, use_seo_links, modified_on) VALUES (?, ?, ?)",
                hostName, useSeoLinks, modifiedOn);
    }

    public void insertIgnoreBatch(final Iterator<String> hostIterator) {
        CassandraClient client = cassandraClientFactory.getClientInstance();
        final Session s = client.getSession();

        final String query = "INSERT INTO webmaster.tbl_seo_links (hostname, use_seo_links, modified_on) " +
                "VALUES (?, True, NULL) IF NOT EXISTS USING TTL 2678400";
        final PreparedStatement ps = s.prepare(query);

        while (hostIterator.hasNext()) {
            final String hostName = hostIterator.next().toLowerCase();
            s.execute(new BoundStatement(ps).bind(hostName));
        }
    }

    @Required
    public void setCassandraClientFactory(ClientFactory clientFactory) {
        this.cassandraClientFactory = clientFactory;
    }
}
