package ru.yandex.webmaster.common.seolinks.dao;

import com.datastax.driver.core.ConsistencyLevel;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.Clause;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import com.datastax.driver.core.querybuilder.Select;
import com.datastax.driver.core.utils.UUIDs;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster.common.cassandra.ClientFactory;
import ru.yandex.webmaster.common.seolinks.info.SeoLinksHistoryRecord;

import java.util.*;

/**
 * @author avhaliullin
 */
public class TblSeoLinksHistoryDao {
    private static final String TABLE_SEO_LINKS_HISTORY = "tbl_seo_links_history";

    private static final String F_HOSTNAME = "hostname";
    private static final String F_RECORD_ID = "record_id";
    private static final String F_USER_ID = "user_id";
    private static final String F_USE_SEO_LINKS = "use_seo_links";

    private ClientFactory cassandraClientFactory;

    public void addHistoryRecord(String hostName, boolean useSeoLinks, long userId) {
        Statement st = QueryBuilder.insertInto("webmaster", TABLE_SEO_LINKS_HISTORY)
                .value(F_HOSTNAME, hostName)
                .value(F_RECORD_ID, QueryBuilder.fcall("now"))
                .value(F_USER_ID, userId)
                .value(F_USE_SEO_LINKS, useSeoLinks);
        st.setConsistencyLevel(ConsistencyLevel.ONE);
        cassandraClientFactory.getClientInstance().getSession().execute(st);
    }

    public List<SeoLinksHistoryRecord> listHistoryRecordsForHost(String hostName) {
        return listHistoryRecords(QueryBuilder.eq(F_HOSTNAME, hostName));
    }

    public List<SeoLinksHistoryRecord> listHistoryRecordsForHostAndUser(String hostName, long userId) {
        return listHistoryRecords(QueryBuilder.eq(F_HOSTNAME, hostName), QueryBuilder.eq(F_USER_ID, userId));
    }

    public List<SeoLinksHistoryRecord> listHistoryRecordsForUser(long userId) {
        return listHistoryRecords(QueryBuilder.eq(F_USER_ID, userId));
    }

    private List<SeoLinksHistoryRecord> listHistoryRecords(Clause... clauses) {
        Select.Where where = QueryBuilder.select(F_RECORD_ID, F_HOSTNAME, F_USER_ID, F_USE_SEO_LINKS)
                .from("webmaster", TABLE_SEO_LINKS_HISTORY)
                .where();
        for (Clause clause : clauses) {
            where = where.and(clause);
        }
        Statement st = where;
        st.setConsistencyLevel(ConsistencyLevel.LOCAL_QUORUM);

        ResultSet rs = cassandraClientFactory.getClientInstance().getSession().execute(st);
        List<SeoLinksHistoryRecord> result = new ArrayList<>();
        for (Row row : rs) {
            result.add(new SeoLinksHistoryRecord(
                    row.getString(F_HOSTNAME),
                    row.getLong(F_USER_ID),
                    new Date(UUIDs.unixTimestamp(row.getUUID(F_RECORD_ID))),
                    row.getBool(F_USE_SEO_LINKS)
            ));
        }
        Collections.sort(result, new Comparator<SeoLinksHistoryRecord>() {
            @Override
            public int compare(SeoLinksHistoryRecord o1, SeoLinksHistoryRecord o2) {
                return o1.getCreatedAt().compareTo(o2.getCreatedAt());
            }
        });
        return result;
    }

    @Required
    public void setCassandraClientFactory(ClientFactory cassandraClientFactory) {
        this.cassandraClientFactory = cassandraClientFactory;
    }
}
