package ru.yandex.webmaster.common.util.log;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.AppenderBase;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.DateTimeConstants;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.misc.enums.EnumResolver;
import ru.yandex.misc.enums.IntEnum;
import ru.yandex.misc.enums.IntEnumResolver;

/**
 * RFC 5424 syslog appender
 * @author aherman
 */
public class SyslogAppender extends AppenderBase<ILoggingEvent> {
    private static final int DEFAULT_SYSLOG_NG_PORT = 514;
    private static final int DEFAULT_MAX_MESSAGE_SIZE = 65536;

    private static final int SYSLOG_MESSAGE_VERSION = 1;

    private String syslogHost;
    private int syslogPort = DEFAULT_SYSLOG_NG_PORT;
    private String applicationName;
    private String facilityStr;
    private String applicationHostname;
    private String messageFormatStr;

    private MessageFormat messageFormat = MessageFormat.BSD;
    private Facility facility;
    private DatagramSocket syslogSocket;
    private InetAddress syslogIA;
    private int maxMessageSize;

    private byte[] messageBuffer;

    @Override
    public void start() {
        if (!StringUtils.isEmpty(messageFormatStr)) {
            MessageFormat mf = MessageFormat.R.valueOfOrNull(messageFormatStr);
            if (mf == null) {
                addError("Unsupported message format: " + messageFormatStr);
                return;
            }
            messageFormat = mf;
            if (messageFormat != MessageFormat.BSD && messageFormat != MessageFormat.IEFT) {
                addError("Message format not implemented: " + messageFormat);
                return;
            }
        }
        addInfo("Message format: " + messageFormat);

        if (StringUtils.isEmpty(facilityStr)) {
            addError("Facility must be set");
            return;
        }
        Facility facilityOrNull = Facility.R.valueOfOrNull(facilityStr);
        if (facilityOrNull == null) {
            addError("Unknown facility: " + facilityStr);
        }
        facility = facilityOrNull;
        addInfo("Facility: " + facility);

        if (StringUtils.isEmpty(applicationName)) {
            addError("Application name must be set");
            return;
        }
        addInfo("Application: " + applicationName);

        if (syslogPort < 0 || syslogPort > 65536) {
            addError("Illegal value for port: " + syslogPort);
            return;
        }
        if (StringUtils.isEmpty(syslogHost)) {
            addError("Syslog host must be set");
            return;
        }
        addInfo("Syslog address: " + syslogHost + ":" + syslogPort);

        try {
            syslogIA = InetAddress.getByName(syslogHost);
        } catch (UnknownHostException e) {
            addError("Unknown syslog host: " + syslogHost, e);
            return;
        }

        try {
            syslogSocket = new DatagramSocket();
            int sendBufferSize = syslogSocket.getSendBufferSize();
            maxMessageSize = Math.min(sendBufferSize, DEFAULT_MAX_MESSAGE_SIZE);
        } catch (SocketException e) {
            addError("Unable to create local UDP socket", e);
            return;
        }

        addInfo("Max syslog message size is: " + maxMessageSize);
        messageBuffer = new byte[maxMessageSize];

        super.start();
    }

    @Override
    public void stop() {
        if (syslogSocket != null) {
            IOUtils.closeQuietly(syslogSocket);
        }
        super.stop();
    }

    @Override
    protected void append(ILoggingEvent eventObject) {
        if (!isStarted()) {
            return;
        }

        try {
            ByteArrayBuilder builder = new ByteArrayBuilder(messageBuffer);
            if (messageFormat == MessageFormat.BSD) {
                BsdMessage.formatMessage(builder,
                        facility,
                        getSeverity(eventObject),
                        applicationHostname,
                        applicationName,
                        eventObject.getFormattedMessage()
                );
            } else if (messageFormat == MessageFormat.IEFT) {
                IETFMessage.formatMessage(builder,
                        facility,
                        getSeverity(eventObject),
                        applicationHostname,
                        applicationName,
                        getMessageId(eventObject),
                        eventObject.getFormattedMessage()
                );
            }
            sendMessage(builder);
        } catch (IllegalArgumentException e) {
            addError("Unable to create message: " + e.getMessage(), e);
        } catch (IOException e) {
            addError("Unable to create send message: " + e.getMessage(), e);
        }
    }


    private void sendMessage(ByteArrayBuilder builder) throws IOException {
        DatagramPacket pt = new DatagramPacket(messageBuffer, builder.getPosition(), syslogIA, syslogPort);
        syslogSocket.send(pt);
    }

    private String getMessageId(ILoggingEvent eventObject) {
        return eventObject.getThreadName();
    }

    private static Severity getSeverity(ILoggingEvent eventObject) {
        Level level = eventObject.getLevel();
        switch(level.toInt()) {
            case Level.TRACE_INT: return Severity.DEBUG;
            case Level.DEBUG_INT: return Severity.DEBUG;
            case Level.INFO_INT: return Severity.INFORMATIONAL;
            case Level.WARN_INT: return Severity.WARNING;
            case Level.ERROR_INT: return Severity.ERROR;
        }
        throw new IllegalArgumentException("Unsupported level: " + level);
    }

    public void setFacility(String facilityStr) {
        this.facilityStr = StringUtils.trimToEmpty(facilityStr);
    }

    public void setSyslogHost(String syslogHost) {
        this.syslogHost = StringUtils.trimToEmpty(syslogHost);
    }

    public void setSyslogPort(int port) {
        syslogPort = port;
    }

    public void setApplicationName(String applicationName) {
        this.applicationName = StringUtils.trimToEmpty(applicationName);
    }

    public void setApplicationHostname(String applicationHostname) {
        this.applicationHostname = applicationHostname;
    }

    public void setMaxMessageSize(int maxMessageSize) {
        this.maxMessageSize = maxMessageSize;
    }

    public void setMessageFormat(String messageFormatStr) {
        this.messageFormatStr = messageFormatStr;
    }

    private static enum Facility implements IntEnum{
        KERN(       0 << 3),
        USER(       1 << 3),
        MAIL(       2 << 3),
        DAEMON(     3 << 3),
        AUTH(       4 << 3),
        SYSLOG(     5 << 3),
        LPR(        6 << 3),
        NEWS(       7 << 3),
        UUCP(       8 << 3),
        CLOCK(      9 << 3),
        AUTHPRIV(   10 << 3),
        FTP(        11 << 3),
        NTP(        12 << 3),
        LOG_AUDIT(  13 << 3),
        LOG_ALERT(  14 << 3),
        CRON(       15 << 3),
        LOCAL0(     16 << 3),
        LOCAL1(     17 << 3),
        LOCAL2(     18 << 3),
        LOCAL3(     19 << 3),
        LOCAL4(     20 << 3),
        LOCAL5(     21 << 3),
        LOCAL6(     22 << 3),
        LOCAL7(     23 << 3),
        ;

        private final int value;

        Facility(int value) {
            this.value = value;
        }

        @Override
        public int value() {
            return value;
        }

        private static final IntEnumResolver<Facility> R = IntEnumResolver.r(Facility.class);
    }

    private static enum Severity implements IntEnum {
        EMERGENCY(      0),
        ALERT(          1),
        CRITICAL(       2),
        ERROR(          3),
        WARNING(        4),
        NOTICE(         5),
        INFORMATIONAL(  6),
        DEBUG(          7),
        ;

        private final int value;

        Severity(int value) {
            this.value = value;
        }

        @Override
        public int value() {
            return value;
        }
    }

    private static enum MessageFormat {
        BSD,
        IEFT,
        ;
        public static EnumResolver<MessageFormat> R = EnumResolver.er(MessageFormat.class);
    }

    static class BsdMessage {
        /**
         *
         * <code><pre>

         SYSLOG-MSG      = HEADER MSG

         HEADER          = PRI TIMESTAMP SP HOSTNAME SP
         PRI             = "<" PRIVAL ">"
         PRIVAL          = 1*3DIGIT ; range 0 .. 191
         HOSTNAME        = NILVALUE / 1*255PRINTUSASCII

         TIMESTAMP       = SYSLOG-TIMESTAMP
         SYSLOG-TIMESTAMP = MONTH SP DAY SP TIME
         MONTH           = "Jan" / "Feb" / "Mar" / "Apr" / "May" / "Jun" / "Jul" / "Aug" / "Sep" / "Oct" / "Nov" / "Dec"
         DAY             = SP 1DIGIT / 2DIGIT ; if day less than 10 it must contain space instead of leading 0 "Jan  1"
         TIME            = TIME-HOUR ":" TIME-MINUTE ":" TIME-SECOND
         TIME-HOUR       = 2DIGIT  ; 00-23
         TIME-MINUTE     = 2DIGIT  ; 00-59
         TIME-SECOND     = 2DIGIT  ; 00-59

         MSG             = APP-NAME ":" SP CONTENT
         APP-NAME        = 1*32PRINTUSASCII

         OCTET           = %d00-255
         SP              = %d32
         PRINTUSASCII    = %d33-126
         NONZERO-DIGIT   = %d49-57
         DIGIT           = %d48 / NONZERO-DIGIT
         NILVALUE        = "-"

         * </pre></code>

         * @param builder
         * @param facility
         * @param severity
         * @param hostname
         * @param applicationName
         * @param message
         */
        public static void formatMessage(ByteArrayBuilder builder, Facility facility, Severity severity, String hostname, String applicationName, String message) {
            // Header
            // Pri
            builder.appendAscii('<');
            builder.appendUnsignedInt(facility.value() + severity.value());
            builder.appendAscii('>');
            // Date
            appendBSDDateTime(builder, DateTime.now());
            builder.appendSP();
            // Hostname
            builder.appendAscii(hostname);
            builder.appendSP();

            // Msg
            builder.appendAscii(applicationName);
            builder.appendAscii(':');
            builder.appendSP();

            builder.append(message);
        }

        public static void appendBSDDateTime(ByteArrayBuilder builder, DateTime dateTime) {
            switch(dateTime.getMonthOfYear()) {
                case DateTimeConstants.JANUARY:     builder.appendAscii("Jan"); break;
                case DateTimeConstants.FEBRUARY:    builder.appendAscii("Feb"); break;
                case DateTimeConstants.MARCH:       builder.appendAscii("Mar"); break;
                case DateTimeConstants.APRIL:       builder.appendAscii("Apr"); break;
                case DateTimeConstants.MAY:         builder.appendAscii("May"); break;
                case DateTimeConstants.JUNE:        builder.appendAscii("Jun"); break;
                case DateTimeConstants.JULY:        builder.appendAscii("Jul"); break;
                case DateTimeConstants.AUGUST:      builder.appendAscii("Aug"); break;
                case DateTimeConstants.SEPTEMBER:   builder.appendAscii("Sep"); break;
                case DateTimeConstants.OCTOBER:     builder.appendAscii("Oct"); break;
                case DateTimeConstants.NOVEMBER:    builder.appendAscii("Nov"); break;
                case DateTimeConstants.DECEMBER:    builder.appendAscii("Dec"); break;
            }

            builder.appendSP();

            int dayOfMonth = dateTime.getDayOfMonth();
            addSmallIntWithPrefix(builder, dayOfMonth, ' ');

            builder.appendSP();

            addSmallIntWithPrefix(builder, dateTime.getHourOfDay(), '0');
            builder.appendAscii(':');
            addSmallIntWithPrefix(builder, dateTime.getMinuteOfHour(), '0');
            builder.appendAscii(':');
            addSmallIntWithPrefix(builder, dateTime.getSecondOfMinute(), '0');
        }

        static void addSmallIntWithPrefix(ByteArrayBuilder builder, int value, char prefix) {
            if (value < 10) {
                builder.appendAscii(prefix);
            }
            builder.appendUnsignedInt(value);
        }
    }

    static class IETFMessage {
        private static final DateTimeFormatter ISO_DATE_TIME_FORMAT = ISODateTimeFormat.dateTime();

        /**
         * <p>
         * Syslog message grammar
         * </p>
         *
         * <code>
         * <pre>
         *
         SYSLOG-MSG      = HEADER SP STRUCTURED-DATA [SP MSG]

         HEADER          = PRI VERSION SP TIMESTAMP SP HOSTNAME SP APP-NAME SP PROCID SP MSGID
         PRI             = "<" PRIVAL ">"
         PRIVAL          = 1*3DIGIT ; range 0 .. 191
         VERSION         = NONZERO-DIGIT 0*2DIGIT
         HOSTNAME        = NILVALUE / 1*255PRINTUSASCII

         APP-NAME        = NILVALUE / 1*48PRINTUSASCII
         PROCID          = NILVALUE / 1*128PRINTUSASCII
         MSGID           = NILVALUE / 1*32PRINTUSASCII

         TIMESTAMP       = NILVALUE / FULL-DATE "T" FULL-TIME
         FULL-DATE       = DATE-FULLYEAR "-" DATE-MONTH "-" DATE-MDAY
         DATE-FULLYEAR   = 4DIGIT
         DATE-MONTH      = 2DIGIT  ; 01-12
         DATE-MDAY       = 2DIGIT  ; 01-28, 01-29, 01-30, 01-31 based on month/year
         FULL-TIME       = PARTIAL-TIME TIME-OFFSET
         PARTIAL-TIME    = TIME-HOUR ":" TIME-MINUTE ":" TIME-SECOND [TIME-SECFRAC]
         TIME-HOUR       = 2DIGIT  ; 00-23
         TIME-MINUTE     = 2DIGIT  ; 00-59
         TIME-SECOND     = 2DIGIT  ; 00-59
         TIME-SECFRAC    = "." 1*6DIGIT
         TIME-OFFSET     = "Z" / TIME-NUMOFFSET
         TIME-NUMOFFSET  = ("+" / "-") TIME-HOUR ":" TIME-MINUTE

         STRUCTURED-DATA = NILVALUE / 1*SD-ELEMENT
         SD-ELEMENT      = "[" SD-ID *(SP SD-PARAM) "]"
         SD-PARAM        = PARAM-NAME "=" %d34 PARAM-VALUE %d34
         SD-ID           = SD-NAME
         PARAM-NAME      = SD-NAME
         PARAM-VALUE     = UTF-8-STRING ; characters '"', '\' and ']' MUST be escaped.
         SD-NAME         = 1*32PRINTUSASCII ; except '=', SP, ']', %d34 (")

         MSG             = MSG-ANY / MSG-UTF8
         MSG-ANY         = *OCTET ; not starting with BOM
         MSG-UTF8        = BOM UTF-8-STRING
         BOM             = %xEF.BB.BF

         UTF-8-STRING    = *OCTET ; UTF-8 string as specified in RFC 3629

         OCTET           = %d00-255
         SP              = %d32
         PRINTUSASCII    = %d33-126
         NONZERO-DIGIT   = %d49-57
         DIGIT           = %d48 / NONZERO-DIGIT
         NILVALUE        = "-"

         * </pre>
         * </code>
         *
         * @param builder
         * @param facility
         * @param severity
         * @param hostname
         * @param applicationName
         * @param messageId
         * @param message
         */
        public static void formatMessage(ByteArrayBuilder builder, Facility facility, Severity severity, String hostname, String applicationName, String messageId, String message) {
            // Header
            builder.appendAscii('<');
            builder.appendUnsignedInt(facility.value() + severity.value());
            builder.appendAscii('>');
            builder.appendUnsignedInt(SYSLOG_MESSAGE_VERSION);
            builder.appendSP();
            appendISODateTime(builder, DateTime.now());
            builder.appendAscii(ISO_DATE_TIME_FORMAT.print(DateTime.now()));
            builder.appendSP();
            builder.appendAscii(hostname);
            builder.appendSP();
            builder.appendAscii(applicationName);
            builder.appendSP();
            builder.appendNILVALUE();
            builder.appendSP();
            builder.append(messageId);

            // SP
            builder.appendSP();

            // Structured data
            builder.appendNILVALUE();

            // SP
            builder.appendSP();

            // Message
            builder.appendBOM();
            builder.append(message);
        }

        public static void appendISODateTime(ByteArrayBuilder builder, DateTime dateTime) {
            builder.appendAscii(ISO_DATE_TIME_FORMAT.print(dateTime));
        }
    }

    static class ByteArrayBuilder {
        private static final int[] SIZES = {9, 99, 999, 9999, 99999, 999999, 9999999, 99999999, 999999999};
        private final byte[] buffer;
        private int position;

        public ByteArrayBuilder(byte[] buffer) {
            this.buffer = buffer;
        }

        private void checkSize(int size) {
            if ((position + size) >= buffer.length) {
                throw new IllegalArgumentException("Unable to add data, buffer limit exceeded");
            }
        }

        public void appendSP() {
            buffer[position] = ' ';
            position++;
        }

        public void appendNILVALUE() {
            buffer[position] = '-';
            position++;
        }

        public void appendBOM() {
            checkSize(3);
            buffer[position++] = (byte) 0xEF;
            buffer[position++] = (byte) 0xBB;
            buffer[position++] = (byte) 0xBF;
        }

        public void appendAscii(char ch) {
            if (ch < 32 || ch > 126) {
                throw new IllegalArgumentException("Must be PRINT US ASCII: " + ch);
            }
            checkSize(1);
            buffer[position] = (byte) ch;
            position++;
        }

        public void appendAscii(String st) {
            for (int i = 0; i < st.length(); i++) {
                appendAscii(st.charAt(i));
            }
        }

        public void append(String str) {
            byte[] bytes = str.getBytes(StandardCharsets.UTF_8);
            checkSize(bytes.length);
            System.arraycopy(bytes, 0, buffer, position, bytes.length);
            position += bytes.length;
        }

        public void appendUnsignedInt(int i) {
            if (i < 0) {
                throw new IllegalArgumentException("Number bust be greater than 0: " + i);
            }

            int size = Arrays.binarySearch(SIZES, i);
            if (size < 0) {
                size = -size;
            } else {
                size = size + 1;
            }

            checkSize(size);
            int t = i;
            for (int digitPosition = size - 1; digitPosition >= 0; digitPosition--) {
                int r = t % 10;
                buffer[position + digitPosition] = (byte) ('0' + r);
                t = (t - r) / 10;
            }
            position += size;
        }

        public int getPosition() {
            return position;
        }
    }
}
