package ru.yandex.wmconsole.common.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.ServiceTransactionCallback;

/**
 * While this class incapsulates useful methods and a layer of abstraction, it can be used as a super-class
 * for classes, that implement such a logic: class has one or more threads, that must do some action (usually,
 * crawling URLs) after some period, or by an event.
 * <p/>
 * IMPORTANT NOTICE!
 * Because of designed for parallel usage in several servants, this base class does not use synchronization
 * for block between "Get first waiting" and "Update to state IN_PROGESS". That's why you need to use
 * exclusive locks in db transaction to guarantee, that someone else will not change the state of
 * "first waiting" before you do it or read it. One of the ways to do it is to use "SELECT ... FOR UPDATE"
 * query of MySQL.
 * <p/>
 * User: baton
 */
public abstract class AbstractUrlVisitorService<T> extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(AbstractUrlVisitorService.class);

    private final long DEADLINE_MS = 9 * 60 * 1000; // 9 минут

    public final void checkDatabase(int databaseIndex) {
        log.info(getClass().getName() + " fired for database " + databaseIndex);

        long deadline = System.currentTimeMillis() + DEADLINE_MS;
        while (true) {
            try {
                log.info("Correctness checked...");
                T t = getFirstWaiting(databaseIndex);

                if (t != null) {
                    T newT = check(t);
                    updateStateToCurrentStateValue(newT);
                    onStateChanged(t, newT);
                    log.info("State updated successfully.");
                } else {
                    log.info(getClass().getName() + " finished for database " + databaseIndex);
                    return;
                }

                if (System.currentTimeMillis() > deadline) {
                    return;
                }
            } catch (InternalException e) {
                log.error("InternalException occured in thread " + Thread.currentThread().getName(), e);
                log.info(getClass().getName() + " finished for database " + databaseIndex);
                return;
            } catch (UserException e) {
                log.error("UserException occured in thread " + Thread.currentThread().getName(), e);
                log.info(getClass().getName() + " finished for database " + databaseIndex);
                return;
            } catch (RuntimeException e) {
                log.error("RuntimeException occured in thread " + Thread.currentThread().getName(), e);
                log.info(getClass().getName() + " finished for database " + databaseIndex);
                return;
            }
        }
    }

    private T getFirstWaiting(final int databaseIndex) throws UserException, InternalException {
        //noinspection unchecked
        return (T) getServiceTransactionTemplate(new WMCPartition(databaseIndex)).executeInService(new ServiceTransactionCallback() {
            @Override
            public Object doInTransaction(TransactionStatus transactionStatus) throws InternalException {
                T t = listFirstWaiting(databaseIndex);
                if (t != null) {
                    updateStateToInProgress(t);
                }
                return t;
            }
        });
    }

    protected abstract T listFirstWaiting(int databaseIndex) throws InternalException;

    protected abstract T check(T oldT);

    protected abstract void updateStateToInProgress(T t) throws InternalException;

    protected abstract void updateStateToCurrentStateValue(T newT) throws InternalException;

    protected abstract void onStateChanged(T t, T newT) throws InternalException;

    protected abstract void fireCheck();
}
