package ru.yandex.wmconsole.data.info;

import java.util.Date;

import ru.yandex.wmconsole.data.MainPageErrorEnum;
import ru.yandex.wmconsole.data.SeverityEnum;
import ru.yandex.wmconsole.data.UserErrorOptions;
import ru.yandex.wmconsole.util.UrlErrorGrouper;

/**
 * User: azakharov
 * Date: 26.11.12
 * Time: 17:01
 */
public class MainPageErrorInfo {
    private final Integer prodCode;
    private final Date prodDate;
    private final Integer curCode;
    private final Date curDate;
    private final MainPageErrorEnum mainPageError;

    // Настройки по умолчанию. Нужны для случая, если встречаются старые коды ошибок
    private static final UserErrorOptions defaultErrorOptions = new UserErrorOptions(0L);

    public MainPageErrorInfo(Integer prodCode, Date prodDate, Integer curCode, Date curDate,
                             UserErrorOptions userErrorOptions) {
        this.prodCode = prodCode;
        this.prodDate = prodDate;
        this.curCode = curCode;
        this.curDate = curDate;
        this.mainPageError = calculateMainPageError(userErrorOptions);
    }

    public Integer getCurCode() {
        return curCode;
    }

    public Integer getProdCode() {
        return prodCode;
    }

    public Date getProdDate() {
        return prodDate;
    }

    public Date getCurDate() {
        return curDate;
    }

    public MainPageErrorEnum getMainPageError() {
        return mainPageError;
    }

    private MainPageErrorEnum calculateMainPageError(UserErrorOptions userErrorOptions) {
        if (isIgnoredCode(prodCode, userErrorOptions) ||
                isOkCode(prodCode, userErrorOptions)) {
            if (isIgnoredCode(curCode, userErrorOptions) ||
                    isOkCode(curCode, userErrorOptions)) {
                // mh in {ignore, ok} && mp = {ignore, ok)
                return MainPageErrorEnum.NONE;
            } else if (isErrorCode(curCode, userErrorOptions)) {
                // mh in {ignore, ok} && mp = err
                return MainPageErrorEnum.WARNING_MAIN_PAGE_EXCLUDED;
            } else {
                // unknown error (mh in {ignore, ok} && mp = ignore)
                return MainPageErrorEnum.NONE;
            }
        } else if (isErrorCode(prodCode, userErrorOptions)) {
            if (isIgnoredCode(curCode, userErrorOptions) ||
                    isErrorCode(curCode, userErrorOptions)) {
                // mh = err && mp in {ignore, err}
                return MainPageErrorEnum.ERROR_MAIN_PAGE_EXCLUDED;
            } else if (isOkCode(curCode, userErrorOptions)) {
                // mh = err && mp = ok
                return MainPageErrorEnum.WILL_ERROR_MAIN_PAGE_EXCLUDED;
            } else {
                // unknown error (mh = err && mp = ignore)
                return MainPageErrorEnum.ERROR_MAIN_PAGE_EXCLUDED;
            }
        } else {
            // unknown error (mh = ignore)
            if (isErrorCode(curCode, userErrorOptions)) {
                // mh = ignore && mp = err
                return MainPageErrorEnum.WARNING_MAIN_PAGE_EXCLUDED;
            } else {
                // mh = ignore && mp in {ok, ignore}
                return MainPageErrorEnum.NONE;
            }
        }
    }

    public static boolean isIgnoredCode(Integer code, UserErrorOptions userErrorOptions) {
        SeverityEnum severity = code != null ? userErrorOptions.getSeverityByCode(code) : SeverityEnum.OK;
        if (SeverityEnum.DISALLOWED_BY_USER.equals(severity) ||
                SeverityEnum.UNSUPPORTED_BY_ROBOT.equals(severity)) {
            return true;
        } else {
            return false;
        }
    }

    public static boolean isOkCode(Integer code, UserErrorOptions userErrorOptions) {
        SeverityEnum severity = code != null ? userErrorOptions.getSeverityByCode(code) : SeverityEnum.OK;
        return SeverityEnum.OK.equals(severity) || code != null && UrlErrorGrouper.isOk(code);
    }

    public static boolean isErrorCode(Integer code, UserErrorOptions userErrorOptions) {
        SeverityEnum severity = code != null ? userErrorOptions.getSeverityByCode(code) : SeverityEnum.OK;
        return SeverityEnum.SITE_ERROR.equals(severity);
    }
}
