package ru.yandex.wmconsole.data.info;

import java.util.Date;

import ru.yandex.common.framework.user.UserInfo;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.verification.VerificationTypeEnum;

/**
 * Information that is used to verify that user is an owner of given host.
 *
 * User: baton
 * Date: 23.03.2007
 * Time: 19:50:58
 */
public class UsersHostsInfo extends ShortUsersHostsInfo{
    /**
     * Type of verification, used to verify host. Null is not allowed.
     */
    private final VerificationTypeEnum verificationType;

    /**
     * Date of last verification. Null if not state.
     */
    private final Date verificationDate;
    private final VerificationStateEnum verificationState;
    private final long verificationUin;
    private final String verifyFaultLog;
    private final Integer httpCode;
    private final String hostName;
    private final UserInfo userInfo;

    public UsersHostsInfo(VerificationStateEnum verificationState, long verificationUin,
                          VerificationTypeEnum verificationType, Date verificationDate,
                          String verifyFaultLog, long userId, long hostId, String hostName,
                          UserInfo userInfo, Integer httpCode) {
        super(userId, hostId);
        if (verificationType == null) {
            throw new IllegalArgumentException("verificationType must not be null.");
        }

        this.verificationState = verificationState;
        this.verificationUin = verificationUin;
        this.verificationType = verificationType;
        this.verificationDate = verificationDate;
        this.verifyFaultLog = verifyFaultLog;
        this.httpCode = httpCode;
        this.hostName = hostName;
        this.userInfo = userInfo;
    }

    public static UsersHostsInfo createUsersHostsInfoForJustAddedHostAutoVerification(long verificationUin, long userId,
            long hostId, String hostName, UserInfo userInfo)
    {
        return new UsersHostsInfo(VerificationStateEnum.NEVER_VERIFIED, verificationUin, VerificationTypeEnum.AUTO,
                new Date(), "", userId, hostId, hostName, userInfo, null);
    }

    /**
     * Creates an instance of UsersHostsInfo for not state hosts.
     * @param verificationState ...
     * @param verificationUin ...
     * @param userId ...
     * @param hostId ...
     * @param hostName ...
     * @param userInfo ...
     * @param verifyFaultLog ...
     * @return Returns instance of verification information object.
     */
    public static UsersHostsInfo createNotVerifiedNow(VerificationStateEnum verificationState, long verificationUin,
            long userId, long hostId, String hostName, UserInfo userInfo, String verifyFaultLog, Integer httpCode)
    {
        return new UsersHostsInfo(verificationState, verificationUin, VerificationTypeEnum.DEFAULT, new Date(),
                verifyFaultLog, userId, hostId, hostName, userInfo, httpCode);
    }

    /**
     * Creates not NOT VERIFIED verification info based on data from given
     * verification information object.
     *
     * @param verificationState ...
     * @param ver ...
     * @return ...
     */
    public static UsersHostsInfo createNotVerifiedNow(UsersHostsInfo ver, VerificationStateEnum verificationState) {
        return createNotVerifiedNow(verificationState, ver.getVerificationUin(), ver.getUserId(), ver.getHostId(),
                ver.getHostName(), ver.getUserInfo(), ver.getVerifyFaultLog(), null);
    }

    /**
     * Creates not NOT VERIFIED verification info based on data from given
     * verification information object.
     *
     * @param verificationState ...
     * @param verifyFaultLog ...
        * @param ver ...
  * @return ...
     */
    public static UsersHostsInfo createNotVerifiedNow(UsersHostsInfo ver, VerificationStateEnum verificationState,
            String verifyFaultLog)
    {
        return createNotVerifiedNow(verificationState, ver.getVerificationUin(), ver.getUserId(), ver.getHostId(),
                ver.getHostName(), ver.getUserInfo(), verifyFaultLog, null);
    }

    /**
     * Creates not NOT VERIFIED verification info based on data from given
     * verification information object.
     *
     * @param verificationState ...
     * @param verifyFaultLog ...
     * @param ver ...
     * @return ...
     */
    public static UsersHostsInfo createNotVerifiedNow(UsersHostsInfo ver, VerificationStateEnum verificationState,
            String verifyFaultLog, int httpCode)
    {
        return createNotVerifiedNow(verificationState, ver.getVerificationUin(), ver.getUserId(), ver.getHostId(),
                ver.getHostName(), ver.getUserInfo(), verifyFaultLog, httpCode);
    }

    /**
     * Creates not VERIFIED verification info based on data from given
     * verification information object.
     *
     * @param ver ...
     * @return ...
     */
    public static UsersHostsInfo createVerifiedNow(UsersHostsInfo ver) {
        return createVerifiedNow(ver.getVerificationUin(), ver.getVerificationType(), ver.getUserId(), ver.getHostId(),
                ver.getHostName(), ver.getUserInfo());
    }

    public static UsersHostsInfo createVerifiedNow(UsersHostsInfo ver, VerificationTypeEnum verificationType) {
        return createVerifiedNow(ver.getVerificationUin(), verificationType, ver.getUserId(), ver.getHostId(),
                ver.getHostName(), ver.getUserInfo());
    }

    /**
     * Creates an instance of UsersHostsInfo for state hosts.
     * @param verificationUin Verification identificator in data-base.
     * @param verificationType Used type of verification.
     * @param userId ...
     * @param hostId ...
     * @param hostName ...
     * @param userInfo ...
     * @return Returns instance of verification information object.
     */
    public static UsersHostsInfo createVerifiedNow(long verificationUin, VerificationTypeEnum verificationType,
            long userId, long hostId, String hostName, UserInfo userInfo)
    {
        return new UsersHostsInfo(VerificationStateEnum.VERIFIED, verificationUin, verificationType, new Date(), null,
                userId, hostId, hostName, userInfo, null);
    }

    public static UsersHostsInfo createWaitingDNS(UsersHostsInfo info) {
        return new UsersHostsInfo(VerificationStateEnum.LONG_DNS_WAITING, info.getVerificationUin(),
                VerificationTypeEnum.DNS_RECORD, info.getVerificationDate(),
                info.getVerificationState().name() + ": " + info.getVerifyFaultLog(), info.getUserId(),
                info.getHostId(), info.getHostName(), info.getUserInfo(), null);
    }

    public long getVerificationUin() {
        return verificationUin;
    }

    public VerificationStateEnum getVerificationState() {
        return verificationState;
    }

    public String getHostName() {
        return hostName;
    }

    public VerificationTypeEnum getVerificationType() {
        return verificationType;
    }

    public Date getVerificationDate() {
        return verificationDate;
    }

    public UserInfo getUserInfo() {
        return userInfo;
    }

    public String getVerifyFaultLog() {
        return verifyFaultLog;
    }

    public Integer getHttpCode() {
        return httpCode;
    }
}
