package ru.yandex.wmconsole.data.partition;

import java.util.HashMap;
import java.util.Map;
import java.util.Random;

import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmtools.common.data.partition.AbstractPartition;
import ru.yandex.wmtools.common.data.partition.IPartition;

/**
 * This class simplifies use of multiple databases and partitions.
 * Anywhere hostname is passed as a parameter it should be in a 'canonical hostname' format.
 *
 * @see ru.yandex.wmtools.common.SupportedProtocols#getCanonicalHostname(java.net.URL)
 * @author baton
 */
public class WMCPartition extends AbstractPartition {
    private static final WMCPartition NULL_PARTITION = new WMCPartition(null, null, null);

    private static Map<String, Integer> hostIdPartMap = new HashMap<String, Integer>();
    private static Map<String, Integer> userIdPartMap = new HashMap<String, Integer>();
    private static Map<String, Integer> reportUserIdPartMap = new HashMap<String, Integer>();

    public static final int PARTITIONS_COUNT = 512;

    private static final Random r = new Random();

    static {
        // put here all tables that are partitioned.
        hostIdPartMap.put("tbl_url_trees", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_host_daily_state", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_sitemaps", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_sitemap_errors", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_sitemap_parse_errors", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_sitemaps_indexes", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_sitemaps_cur", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_sitemap_cur_errors", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_sitemap_cur_parse_errors", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_sitemaps_indexes_cur", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_error_urls", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_code_agr_error_trees", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_code_error_sitemap", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_error_urls_sitemaps", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_queries", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_query_stats", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_total_queries_stats", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_custom_queries", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_customquery_history", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_custom_regions", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_hostmon_stats", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_trend_tcy", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_trend_links_count", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_trend_robotdb_info_urls", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_trend_robotdb_info_index_count", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_trend_url_trees_urls", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_trend_url_trees_index_count", PARTITIONS_COUNT);

        hostIdPartMap.put("tbl_trend_code_error_trees_count", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_all_about_url_results", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_all_about_url_prod", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_all_about_url_cur", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_all_about_url_htarc", PARTITIONS_COUNT);
        hostIdPartMap.put("tbl_all_about_url_canonical", PARTITIONS_COUNT);

        userIdPartMap.put("tbl_messages", PARTITIONS_COUNT);
        userIdPartMap.put("tbl_user_host_error_options", PARTITIONS_COUNT);

        reportUserIdPartMap.put("tbl_queries", PARTITIONS_COUNT);
        reportUserIdPartMap.put("tbl_request_queries", PARTITIONS_COUNT);
    }

    private final Long hostDbHostId;
    private final String hostname;
    private final Integer databaseIndex;
    private final Long userId;

    public WMCPartition(HostDbHostInfo hostInfo, Long userId) {
        if (hostInfo == null) {
            this.hostDbHostId = null;
            this.hostname = null;
        } else {
            if (hostInfo.getName() == null) {
                throw new IllegalArgumentException("If you provide hostDbHostId, you should also provide hostname.");
            }

            this.hostDbHostId = hostInfo.getHostDbHostId();
            this.hostname = hostInfo.getName();
        }
        this.databaseIndex = null;
        this.userId = userId;
    }

    public WMCPartition(Long hostDbHostId, String hostname, Long userId) {
        if ((hostDbHostId != null) && (hostname == null)) {
            throw new IllegalArgumentException("If you provide hostDbHostId, you should also provide hostname.");
        }

        this.hostDbHostId = hostDbHostId;
        this.hostname = hostname;
        this.databaseIndex = null;
        this.userId = userId;
    }

    public WMCPartition(int databaseIndex) {
        this.hostDbHostId = null;
        this.hostname = null;
        this.databaseIndex = databaseIndex;
        this.userId = null;
    }

    public WMCPartition(int databaseIndex, long partIndex) {
        this.hostDbHostId = partIndex;
        this.hostname = null;
        this.databaseIndex = databaseIndex;
        this.userId = null;
    }

    public static WMCPartition nullPartition() {
        return NULL_PARTITION;
    }

    public static WMCPartition randomHostDatabase(int wholeCount) {
        return new WMCPartition(r.nextInt(getHostDbCount(wholeCount)));
    }

    @Override
    public String getReadySqlString(String sqlString) {
        if ((hostDbHostId == null) && (userId == null)) {
            return sqlString;
        }

        StringBuilder sb = new StringBuilder(sqlString);
        if (hostDbHostId != null) {
            partitionById(hostDbHostId, hostIdPartMap, sb);
        }
        if (userId != null) {
            partitionById(userId, userIdPartMap, sb);
            partitionById(userId, reportUserIdPartMap, sb);
        }
        return sb.toString();
    }

    @Override
    public int getDatabaseIndex(int baseCount) {
        if (databaseIndex != null) {
            return databaseIndex;
        }

        if (userId != null) {
            return baseCount - 1;
        }

        if (hostname == null) {
            return baseCount - 1;
        }

        return getStringHash(hostname, getHostDbCount(baseCount));
    }

    @Override
    public IPartition getIPartitionForDb(int dbIndex) {
        return new WMCPartition(dbIndex);
    }

    public static int getHostDbCount(int wholeCount) {
        return wholeCount - 1;
    }

    public static int getDatabaseIndex(int baseCount, String hostname) {
        if (hostname == null) {
            return baseCount - 1;
        }

        return getStringHash(hostname, getHostDbCount(baseCount));
    }

    @Override
    protected int getPartitionsCount() {
        return PARTITIONS_COUNT;
    }
}
