package ru.yandex.wmconsole.servantlet;

import java.net.IDN;

import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmconsole.data.HostDataLoadingWrapper;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.error.WMCExtraTagNameEnum;
import ru.yandex.wmconsole.service.HostInfoService;
import ru.yandex.wmconsole.service.UsersHostsService;
import ru.yandex.wmconsole.util.HostElementWrapper;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;

/**
 * @author ailyin
 */
public class BriefHostInfoHelper {
    public static final String PARAM_HOST = "host";

    private HostInfoService hostInfoService;
    private UsersHostsService usersHostsService;
    private boolean allowNotMainMirrors;

    public BriefHostInfo getHostInfo(ServRequest req, long userId, boolean assertHostAdded,
            boolean assertHostVerified) throws ClientException, InternalException {
        return getHostInfo(req, false, userId, assertHostAdded, assertHostVerified);
    }

    public BriefHostInfo getHostInfo(ServRequest req, boolean assertHostIndexed, long userId,
            boolean assertHostAdded, boolean assertHostVerified) throws ClientException, InternalException {
        BriefHostInfo briefHostInfo = getHostInfo(req);

        if (assertHostAdded) {
            final UsersHostsInfo usersHostsInfo = usersHostsService.getUsersHostsInfo(userId, briefHostInfo.getId());
            if (usersHostsInfo == null) {
                throw new ClientException(ClientProblem.HOST_NOT_OWNED_BY_USER, "Host is not added to the users host list");
            }
            if (assertHostVerified) {
                VerificationStateEnum verificationState = usersHostsInfo.getVerificationState();
                if (!verificationState.isVerified()) {
                    throw new ClientException(ClientProblem.HOST_NOT_VERIFIED_BY_USER,
                            "Host is not verified by specified user");
                }
            }
        }

        if (assertHostIndexed && !hostInfoService.isIndexed(briefHostInfo)) {
            throw new ClientException(ClientProblem.NO_DATA_FOR_HOST, "Host is not indexed");
        }

        return briefHostInfo;
    }

    public BriefHostInfo getHostInfo(ServRequest req, boolean assertHostIndexed, ServResponse res,
            boolean showHostDataFirstLoading, boolean returnNullIfHostDataFirstLoading)
            throws InternalException, UserException {
        return getHostInfo(req, assertHostIndexed, 0, false, false, res, showHostDataFirstLoading,
                returnNullIfHostDataFirstLoading);
    }

    public BriefHostInfo getHostInfo(ServRequest req, boolean assertHostIndexed, long userId,
            boolean assertHostAdded, boolean assertHostVerified, ServResponse res, boolean showHostDataFirstLoading,
            boolean returnNullIfHostDataFirstLoading) throws ClientException, InternalException {
        BriefHostInfo briefHostInfo = getHostInfo(req, assertHostIndexed, userId, assertHostAdded, assertHostVerified);

        if ((res != null) && (showHostDataFirstLoading || returnNullIfHostDataFirstLoading)) {
            if (hostInfoService.isHostDataFirstLoading(briefHostInfo)) {
                if (showHostDataFirstLoading) {
                    res.addData(new HostElementWrapper(new HostDataLoadingWrapper(), briefHostInfo));
                }
                if (returnNullIfHostDataFirstLoading) {
                    return null;
                }
            }
        }

        return briefHostInfo;
    }

    public BriefHostInfo getHostInfoAndVerify(ServRequest req, long userId) throws UserException, InternalException {
        return getHostInfo(req, userId, true, true);
    }

    public BriefHostInfo getHostInfoAndCheckHostAdded(ServRequest req, long userId)
            throws ClientException, InternalException {
        return getHostInfo(req, userId, true, false);
    }

    public BriefHostInfo getHostInfo(ServRequest req, boolean nullable) throws ClientException, InternalException {
        String host = req.getParam(PARAM_HOST, true);
        if (host == null) {
            throw new ClientException(ClientProblem.REQUIRED_PARAM_MISSED,
                    "Required param missed: " + PARAM_HOST, PARAM_HOST);
        }

        BriefHostInfo result = hostInfoService.getBriefHostInfoByIdOrName(IDN.toASCII(host));
        if (result == null) {
            if (!nullable) {
                throw new ClientException(ClientProblem.HOST_NOT_OWNED_BY_USER, "Host is not added to the users host list");
            }
        } else {
            if (!allowNotMainMirrors && result.getMainMirrorId() != null) {
                String mainMirrorName = hostInfoService.getHostNameByHostId(result.getMainMirrorId());
                throw new ClientException(ClientProblem.MIRROR_IS_NOT_ALLOWED, "Host is a mirror of " + result.getMainMirrorId(),
                        new ExtraTagInfo(WMCExtraTagNameEnum.ORIGINAL_HOST_ID, Long.toString(result.getId())),
                        new ExtraTagInfo(WMCExtraTagNameEnum.ORIGINAL_HOST_NAME, result.getName()),
                        new ExtraTagInfo(WMCExtraTagNameEnum.MAIN_MIRROR, mainMirrorName),
                        new ExtraTagInfo(WMCExtraTagNameEnum.MAIN_MIRROR_HOST_ID, Long.toString(result.getMainMirrorId())));
            }
        }
        return result;
    }

    public BriefHostInfo getHostInfo(ServRequest req) throws ClientException, InternalException {
        return getHostInfo(req, false);
    }

    @Required
    public void setHostInfoService(HostInfoService hostInfoService) {
        this.hostInfoService = hostInfoService;
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }

    @Required
    public void setAllowNotMainMirrors(boolean allowNotMainMirrors) {
        this.allowNotMainMirrors = allowNotMainMirrors;
    }
}
