package ru.yandex.wmconsole.service;

import java.net.MalformedURLException;
import java.net.URL;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.common.util.db.OrderByClause;
import ru.yandex.webmaster.common.host.dao.TblHostsHostDao;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.webmaster.common.urltree.YandexSearchShard;
import ru.yandex.wmconsole.data.HostInfoStatusEnum;
import ru.yandex.wmconsole.data.SeverityEnum;
import ru.yandex.wmconsole.data.UserErrorOptions;
import ru.yandex.wmconsole.data.info.*;
import ru.yandex.wmconsole.data.info.all.about.url.*;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.error.WMCExtraTagNameEnum;
import ru.yandex.wmconsole.service.dao.*;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmconsole.util.UrlErrorGrouper;
import ru.yandex.wmconsole.util.WwwUtil;
import ru.yandex.wmtools.common.data.xmlsearch.FastIndexCountRequest;
import ru.yandex.wmtools.common.data.xmlsearch.IndexCountRequest;
import ru.yandex.wmtools.common.data.xmlsearch.InternalLinksCountRequest;
import ru.yandex.wmtools.common.data.xmlsearch.LinksCountRequest;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.servantlet.AbstractServantlet;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.service.IndexInfoService;
import ru.yandex.wmtools.common.util.ServiceTransactionCallbackWithoutResult;
import ru.yandex.wmtools.common.util.SqlUtil;
import ru.yandex.wmtools.common.util.URLUtil;

/**
 * All Information About Url
 *
 * see http://wiki.yandex-team.ru/JandeksPoisk/Interfejjsy/CentrWebmastera/Specs/VseOStranice
 *
 * User: azakharov
 * Date: 08.02.13
 * Time: 18:05
 */
public class AllAboutUrlService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(AllAboutUrlService.class);

    private static final String FIELD_NUM = "num";
    private static final String FIELD_URL = "url";
    private static final String FIELD_NORMALIZED_URL = "normalized_url";
    private static final String FIELD_CREATE_DATE = "create_date";
    private static final String FIELD_IS_ERROR = "is_error";
    private static final String FIELD_IS_FROM_FAST_ROBOT = "is_from_fast_robot";
    private static final String FIELD_HOST_INFO_STATUS = "host_info_status";
    private static final String FIELD_PENALTY_LAST_ACCESS = "penalty_last_access";
    private static final String FIELD_SHARD_ID = "search_shard_id";
    private static final String FIELD_SEARCHABLE_XML = "searchable_xml";
    private static final String FIELD_SEARCHABLE_XML_TIME = "searchable_xml_time";
    private static final String FIELD_LINKS_COUNT = "links_count";
    private static final String FIELD_INTERNAL_LINKS_COUNT = "internal_links_count";

    private static final String FIELD_CANONICAL_URLS_COUNT = "canonical_urls_count";

    private static final String FIELD_PROD_LAST_ACCESS = "prod_last_access";
    private static final String FIELD_PROD_HTTP_CODE = "prod_http_code";
    private static final String FIELD_PROD_SEARCH_URL = "prod_search_base_main_url";
    private static final String FIELD_PROD_REDIR_TARGET = "prod_redir_target";
    private static final String FIELD_PROD_REL_CANONICAL_TARGET = "prod_rel_canonical_target";
    private static final String FIELD_PROD_STATUS = "prod_status";
    private static final String FIELD_PROD_SEARCHABLE = "prod_is_searchable";
    private static final String FIELD_PROD_NO_UPLOAD_REASON = "prod_no_upload_reason";
    private static final String FIELD_PROD_LINKS = "prod_total_links";
    private static final String FIELD_PROD_INTERNAL_LINKS = "prod_total_internal_links";
    private static final String FIELD_PROD_UPLOAD_RANK_VALUES = "prod_upload_rank_values";

    private static final String FIELD_CUR_LAST_ACCESS = "cur_last_access";
    private static final String FIELD_CUR_HTTP_CODE = "cur_http_code";
    private static final String FIELD_CUR_SEARCH_URL = "cur_search_base_main_url";
    private static final String FIELD_CUR_REDIR_TARGET = "cur_redir_target";
    private static final String FIELD_CUR_REL_CANONICAL_TARGET = "cur_rel_canonical_target";
    private static final String FIELD_CUR_STATUS = "cur_status";
    private static final String FIELD_CUR_SEARCHABLE = "cur_is_searchable";
    private static final String FIELD_CUR_NO_UPLOAD_REASON = "cur_no_upload_reason";
    private static final String FIELD_CUR_LINKS = "cur_total_links";
    private static final String FIELD_CUR_INTERNAL_LINKS = "cur_total_internal_links";
    private static final String FIELD_CUR_UPLOAD_RANK_VALUES = "cur_upload_rank_values";

    private static final String FIELD_HTARC_LAST_ACCESS = "htarc_last_access";
    private static final String FIELD_HTARC_HTTP_CODE = "htarc_http_code";
    private static final String FIELD_HTARC_STATUS = "htarc_status";

    private static final String SELECT_URL_INFO_QUERY =
            "SELECT " +
                    "       r.num AS " + FIELD_NUM + ", " +
                    "       r.url AS " + FIELD_URL + ", " +
                    "       r.is_error AS " + FIELD_IS_ERROR + ", " +
                    "       r.normalized_url AS " + FIELD_NORMALIZED_URL + ", " +
                    "       r.create_date AS " + FIELD_CREATE_DATE + ", " +
                    "       r.is_from_fast_robot AS " + FIELD_IS_FROM_FAST_ROBOT + ", " +
                    "       r.host_info_status AS " + FIELD_HOST_INFO_STATUS + ", " +
                    "       r.penalty_last_access AS " + FIELD_PENALTY_LAST_ACCESS + ", " +
                    "       r.links_count AS " + FIELD_LINKS_COUNT + ", " +
                    "       r.internal_links_count AS " + FIELD_INTERNAL_LINKS_COUNT + ", " +
                    "       r.searchable_xml AS " + FIELD_SEARCHABLE_XML + ", " +
                    "       r.searchable_xml_time AS " + FIELD_SEARCHABLE_XML_TIME + ", " +
                    "       r.shard_id AS " + FIELD_SHARD_ID + ", " +
                    "       (SELECT COUNT(n.url) FROM tbl_all_about_url_canonical n " +
                    "           WHERE n.num = r.num) AS " + FIELD_CANONICAL_URLS_COUNT + ", " +
                    "       p.last_access AS " + FIELD_PROD_LAST_ACCESS + ", " +
                    "       p.http_code AS " + FIELD_PROD_HTTP_CODE + ", " +
                    "       p.search_base_main_url AS " + FIELD_PROD_SEARCH_URL + ", " +
                    "       p.redir_target AS " + FIELD_PROD_REDIR_TARGET + ", " +
                    "       p.rel_canonical_target AS " + FIELD_PROD_REL_CANONICAL_TARGET + ", " +
                    "       p.status AS " + FIELD_PROD_STATUS + "," +
                    "       p.is_searchable AS " + FIELD_PROD_SEARCHABLE + ", " +
                    "       p.no_upload_reason AS " + FIELD_PROD_NO_UPLOAD_REASON + ", " +
                    "       p.upload_rank_values AS " + FIELD_PROD_UPLOAD_RANK_VALUES + ", " +
                    "       p.total_links AS " + FIELD_PROD_LINKS + ", " +
                    "       p.total_internal_links AS " + FIELD_PROD_INTERNAL_LINKS + ", " +
                    "       c.last_access AS " + FIELD_CUR_LAST_ACCESS + ", " +
                    "       c.http_code AS " + FIELD_CUR_HTTP_CODE + ", " +
                    "       c.search_base_main_url AS " + FIELD_CUR_SEARCH_URL + ", " +
                    "       c.redir_target AS " + FIELD_CUR_REDIR_TARGET + ", " +
                    "       c.rel_canonical_target AS " + FIELD_CUR_REL_CANONICAL_TARGET + ", " +
                    "       c.status AS " + FIELD_CUR_STATUS + "," +
                    "       c.is_searchable AS " + FIELD_CUR_SEARCHABLE + ", " +
                    "       c.no_upload_reason AS " + FIELD_CUR_NO_UPLOAD_REASON + ", " +
                    "       c.upload_rank_values AS " + FIELD_CUR_UPLOAD_RANK_VALUES + ", " +
                    "       c.total_links AS " + FIELD_CUR_LINKS + ", " +
                    "       c.total_internal_links AS " + FIELD_CUR_INTERNAL_LINKS + ", " +
                    "       h.http_code AS " + FIELD_HTARC_HTTP_CODE + ", " +
                    "       h.last_access AS " + FIELD_HTARC_LAST_ACCESS + ", " +
                    "       h.status AS " + FIELD_HTARC_STATUS + " " +
                    "FROM " +
                    "       tbl_all_about_url_results r " +
                    "LEFT JOIN " +
                    "       tbl_all_about_url_prod p " +
                    "ON " +
                    "       r.num = p.num " +
                    "LEFT JOIN " +
                    "       tbl_all_about_url_cur c " +
                    "ON " +
                    "       r.num = c.num " +
                    "LEFT JOIN " +
                    "       tbl_all_about_url_htarc h " +
                    "ON " +
                    "       r.num = h.num " +
                    "WHERE " +
                    "       r.host_id = ? " +
                    "AND " +
                    "       r.num = ? " +
                    "AND " +
                    "       r.from_support = ? ";

    private HostDbHostInfoService hostDbHostInfoService;
    private IndexInfoService indexInfoService;
    private IndexInfoService uaIndexInfoService;
    private IndexInfoService comIndexInfoService;
    private IndexInfoService trIndexInfoService;
    private Integer reportCountConstraint;
    private TblAllAboutUrlResultsDao tblAllAboutUrlResultsDao;
    private TblAllAboutUrlQueueDao tblAllAboutUrlQueueDao;
    private TblAllAboutUrlCanonicalDao tblAllAboutUrlCanonicalDao;
    private TblAllAboutUrlProdDao tblAllAboutUrlProdDao;
    private TblAllAboutUrlCurDao tblAllAboutUrlCurDao;
    private TblAllAboutUrlHtarcDao tblAllAboutUrlHtarcDao;
    private TblPenaltyInfoDao tblPenaltyInfoDao;
    private TblHostInfoDao tblHostInfoDao;
    private TblHostsHostDao tblHostsHostDao;
    private TblHostsMainDao tblHostsMainDao;
    private UsersHostsService usersHostsService;
    private TblUrlTreesDao tblUrlTreesDao;

    /**
     * Запросить отчет по URL. Ставит URL в очередь для обработки.
     *
     * @param url   запрашиваемый URL
     * @throws InternalException
     */
    public void addUrlRequest(final BriefHostInfo hostInfo, final URL url, final boolean fromSupport) throws InternalException, UserException {
        // Сохраняем запросы для www!www сайтов в контексте того сайта, для которого делали запрос
        final HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        final WMCPartition partition = new WMCPartition(hostDbHostInfo, null);

        getServiceTransactionTemplate(partition).executeInService(new ServiceTransactionCallbackWithoutResult() {
            @Override
            protected void doInTransactionWithoutResult(TransactionStatus transactionStatus) throws UserException, InternalException {

                tblAllAboutUrlQueueDao.insertRequest(hostDbHostInfo, url.toString(), fromSupport);

                int reportsCount = tblAllAboutUrlResultsDao.getUrlInfoResultsCount(hostDbHostInfo, fromSupport);
                int queuedCount = tblAllAboutUrlQueueDao.getUrlInfoRequestCount(hostDbHostInfo, fromSupport);

                if (reportsCount + queuedCount > reportCountConstraint) {
                    int removeCount = reportsCount + queuedCount - reportCountConstraint;

                    // Удаляем отчеты, если есть
                    if (reportsCount > 0) {
                        int limit = Math.min(removeCount, reportsCount);
                        List<Long> removedReports = tblAllAboutUrlResultsDao.getOldestUrlInfoResults(hostDbHostInfo, limit, fromSupport);

                        tblAllAboutUrlResultsDao.deleteUrlInfoResults(hostDbHostInfo, removedReports);
                    }

                    // Если все отчеты удалили, удаляем из очереди
                    if (removeCount > reportsCount) {
                        List<Long> removedFromQueue = tblAllAboutUrlQueueDao.getOldestUrlInfoRequests(
                                hostDbHostInfo, Math.min(removeCount - reportsCount, queuedCount), fromSupport);
                        tblAllAboutUrlQueueDao.deleteUrlInfoRequests(hostDbHostInfo, removedFromQueue);
                    }
                }
            }
        });
    }

    /**
     * Проверяет, что URL относится к хосту.
     * Для случая www!www хостов осуществляет приведение URL к хосту.
     *
     * @param hostInfo  информация о хосте из пользовательской базы
     * @param urlParam  необработанный параметр url из HTTP-запроса
     * @return          объект URL
     * @throws UserException
     */
    public URL getUrlAndCheckUrlIsFromHost(final BriefHostInfo hostInfo, String urlParam, final long userId, final boolean fromSupport) throws UserException, InternalException {
        if (urlParam.startsWith("/")) {
            urlParam = hostInfo.getName() + urlParam;
        }
        URL url = AbstractServantlet.prepareUrl(urlParam, true);
        final String hostName = URLUtil.getHostName(url, false);

        if (!WwwUtil.equalsIgnoreWww(hostName, hostInfo.getName())) {
            BriefHostInfo hostInfoFromUrl = tblHostsMainDao.getHostIdByHostname(hostName);
            if (hostInfoFromUrl == null || hostInfoFromUrl.getMainMirrorId() != null || fromSupport) {
                // Сайт из URL отсутствует в базе или является неглавным зеркалом или пытаемся добавить из админки для другого хоста
                throw new ClientException(ClientProblem.URL_IS_NOT_FROM_HOST, "url " + urlParam + " is not from host " + hostInfo.getName());
            }
            final UsersHostsInfo usersHostsInfo = usersHostsService.getUsersHostsInfo(userId, hostInfoFromUrl.getId());
            if (usersHostsInfo != null && usersHostsInfo.getVerificationState() != null && usersHostsInfo.getVerificationState().isVerified()) {
                // Предлагаем добавить URL в контексте сайта из URL
                throw new ClientException(ClientProblem.URL_IS_NOT_FROM_HOST_REDIR,
                        "url " + urlParam + " is not from host " + hostInfo.getName(),
                        new ExtraTagInfo(WMCExtraTagNameEnum.SUGGESTED_HOST_ID, Long.toString(hostInfoFromUrl.getId())),
                        new ExtraTagInfo(WMCExtraTagNameEnum.SUGGESTED_HOST_NAME, hostInfoFromUrl.getName()));
            } else {
                // Сайт из URL не добавлен или не подтверждён
                throw new ClientException(ClientProblem.URL_IS_NOT_FROM_HOST, "url " + urlParam + " is not from host " + hostInfo.getName());
            }
        } else if (!hostName.equalsIgnoreCase(hostInfo.getName())) {
            try {
                url = new URL(url.getProtocol(), WwwUtil.switchWWW(url.getHost()), url.getPort(), url.getFile());
            } catch (MalformedURLException e) {
                log.error("Malformed url for urlParam=" + urlParam + " and url=" + url.toString());
            }
        }

        if (URLUtil.isHomePage(url)) {
            if (! "/".equals(url.getPath())) {
                try {
                    url = new URL(url.toExternalForm() + "/");
                } catch (MalformedURLException e) {
                    log.error("Malformed url for home page urlParam=" + urlParam + " and url=" + url.toString());
                }
            }
        }

        return url;
    }

    /**
     * Удалить отчет
     */
    public void removeUrlReport(final BriefHostInfo hostInfo, final UrlRequestId reportId, final boolean fromSupport) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        tblAllAboutUrlQueueDao.deleteUrlInfoRequest(hostDbHostInfo, reportId, fromSupport);
        tblAllAboutUrlResultsDao.deleteUrlInfoResult(hostDbHostInfo, reportId, fromSupport);
    }

    /**
     * Получить историю запросов информации об урлах заданного хоста
     */
    public List<ShortUrlInfo> getUrlInfoList(
            final BriefHostInfo briefHostInfo,
            final List<Long> ids,
            final @Nullable OrderByClause orderByClause,
            final boolean fromSupport) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());
        return getUrlInfoList(hostDbHostInfo, ids, orderByClause, fromSupport);
    }

    /**
     * Получить историю запросов информации об урлах заданного хоста
     */
    public List<ShortUrlInfo> getUrlInfoList(final HostDbHostInfo hostDbHostInfo,
                                             final List<Long> ids,
                                             final @Nullable OrderByClause orderByClause,
                                             final boolean fromSupport) throws InternalException {
        final List<ShortUrlInfo> queue;
        final List<ShortUrlInfo> ready;
        if (ids.isEmpty()) {
            queue = tblAllAboutUrlQueueDao.getUrlInfoQueue(hostDbHostInfo, fromSupport);
            ready = tblAllAboutUrlResultsDao.getUrlInfoResultsList(hostDbHostInfo, fromSupport);
        } else {
            queue = tblAllAboutUrlQueueDao.getUrlInfoQueue(hostDbHostInfo, ids, fromSupport);
            ready = tblAllAboutUrlResultsDao.getUrlInfoResultsList(hostDbHostInfo, ids, fromSupport);
        }
        queue.addAll(ready);
        if (orderByClause != null) {
            Collections.sort(queue, new UrlInfoComparator(orderByClause));
        }
        return queue;
    }

    /**
     * Получить историю запросов конкретного урла для заданного хоста
     * @param id  идентификатор отчёта
     */
    public List<ShortUrlInfo> getUrlInfoHistory(BriefHostInfo hostInfo, UrlRequestId id, @Nullable OrderByClause orderByClause, boolean fromSupport) throws InternalException, UserException {

        // Получаем хост в хостовой базе
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());

        // Получаем отчёт
        ShortUrlInfo urlInfo = tblAllAboutUrlResultsDao.getUrlInfoResultSummary(hostDbHostInfo, id, fromSupport);
        if (urlInfo == null) {
            urlInfo = tblAllAboutUrlQueueDao.getUrlInfoQueueSummary(hostDbHostInfo, id, fromSupport);
        }

        if (urlInfo == null) {
            throw new UserException(WMCUserProblem.URL_INFO_NO_SUCH_REPORT, "No report with id = " + id.getId() + " available");
        }
        if (UrlInfoStatusEnum.ERROR.equals(urlInfo.getStatus())) {
            throw new UserException(WMCUserProblem.URL_INFO_ERROR, "Requested report is not ready due to error");
        }

        // запрашиваем заявки
        List<ShortUrlInfo> queue = tblAllAboutUrlQueueDao.getUrlInfoQueueForUrl(hostDbHostInfo, urlInfo.getUrl(), fromSupport);

        // запрашиваем готовые отчёты
        List<ShortUrlInfo> ready = tblAllAboutUrlResultsDao.getUrlInfoResultsForUrl(hostDbHostInfo, urlInfo.getUrl(), fromSupport);
        queue.addAll(ready);

        // Сортируем, если нужно
        if (orderByClause != null) {
            Collections.sort(queue, new UrlInfoComparator(orderByClause));
        }

        return queue;
    }

    /**
     * Получить информацию об URL (готовый отчет для верстки)
     *
     * @param hostInfo      информация о хосте из пользовательской базы
     * @param id            идентификатор отчета
     * @param fromSupport   запрос через интерфейс поддержки
     * @return              отчет
     * @throws InternalException
     * @throws UserException
     */
    public UrlInfo getUrlInfo(BriefHostInfo hostInfo, UrlRequestId id, boolean fromSupport) throws InternalException, UserException {
        final HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        return getUrlInfo(hostDbHostInfo, id, fromSupport);
    }

    /**
     * Получить информацию об URL (готовый отчет для верстки)
     *
     * @param hostDbHostInfo    информация о хосте из хостовой базы
     * @param id                идентификатор отчета
     * @param fromSupport       запрос через интерфейс поддержки
     * @return                  отчет
     * @throws InternalException
     * @throws UserException
     */
    public UrlInfo getUrlInfo(HostDbHostInfo hostDbHostInfo, UrlRequestId id, boolean fromSupport) throws InternalException, UserException {
        DbUrlInfo dbUrlInfo = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).safeQueryForObject(
                SELECT_URL_INFO_QUERY, urlInfoMapper, hostDbHostInfo.getHostDbHostId(), id.getId(), fromSupport);
        if (dbUrlInfo == null) {
            throw new UserException(WMCUserProblem.URL_INFO_NO_SUCH_REPORT, "No report with id = " + id.getId() + " available");
        }

        if (UrlInfoStatusEnum.ERROR.equals(dbUrlInfo.getUrlProcessingStatus()) ||
                UrlInfoStatusEnum.PROCESSING.equals(dbUrlInfo.getUrlProcessingStatus())) {
            throw new UserException(WMCUserProblem.URL_INFO_ERROR, "Requested report is not ready");
        }

        if (dbUrlInfo.getRelCanonicalTargetsCount() > 0) {
            List<String> canonicals = tblAllAboutUrlCanonicalDao.getRelCanonicalTargetUrls(hostDbHostInfo, id);
            dbUrlInfo.setRelCanonicalTargetUrls(canonicals);
        }

        final UrlStatusEnum status = calculateStatus(dbUrlInfo);

        return createUrlInfo(status, dbUrlInfo);
    }

    /**
     * http://wiki.yandex-team.ru/JandeksPoisk/Interfejjsy/CentrWebmastera/Specs/VseOStranice/codediscription
     *
     * @param dbUrlInfo     сырая информация об URL из базы данных
     * @return              статус по схеме
     */
    public UrlStatusEnum calculateStatus(final DbUrlInfo dbUrlInfo) {
        final DbPartialUrlInfo prodUrlInfo = dbUrlInfo.getProdUrlInfo();
        final DbPartialUrlInfo curUrlInfo = dbUrlInfo.getCurUrlInfo();
        final DbPartialUrlInfo htarcUrlInfo = dbUrlInfo.getHtarcUrlInfo();

        final UrlStatusEnum status;

        if (hasKiwiProdData(prodUrlInfo)) { // Есть ли данные в киви prod
            if (!hasData(curUrlInfo)) {
                log.warn("All about url logic error: has data in kiwi prod, but has no data in kiwi cur");
            }

            if (prodUrlInfo.getSearchable()) {
                if (!dbUrlInfo.getSearchableFromXmlSearch()) {
                    status = UrlStatusEnum.SEMIDUP_OR_ANTISPAM;
                } else if (!hasHttpCode(prodUrlInfo)) {
                    status = UrlStatusEnum.SEARCHABLE_FAKE_PROD;
                } else if (prodUrlInfo.getSearchBaseMainUrl() != null &&
                        prodUrlInfo.getSearchBaseMainUrl().equalsIgnoreCase(dbUrlInfo.getUrl())) {
                    status = UrlStatusEnum.SEARCHABLE_PROD;
                } else if (isRedirect(prodUrlInfo.getHttpCode())) {
                    status = UrlStatusEnum.SEARCHABLE_REDIRECT_PROD;
                } else if (isCode(prodUrlInfo.getHttpCode(), 2004)) {
                    status = UrlStatusEnum.SEARCHABLE_REFRESH_PROD;
                } else if (equalsIgnoreSlash(prodUrlInfo.getSearchBaseMainUrl(), dbUrlInfo.getUrl())) {
                    status = UrlStatusEnum.SEARCHABLE_DUPLICATE_SLASH;
                } else {
                    status = UrlStatusEnum.SEARCHABLE_DUPLICATE_PROD;
                }
            } else {
                if (hasHtarcData(htarcUrlInfo)) {
                    if (isCode(prodUrlInfo.getHttpCode(), 200) || isCode(prodUrlInfo.getHttpCode(), 304)) {
                        if (isSR(prodUrlInfo.getNoUploadReason()) ||
                                isCode(htarcUrlInfo.getHttpCode(), 3021)) {
                            status = UrlStatusEnum.SELECTION_RANK;
                        } else if (isCode(htarcUrlInfo.getHttpCode(), 2025)) {
                            status = UrlStatusEnum.NOT_CANONICAL;
                        } else if (isCode(htarcUrlInfo.getHttpCode(), 2021)) {
                            status = UrlStatusEnum.CLEAN_PARAMS;
                        } else if (dbUrlInfo.getUrl().equalsIgnoreCase(prodUrlInfo.getSearchBaseMainUrl()) ||
                                       prodUrlInfo.getSearchBaseMainUrl() == null) {
                            status = UrlStatusEnum.SEMIDUP_OR_ANTISPAM;
                        } else if (dbUrlInfo.getRelCanonicalTargetUrls().isEmpty() ||
                                containsIgnoreCase(dbUrlInfo.getRelCanonicalTargetUrls(), dbUrlInfo.getUrl())) {
                            // rel_canonical_target пуст или содержит URL из запроса
                            status = UrlStatusEnum.DUPLICATE_PROD;
                        } else {
                            status = UrlStatusEnum.NOT_CANONICAL_PROD;
                        }
                    } else if (isRedirect(prodUrlInfo.getHttpCode())) {
                        status = UrlStatusEnum.REDIRECT_PROD;
                    } else if (isCode(prodUrlInfo.getHttpCode(), 2004)) {
                        status = UrlStatusEnum.REFRESH_PROD;
                    } else if (isSiteError(prodUrlInfo.getHttpCode())) {
                        status = UrlStatusEnum.SITE_ERROR_PROD;
                    } else {
                        status = UrlStatusEnum.SEMIDUP_OR_ANTISPAM;
                    }
                } else {
                    log.warn("All about url logic error: has data kiwi prod and has no data in htarc");

                    if (isCode(prodUrlInfo.getHttpCode(), 200) || isCode(prodUrlInfo.getHttpCode(), 304)) {
                        if (isSR(prodUrlInfo.getNoUploadReason())) {
                            status = UrlStatusEnum.SELECTION_RANK;
                        } else if (dbUrlInfo.getUrl().equalsIgnoreCase(prodUrlInfo.getSearchBaseMainUrl()) ||
                                prodUrlInfo.getSearchBaseMainUrl() == null) {
                            status = UrlStatusEnum.SEMIDUP_OR_ANTISPAM;
                        } else if (dbUrlInfo.getRelCanonicalTargetUrls().isEmpty() ||
                                containsIgnoreCase(dbUrlInfo.getRelCanonicalTargetUrls(), dbUrlInfo.getUrl())) {
                            // rel_canonical_target пуст или содержит URL из запроса
                            status = UrlStatusEnum.DUPLICATE_PROD;
                        } else {
                            status = UrlStatusEnum.NOT_CANONICAL_PROD;
                        }
                    } else if (isRedirect(prodUrlInfo.getHttpCode())) {
                        status = UrlStatusEnum.REDIRECT_PROD;
                    } else if (isCode(prodUrlInfo.getHttpCode(), 2004)) {
                        status = UrlStatusEnum.REFRESH_PROD;
                    } else if (isSiteError(prodUrlInfo.getHttpCode())) {
                        status = UrlStatusEnum.SITE_ERROR_PROD;
                    } else {
                        status = UrlStatusEnum.SEMIDUP_OR_ANTISPAM;
                    }
                }
            }
        } else if (hasHtarcData(htarcUrlInfo)) { // Есть ли данные на htarc
            if (dbUrlInfo.getSearchableFromXmlSearch()) {
                status = isFake(htarcUrlInfo) ? UrlStatusEnum.SEARCHABLE_FAKE_HTARC : UrlStatusEnum.SEARCHABLE_HTARC;
            } else {
                if (isCode(htarcUrlInfo.getHttpCode(), 3021)) {
                    status = UrlStatusEnum.SELECTION_RANK;
                } else if (isCode(htarcUrlInfo.getHttpCode(), 200) || isCode(htarcUrlInfo.getHttpCode(), 304)) {
                    status = UrlStatusEnum.SEMIDUP_OR_ANTISPAM;
                } else if (isRedirect(htarcUrlInfo.getHttpCode())) {
                    status = UrlStatusEnum.REDIRECT_HTARC;
                } else if (isCode(htarcUrlInfo.getHttpCode(), 2004)) {
                    status = UrlStatusEnum.REFRESH_HTARC;
                } else if (isCode(htarcUrlInfo.getHttpCode(), 2025)) {
                    // канонического URL не будет
                    status = UrlStatusEnum.NOT_CANONICAL_HTARC;
                } else if (isSiteError(htarcUrlInfo.getHttpCode())) {
                    status = UrlStatusEnum.SITE_ERROR_HTARC;
                } else {
                    status = UrlStatusEnum.SEMIDUP_OR_ANTISPAM;
                }
            }
        } else if (hasData(curUrlInfo)) { // Есть ли данные в киви cur
            status = isSiteError(curUrlInfo.getHttpCode()) ? UrlStatusEnum.NOT_CRAWLED_SITE_ERROR : UrlStatusEnum.NOT_CRAWLED;
        } else {
            status = UrlStatusEnum.NEW_NOT_CRAWLED;
        }
        return status;
    }

    /**
     * Получает данные для отчёта из различных источников: xml-поиск, MySQL база данных вебмастера
     *
     * @param db        номер базы данных
     * @param partition номер партиции
     * @throws InternalException
     */
    public void updateNonKiwiData(final int db, final int partition) throws InternalException {
        final List<UrlResultInfo> reports = tblAllAboutUrlResultsDao.getReportsWaitingForExtraData(db, partition);
        HostDbHostInfo hostDbHostInfo = null;
        // update necessary data for reports
        for (UrlResultInfo urlResultInfo : reports) {
            try {
                long hostId = urlResultInfo.getHostId();

                // get hostDbHostInfo
                if (hostDbHostInfo == null || hostDbHostInfo.getHostDbHostId() != hostId) {
                    hostDbHostInfo = tblHostsHostDao.getHostDbHostInfoByHostId(db, hostId);
                }
                // try update xmlsearch data
                updateXmlSearchData(hostDbHostInfo, urlResultInfo);

                // try update penalty
                updatePenalty(hostDbHostInfo, urlResultInfo);

                final UrlInfo urlInfo = getUrlInfo(hostDbHostInfo, urlResultInfo.getRequestId(), urlResultInfo.isFromSupport());
                log.info("ALL_ABOUT_URL_STATUS {} for {} ({}, {})",
                        urlInfo.getStatus(), urlInfo.getUrl(), hostDbHostInfo.getName(), urlInfo.getReportId());
            } catch (Exception e) {
               log.error("Exception in updateNonKiwiData", e);
            }
        }

        // mark old unprocessed reports as erroneous
        tblAllAboutUrlResultsDao.updateOldReportsWaitingForExtraData(db, partition);
    }

    public Map<String, Map<String, Object>> dumpAllAboutUrlInfo(BriefHostInfo hostInfo, UrlRequestId id) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());

        Map<String, Object> queueInfo = tblAllAboutUrlQueueDao.dumpAllAboutUrlQueueInfo(hostDbHostInfo, id);
        Map<String, Object> resultsInfo = tblAllAboutUrlResultsDao.dumpAllAboutUrlResultsInfo(hostDbHostInfo, id);
        Map<String, Object> prodInfo = tblAllAboutUrlProdDao.dumpAllAboutUrlProdInfo(hostDbHostInfo, id);
        Map<String, Object> curInfo = tblAllAboutUrlCurDao.dumpAllAboutUrlCurInfo(hostDbHostInfo, id);
        Map<String, Object> htarcInfo = tblAllAboutUrlHtarcDao.dumpAllAboutUrlHtarcInfo(hostDbHostInfo, id);
        Map<String, Object> canonicalInfo = tblAllAboutUrlCanonicalDao.dumpAllAboutUrlCanonicalInfo(hostDbHostInfo, id);

        Map<String, Map<String, Object>> r = new LinkedHashMap<>();
        r.put("tbl_all_about_url_queue", queueInfo);
        r.put("tbl_all_about_url_results", resultsInfo);
        r.put("tbl_all_about_url_prod", prodInfo);
        r.put("tbl_all_about_url_cur", curInfo);
        r.put("tbl_all_about_url_htarc", htarcInfo);
        r.put("tbl_all_about_url_canonical", canonicalInfo);

        return r;
    }

    private void updateXmlSearchData(final HostDbHostInfo hostDbHostInfo, final UrlResultInfo urlResultInfo) throws InternalException {
        if (urlResultInfo.getXmlsearchDataUpdateTime() != null) {
            // already updated
            return;
        }
        YandexSearchShard shard = urlResultInfo.getSearchShard();
        if (shard == null) {
            shard = tblUrlTreesDao.getOptimumShardId(hostDbHostInfo);
            // save shard_id for dispaying to client and for future requests to xmlsearch
            tblAllAboutUrlResultsDao.updateSearchShard(hostDbHostInfo, urlResultInfo.getRequestId(), shard);
        }

        // update searchable fast robot
        boolean res0 = updateFastIndexCount(hostDbHostInfo, urlResultInfo, shard);
        // update searchable xmlsearch
        boolean res1 = updateSearchableFromXmlSearch(hostDbHostInfo, urlResultInfo, shard);
        // update links count and internal links count
        boolean res2 = updateLinksCount(hostDbHostInfo, urlResultInfo, shard);

        if (res0 && res1 && res2) {
            // save timestamp only if all operations successful otherwise we will retry one more time
            tblAllAboutUrlResultsDao.updateXmlSearchDataUpdateTime(hostDbHostInfo, urlResultInfo.getRequestId());
        }
    }

    private void updatePenalty(final HostDbHostInfo hostDbHostInfo, final UrlResultInfo urlResultInfo) throws InternalException {
        if (urlResultInfo.getPenaltyLastAccess() != null) {
            // already updated
            return;
        }

        PenaltyInfo penaltyInfo = tblPenaltyInfoDao.getPenaltyInfo(hostDbHostInfo);
        Pair<HostInfoStatusEnum, Date> statusAndDate = tblHostInfoDao.getHostInfo(hostDbHostInfo);
        HostInfoStatusEnum status = statusAndDate != null ? statusAndDate.getFirst() : null;
        if (status != HostInfoStatusEnum.ROBOTS_TXT) {
            // in tbl_host_info table we interested only in ROBOTS_TXT status
            status = null;
        }
        Date statusDate = statusAndDate != null ? statusAndDate.getSecond() : null;
        if (penaltyInfo != null) {
            if (statusAndDate != null) {
                penaltyInfo = penaltyInfo.mergeHostStatus(status, statusDate);
            }
        } else {
            penaltyInfo = new PenaltyInfo(
                    new HostStatusInfo(null, null, status, null, null, true, true),
                    null, null, statusDate);
        }
        Date penaltyLastAccess = penaltyInfo.getPenaltyLastAccess();
        HostInfoStatusEnum hostInfoStatus = penaltyInfo.getCalculatedStatus();
        if (penaltyLastAccess == null) {
            // we should set not null date to prevent recalculating penalty info next time
            penaltyLastAccess = new Date();
        }

        try {
            tblAllAboutUrlResultsDao.updateHostInfoStatus(hostDbHostInfo, urlResultInfo.getRequestId(), hostInfoStatus, penaltyLastAccess);
        } catch (InternalException e) {
            log.error("update host info status failed", e);
        }
    }

    private boolean updateLinksCount(final HostDbHostInfo hostDbHostInfo, final UrlResultInfo urlResultInfo, final YandexSearchShard shard) {
        if (urlResultInfo.getLinksCountXml() != null) {
            return true;
        }

        try {
            final URL url = AbstractServantlet.prepareUrl(urlResultInfo.safeGetNormalizedUrl(), true);
            final String hostName = URLUtil.getHostName(url, false);
            final String path = URLUtil.getRelativeUrl(url);
            final IndexInfoService countryIndexInfoService = getIndexInfoServiceForShard(shard);
            final Long linksCount = countryIndexInfoService.extractLinksCount(new LinksCountRequest(hostName, path, 0, 10));
            final Long internalLinksCount = countryIndexInfoService.extractLinksCount(new InternalLinksCountRequest(hostName, path, 0, 10));
            if (linksCount == null || internalLinksCount == null) {
                return false;
            }
            try {
                tblAllAboutUrlResultsDao.updateLinksCount(hostDbHostInfo, urlResultInfo.getRequestId(), linksCount, internalLinksCount);
                return true;
            } catch (InternalException e) {
                log.error("update fast robot info failed", e);
                return false;
            }
        } catch (Exception e) {
            log.error("Exception in updateFastIndexCount" , e);
            return false;
        }
    }

    /**
     * Пытается получить информацию из XML-поиска, если поле is_from_fast_robot содержит NULL
     * и сохранить эту информацию обратно в базу.
     */
    private boolean updateFastIndexCount(final HostDbHostInfo hostDbHostInfo, final UrlResultInfo urlResultInfo, final YandexSearchShard searchShard) {
        if (urlResultInfo.getSearchableInFastRobot() != null) {
            return true;
        }

        try {
            final URL url = AbstractServantlet.prepareUrl(urlResultInfo.safeGetNormalizedUrl(), true);
            final String hostName = URLUtil.getHostName(url, false);
            final String path = URLUtil.getRelativeUrl(url);
            final IndexInfoService countryIndexInfoService = getIndexInfoServiceForShard(searchShard);
            final Long fastIndexCount = countryIndexInfoService.extractLinksCount(new FastIndexCountRequest(hostName, path));
            if (fastIndexCount == null) {
                return false;
            }

            final boolean isFromFastRobot = fastIndexCount > 0;
            urlResultInfo.setSearchableInFastRobot(isFromFastRobot);

            try {
                tblAllAboutUrlResultsDao.updateFastIndexCount(hostDbHostInfo, urlResultInfo.getRequestId(), isFromFastRobot);
                return true;
            } catch (InternalException e) {
                log.error("update fast robot info failed", e);
                return false;
            }
        } catch (Exception e) {
            log.error("Exception in updateFastIndexCount" , e);
            return false;
        }
    }

    private boolean updateSearchableFromXmlSearch(final HostDbHostInfo hostDbHostInfo, final UrlResultInfo urlResultInfo, final YandexSearchShard searchShard) {
        if (urlResultInfo.getSearchableXml() != null) {
            return true;
        }

        try {
            final URL url = AbstractServantlet.prepareUrl(urlResultInfo.safeGetNormalizedUrl(), true);
            final String path = URLUtil.getRelativeUrl(url);

            final IndexInfoService countryIndexInfoService = getIndexInfoServiceForShard(searchShard);

            final Long count = countryIndexInfoService.extractLinksCount(new IndexCountRequest(url.getHost(), path));
            log.debug("xmlsearch count " + count + " for " + url + " shard " + searchShard);
            urlResultInfo.setSearchableXml(count != null && count > 0);
            tblAllAboutUrlResultsDao.updateSearchableXml(hostDbHostInfo, urlResultInfo.getRequestId(), count != null && count > 0);
            return true;
        } catch (UserException e) {
            log.error("UserException in updateSearchableFromXmlSearch", e);
            return false;
        } catch (InternalException e) {
            log.error("InternalException in updateSearchableFromXmlSearch", e);
            return false;
        } catch (Exception e) {
            log.error("InternalException in updateSearchableFromXmlSearch", e);
            return false;
        }
    }

    private IndexInfoService getIndexInfoServiceForShard(YandexSearchShard searchShard) {
        IndexInfoService countryIndexInfoService = indexInfoService;
        switch(searchShard) {
            case UA: countryIndexInfoService = uaIndexInfoService; break;
            case COM: countryIndexInfoService = comIndexInfoService; break;
            case COM_TR: countryIndexInfoService = trIndexInfoService; break;

            default: countryIndexInfoService = indexInfoService;
        }

        return countryIndexInfoService;
    }

    /**
     * Фабричный метод для сборки urlInfo на основе статуса
     *
     * @param status        вычисленный статус для заданного URL
     * @param dbUrlInfo     сырая информация об URL из базы данных
     * @return              объект с информацией об URL для верстки
     */
    private static UrlInfo createUrlInfo(UrlStatusEnum status, DbUrlInfo dbUrlInfo) {
        DbPartialUrlInfo prodUrlInfo = dbUrlInfo.getProdUrlInfo();
        DbPartialUrlInfo curUrlInfo = dbUrlInfo.getCurUrlInfo();
        DbPartialUrlInfo htarcUrlInfo = dbUrlInfo.getHtarcUrlInfo();
        YandexSearchShard searchShard = dbUrlInfo.getSearchShard() != null ? dbUrlInfo.getSearchShard() : YandexSearchShard.RU;
        return new UrlInfo(
                dbUrlInfo.getReportId(),
                dbUrlInfo.getUrl(),
                dbUrlInfo.getNormalizedUrl(),
                dbUrlInfo.getCreateDate(),
                status,
                curUrlInfo.getHttpCode(),
                curUrlInfo.getLastAccess(),
                prodUrlInfo.getHttpCode(),
                prodUrlInfo.getLastAccess(),
                htarcUrlInfo.getHttpCode(),
                htarcUrlInfo.getLastAccess(),
                dbUrlInfo.getLinksCount(),
                dbUrlInfo.getInternalLinksCount(),
                searchShard,
                prodUrlInfo.getSearchBaseMainUrl(),
                prodUrlInfo.getRedirTarget(),
                dbUrlInfo.getRelCanonicalTargetUrls(),
                dbUrlInfo.getFromFastRobot(),
                filterPenaltyStatus(dbUrlInfo.getHostInfoStatus(), isSearchable(dbUrlInfo)),
                dbUrlInfo.getPenaltyLastAccess());
    }

    private static HostInfoStatusEnum filterPenaltyStatus(final HostInfoStatusEnum penaltyStatus, final boolean isSearchable) {
        if (isSearchable) {
            // Если URL в поиске, то показываем статусы только про будущее
            return HostInfoStatusEnum.isWarnStatus(penaltyStatus) || HostInfoStatusEnum.ROBOTS_TXT.equals(penaltyStatus)
                    ? penaltyStatus
                    : null;
        } else {
            return penaltyStatus;
        }
    }

    private static boolean hasHttpCode(DbPartialUrlInfo urlInfo) {
        return urlInfo.getHttpCode() != null;
    }

    private static boolean hasData(DbPartialUrlInfo urlInfo) {
        return urlInfo.getLastAccess() != null;
    }

    private static boolean hasKiwiProdData(DbPartialUrlInfo urlInfo) {
        return urlInfo.getUploadRankValues() != null && urlInfo.getUploadRankValues();
    }

    private static boolean hasHtarcData(DbPartialUrlInfo urlInfo) {
        if (UrlIndexStatusEnum.NEW.equals(urlInfo.getIndexStatus())) {
            return false;
        }

        return urlInfo.getLastAccess() != null;
    }

    public static boolean isSearchable(DbUrlInfo dbUrlInfo) {
        boolean kiwiSearchable = dbUrlInfo.getProdUrlInfo().getSearchable();
        Boolean xmlSearchSearchable = dbUrlInfo.getSearchableFromXmlSearch();
        if (kiwiSearchable) {
            return xmlSearchSearchable != null ? xmlSearchSearchable : kiwiSearchable;
        } else {
            return kiwiSearchable;
        }
    }

    public static boolean isFake(DbPartialUrlInfo urlInfo) {
       return urlInfo.getIndexStatus() != null && UrlIndexStatusEnum.FAKE.equals(urlInfo.getIndexStatus());
    }

    public static boolean isRedirect(Integer code) {
        return code != null && code != 304 && code >= 300 && code < 400;
    }

    public static boolean isCode(Integer code, int desiredCode) {
        return code != null && code == desiredCode;
    }

    public static boolean isSR(UrlNoUploadReasonEnum reason) {
        return UrlNoUploadReasonEnum.SR.equals(reason);
    }

    public static boolean isSiteError(Integer code) {
        if (code == null) {
            return false;
        }

        Integer codeOrGroup = grouper.mapErrorToErrorGroup(code);
        return !SeverityEnum.OK.equals(errorOptions.getSeverityByCode(codeOrGroup));
    }

    public static boolean containsIgnoreCase(List<String> relCanonicalTargets, String url) {
        for (String relCanonicalTargetUrl : relCanonicalTargets) {
            if (relCanonicalTargetUrl.equalsIgnoreCase(url)) {
                return true;
            }
        }
        return false;
    }

    public static boolean equalsIgnoreSlash(final String url1, final String url2) {
        if (url1 == null || url2 == null) {
            return false;
        }

        if (url1.endsWith("/")) {
            return url1.equalsIgnoreCase(url2 + "/");
        } else {
            return (url1 + "/").equalsIgnoreCase(url2);
        }
    }

    public static String getDefaultOrderByDbField() {
        return FIELD_CREATE_DATE;
    }

    public Map<String, String> getOrderByCorrespondence() {
        Map<String, String> m = new LinkedHashMap<String, String>();
        m.put("url", FIELD_URL);
        m.put("date", FIELD_CREATE_DATE);
        return m;
    }

    public static class UrlInfoComparator implements Comparator<ShortUrlInfo> {
        private OrderByClause orderByClause;

        public UrlInfoComparator(OrderByClause orderByClause) {
            this.orderByClause = orderByClause;
        }

        @Override
        public int compare(ShortUrlInfo o1, ShortUrlInfo o2) {
            int res;
            if (FIELD_URL.equals(orderByClause.getColumnName())) {
                if (o1.getUrl().equals(o2.getUrl())) {
                    res = o1.getCreateDate().compareTo(o2.getCreateDate());
                } else {
                    res = o1.getUrl().compareTo(o2.getUrl());
                }
            } else if (FIELD_CREATE_DATE.equals(orderByClause.getColumnName())) {
                if (o1.getCreateDate().equals(o2.getCreateDate())) {
                    res = o1.getUrl().compareTo(o2.getUrl());
                } else {
                    res = o1.getCreateDate().compareTo(o2.getCreateDate());
                }
            } else {
                return o1.hashCode() - o2.hashCode();
            }

            if (!orderByClause.isAscending()) {
                res = -res;
            }

            return res;
        }
    }

    private static final ParameterizedRowMapper<DbUrlInfo> urlInfoMapper = new ParameterizedRowMapper<DbUrlInfo>() {
        @Override
        public DbUrlInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            DbPartialUrlInfo prodInfo = new DbPartialUrlInfo(
                    SqlUtil.safeGetTimestamp(rs, FIELD_PROD_LAST_ACCESS),
                    SqlUtil.getIntNullable(rs, FIELD_PROD_HTTP_CODE),
                    SqlUtil.getStringNullable(rs, FIELD_PROD_SEARCH_URL),
                    SqlUtil.getStringNullable(rs, FIELD_PROD_REDIR_TARGET),
                    SqlUtil.getStringNullable(rs, FIELD_PROD_REL_CANONICAL_TARGET),
                    UrlIndexStatusEnum.getByCode(SqlUtil.getIntNullable(rs, FIELD_PROD_STATUS)),
                    SqlUtil.getBooleanNullable(rs, FIELD_PROD_SEARCHABLE),
                    UrlNoUploadReasonEnum.getByCode(SqlUtil.getIntNullable(rs, FIELD_PROD_NO_UPLOAD_REASON)),
                    SqlUtil.getBooleanNullable(rs, FIELD_PROD_UPLOAD_RANK_VALUES),
                    SqlUtil.getLongNullable(rs, FIELD_PROD_LINKS),
                    SqlUtil.getLongNullable(rs, FIELD_PROD_INTERNAL_LINKS)
            );
            DbPartialUrlInfo curInfo = new DbPartialUrlInfo(
                    SqlUtil.safeGetTimestamp(rs, FIELD_CUR_LAST_ACCESS),
                    SqlUtil.getIntNullable(rs, FIELD_CUR_HTTP_CODE),
                    SqlUtil.getStringNullable(rs, FIELD_CUR_SEARCH_URL),
                    SqlUtil.getStringNullable(rs, FIELD_CUR_REDIR_TARGET),
                    SqlUtil.getStringNullable(rs, FIELD_CUR_REL_CANONICAL_TARGET),
                    UrlIndexStatusEnum.getByCode(SqlUtil.getIntNullable(rs, FIELD_CUR_STATUS)),
                    SqlUtil.getBooleanNullable(rs, FIELD_CUR_SEARCHABLE),
                    UrlNoUploadReasonEnum.getByCode(SqlUtil.getIntNullable(rs, FIELD_CUR_NO_UPLOAD_REASON)),
                    SqlUtil.getBooleanNullable(rs, FIELD_CUR_UPLOAD_RANK_VALUES),
                    SqlUtil.getLongNullable(rs, FIELD_CUR_LINKS),
                    SqlUtil.getLongNullable(rs, FIELD_CUR_INTERNAL_LINKS)
            );
            DbPartialUrlInfo htarcInfo = new DbPartialUrlInfo(
                    SqlUtil.safeGetTimestamp(rs, FIELD_HTARC_LAST_ACCESS),
                    SqlUtil.getIntNullable(rs, FIELD_HTARC_HTTP_CODE),
                    null,
                    null,
                    null,
                    UrlIndexStatusEnum.getByCode(SqlUtil.getIntNullable(rs, FIELD_HTARC_STATUS)),
                    null,
                    null,
                    null,
                    null,
                    null
            );
            Integer hostInfoStatus = SqlUtil.getIntNullable(rs, FIELD_HOST_INFO_STATUS);
            Date penaltyLastAccess = SqlUtil.safeGetTimestamp(rs, FIELD_PENALTY_LAST_ACCESS);
            Date searchableXmlTime = SqlUtil.safeGetTimestamp(rs, FIELD_SEARCHABLE_XML_TIME);
            Boolean searchableXml = SqlUtil.getBooleanNullable(rs, FIELD_SEARCHABLE_XML);

            UrlInfoStatusEnum status = UrlInfoStatusEnum.getStatusByFlags(
                    rs.getBoolean(FIELD_IS_ERROR), searchableXmlTime, penaltyLastAccess);

            Integer shard = SqlUtil.getIntNullable(rs, FIELD_SHARD_ID);
            YandexSearchShard searchShard = null;
            if (shard != null) {
                searchShard = YandexSearchShard.R.fromValueOrNull(shard);
            }

            return new DbUrlInfo(
                    rs.getLong(FIELD_NUM),
                    rs.getString(FIELD_URL),
                    SqlUtil.getStringNullable(rs, FIELD_NORMALIZED_URL),
                    status,
                    SqlUtil.safeGetTimestamp(rs, FIELD_CREATE_DATE),
                    SqlUtil.getBooleanNullable(rs, FIELD_IS_FROM_FAST_ROBOT),
                    hostInfoStatus != null ? HostInfoStatusEnum.R.fromValueOrNull(hostInfoStatus) : null,
                    penaltyLastAccess,
                    rs.getLong(FIELD_CANONICAL_URLS_COUNT),
                    prodInfo,
                    curInfo,
                    htarcInfo,
                    searchShard,
                    searchableXml,
                    SqlUtil.getLongNullable(rs, FIELD_LINKS_COUNT),
                    SqlUtil.getLongNullable(rs, FIELD_INTERNAL_LINKS_COUNT));
        }
    };

    private static final UrlErrorGrouper grouper = UrlErrorGrouper.getInstance();
    private static final UserErrorOptions errorOptions = new UserErrorOptions(0L, true);

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setIndexInfoService(IndexInfoService indexInfoService) {
        this.indexInfoService = indexInfoService;
    }

    @Required
    public void setReportCountConstraint(Integer reportCountConstraint) {
        this.reportCountConstraint = reportCountConstraint;
    }

    @Required
    public void setTblAllAboutUrlResultsDao(TblAllAboutUrlResultsDao tblAllAboutUrlResultsDao) {
        this.tblAllAboutUrlResultsDao = tblAllAboutUrlResultsDao;
    }

    @Required
    public void setTblAllAboutUrlQueueDao(TblAllAboutUrlQueueDao tblAllAboutUrlQueueDao) {
        this.tblAllAboutUrlQueueDao = tblAllAboutUrlQueueDao;
    }

    @Required
    public void setTblAllAboutUrlProdDao(TblAllAboutUrlProdDao tblAllAboutUrlProdDao) {
        this.tblAllAboutUrlProdDao = tblAllAboutUrlProdDao;
    }

    @Required
    public void setTblAllAboutUrlCurDao(TblAllAboutUrlCurDao tblAllAboutUrlCurDao) {
        this.tblAllAboutUrlCurDao = tblAllAboutUrlCurDao;
    }

    @Required
    public void setTblAllAboutUrlHtarcDao(TblAllAboutUrlHtarcDao tblAllAboutUrlHtarcDao) {
        this.tblAllAboutUrlHtarcDao = tblAllAboutUrlHtarcDao;
    }

    @Required
    public void setTblPenaltyInfoDao(TblPenaltyInfoDao tblPenaltyInfoDao) {
        this.tblPenaltyInfoDao = tblPenaltyInfoDao;
    }

    @Required
    public void setTblHostInfoDao(TblHostInfoDao tblHostInfoDao) {
        this.tblHostInfoDao = tblHostInfoDao;
    }

    @Required
    public void setTblHostsHostDao(TblHostsHostDao tblHostsHostDao) {
        this.tblHostsHostDao = tblHostsHostDao;
    }

    @Required
    public void setTblAllAboutUrlCanonicalDao(TblAllAboutUrlCanonicalDao tblAllAboutUrlCanonicalDao) {
        this.tblAllAboutUrlCanonicalDao = tblAllAboutUrlCanonicalDao;
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }

    @Required
    public void setUaIndexInfoService(IndexInfoService uaIndexInfoService) {
        this.uaIndexInfoService = uaIndexInfoService;
    }

    @Required
    public void setComIndexInfoService(IndexInfoService comIndexInfoService) {
        this.comIndexInfoService = comIndexInfoService;
    }

    @Required
    public void setTrIndexInfoService(IndexInfoService trIndexInfoService) {
        this.trIndexInfoService = trIndexInfoService;
    }

    @Required
    public void setTblUrlTreesDao(TblUrlTreesDao tblUrlTreesDao) {
        this.tblUrlTreesDao = tblUrlTreesDao;
    }
}
