package ru.yandex.wmconsole.service;

import java.util.ArrayList;
import java.util.List;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.wmconsole.data.BlockedUserInfo;
import ru.yandex.wmconsole.data.UserBlockReason;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.service.dao.TblBlockedUsersDao;
import ru.yandex.wmtools.common.data.HistoryActionEnum;
import ru.yandex.wmtools.common.data.HistoryObjectTypeEnum;
import ru.yandex.wmtools.common.data.info.WMUserInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.service.HistoryService;
import ru.yandex.wmtools.common.service.ViewerUserIdService;
import ru.yandex.wmtools.common.util.ServiceTransactionCallback;
import ru.yandex.wmtools.common.util.ServiceTransactionCallbackWithoutResult;

/**
 * @author senin
 */
public class BlockedUsersService extends AbstractDbService {
    private IWMCSupportService supportService;
    private HistoryService historyService;
    private IWMCUserInfoService userInfoService;
    private ViewerUserIdService viewerUserIdService;

    private TblBlockedUsersDao tblBlockedUsersDao;

    public List<Pair<WMUserInfo, BlockedUserInfo>> getBlockedUserInfoList(Pager pager) throws InternalException, UserException {
        List<BlockedUserInfo> blockedUsers = tblBlockedUsersDao.getBlockedUsersList(pager);

        List<Pair<WMUserInfo, BlockedUserInfo>> result = new ArrayList<Pair<WMUserInfo, BlockedUserInfo>>();
        for (BlockedUserInfo blockedUser: blockedUsers) {
            WMUserInfo userInfo = userInfoService.getUserInfo(blockedUser.getUserId());
            result.add(Pair.of(userInfo, blockedUser));
        }
        return result;
    }

    public BlockedUserInfo getBlockInfo(long userId) throws InternalException {
        return tblBlockedUsersDao.getBlockInfo(userId);
    }

    public void addBlockedUser(final Long realUserId, final long blockedId, final UserBlockReason userBlockReason)
            throws UserException, InternalException
    {
        viewerUserIdService.assureUserIsInUsersList(blockedId);
        if (supportService.isBlockedUser(blockedId)) {
            throw new UserException(UserProblem.USER_IS_ALREADY_IN_LIST,
                    "User " + blockedId + " is already in list of blocked!");
        }

        ServiceTransactionCallbackWithoutResult updateOperation = new ServiceTransactionCallbackWithoutResult() {
            @Override
            public void doInTransactionWithoutResult(TransactionStatus transactionStatus) throws InternalException {
                tblBlockedUsersDao.addBlockedUser(getJdbcTemplate(WMCPartition.nullPartition()), blockedId,
                        userBlockReason, realUserId);
                historyService.addEvents(realUserId, realUserId, HistoryActionEnum.ADD,
                        HistoryObjectTypeEnum.BLOCKED_USER, blockedId);
            }
        };
        getServiceTransactionTemplate(WMCPartition.nullPartition()).executeInService(updateOperation);
    }

    public int removeBlockedUsers(final Long realUserId, final Long... blockedIds) throws UserException, InternalException {
        return (Integer) getServiceTransactionTemplate(WMCPartition.nullPartition()).executeInService(new ServiceTransactionCallback() {
            @Override
            public Object doInTransaction(TransactionStatus transactionStatus) throws InternalException {
                int res = tblBlockedUsersDao.removeBlockedUser(getJdbcTemplate(WMCPartition.nullPartition()), blockedIds);
                historyService.addEvents(realUserId, realUserId, HistoryActionEnum.REMOVE, HistoryObjectTypeEnum.BLOCKED_USER, blockedIds);
                return res;
            }
        });
    }

    @Required
    public void setSupportService(IWMCSupportService supportService) {
        this.supportService = supportService;
    }

    @Required
    public void setHistoryService(HistoryService historyService) {
        this.historyService = historyService;
    }

    @Required
    public void setUserInfoService(IWMCUserInfoService userInfoService) {
        this.userInfoService = userInfoService;
    }

    @Required
    public void setViewerUserIdService(ViewerUserIdService viewerUserIdService) {
        this.viewerUserIdService = viewerUserIdService;
    }

    @Required
    public void setTblBlockedUsersDao(TblBlockedUsersDao tblBlockedUsersDao) {
        this.tblBlockedUsersDao = tblBlockedUsersDao;
    }
}
