package ru.yandex.wmconsole.service;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;

import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmtools.common.Constants;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.IService;

public class CheckHostNameService implements IService {
    public static enum HostNameStatus {NEEDS_MODERATION, GOOD, BAD}

    private Collection<String> geoList;
    private Collection<String> ibmList;
    private Collection<String> townList;
    private Collection<String> wwwList = Arrays.asList("www", "w3", "ftp");

    private DisplayNameModerationService displayNameModerationService;

    private Resource geoListResource;
    private Resource ibmListResource;
    private Resource townListResource;

    public void init() throws IOException {
        InputStream geoListIS = geoListResource.getInputStream();
        List<String> tmpGeoList;
        try {
            tmpGeoList = new ArrayList<String>(IOUtils.readLines(geoListIS));
        } finally {
            IOUtils.closeQuietly(geoListIS);
        }

        InputStream ibmListIS = ibmListResource.getInputStream();
        List<String> tmpIbmList;
        try {
            tmpIbmList = new ArrayList<String>(IOUtils.readLines(ibmListIS));
        } finally {
            IOUtils.closeQuietly(ibmListIS);
        }

        InputStream townListIS = townListResource.getInputStream();
        List<String> tmpTownList;
        try {
            tmpTownList = new ArrayList<String>(IOUtils.readLines(townListIS));
        } finally {
            IOUtils.closeQuietly(townListIS);
        }

        geoList = tmpGeoList;
        ibmList = tmpIbmList;
        townList = tmpTownList;
    }

    public HostNameStatus checkDisplayName(BriefHostInfo briefHostInfo, String newName)
            throws UserException, InternalException {
        if (!briefHostInfo.getName().equalsIgnoreCase(newName)) {
            throw new UserException(WMCUserProblem.NOT_MATCHING_DISPLAY_NAME,
                    "Specified hostname " + newName + " isn't equal ignore case to " + briefHostInfo.getName());
        }
        if (displayNameModerationService.getDisplayOrCurrentName(briefHostInfo.getId()).equals(newName)) {
            throw new UserException(WMCUserProblem.SAME_DISPLAY_NAME,
                    "Specified hostname " + newName + " doesn't differ in case from the previous one " + briefHostInfo.getName());
        }

        HostNameStatus checkDisplayNameStatus = checkDisplayName(newName);

        if (checkDisplayNameStatus == CheckHostNameService.HostNameStatus.BAD) {
            throw new UserException(WMCUserProblem.INVALID_DISPLAY_NAME,
                    "Specified hostname " + newName + " hasn't passed name check");
        }

        return checkDisplayNameStatus;
    }

    public HostNameStatus checkDisplayName(String displayName) {
            boolean semi = false;

            String[] toks = displayName.split("\\.");
            if (toks.length < 2) {
                return HostNameStatus.BAD;
            }

            String tok0 = toks[0]; // Ex: www
            String tok1 = toks[toks.length - 1]; // Ex: ru
            String tok2 = toks[toks.length - 2] + "." + toks[toks.length - 1]; // Ex: yandex.ru

            if  (!tok1.equals(tok1.toLowerCase())) {
                return HostNameStatus.BAD;
            }


            if (geoList.contains(tok2.toLowerCase())) {
                if (!tok2.equals(tok2.toLowerCase())) {
                    return HostNameStatus.BAD;
                }
            }

            for (int i=0; i < toks.length; i++) {
                String tok = toks[i];
                if (tok.equals(tok.toLowerCase())) {
                    continue;
                }
                if (ibmList.contains(tok)) {
                    continue;
                }
                if (townList.contains(tok)) {
                    continue;
                }

                int[] x  = getUpperCount(tok);
                if (x[1] > 5) {
                    return HostNameStatus.BAD;
                }

                if (x[0] == 2 && twoWordsString(tok) && !tok.contains("-")) {
                    continue;
                }

                semi = true;
            }

            if (wwwList.contains(tok0.toLowerCase()) && ! tok0.equals(tok0.toLowerCase())) {
                return HostNameStatus.BAD;
            }

            if (semi) {
                return HostNameStatus.NEEDS_MODERATION;
            } else {
                return HostNameStatus.GOOD;
            }
    }

    private int[] getUpperCount(String tok) {
        String ltok = tok.toLowerCase();
        int k = 0;
        int m = 0;
        int mm = 0;
        for (int i = 0; i < tok.length(); i++) {
            if (tok.charAt(i) != ltok.charAt(i)) {
                k++;
                m++;
            } else {
                mm = Math.max(mm, m);
                m = 0;
            }
        }
        mm = Math.max(mm, m);
        return new int[] {k, mm};
    }

    private boolean twoWordsString(String tok) {
        String ltok = tok.toLowerCase();
        if (tok.charAt(0) != ltok.charAt(0)) {
            for (int i = 3; i < tok.length() - 2; i++) {
                if (tok.charAt(i) != ltok.charAt(i)) {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean isIDN(URL hostName) {
        return hostName.getHost().startsWith(Constants.ACE_PREFIX) ||
                hostName.getHost().contains("." + Constants.ACE_PREFIX);
    }

    @Required
    public void setDisplayNameModerationService(DisplayNameModerationService displayNameModerationService) {
        this.displayNameModerationService = displayNameModerationService;
    }

    @Required
    public void setGeoListResource(Resource geoListResource) {
        this.geoListResource = geoListResource;
    }

    @Required
    public void setIbmListResource(Resource ibmListResource) {
        this.ibmListResource = ibmListResource;
    }

    @Required
    public void setTownListResource(Resource townListResource) {
        this.townListResource = townListResource;
    }
}
