package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.wmconsole.data.ServiceEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.ForeignPairInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.IServiceJdbcTemplate;

/**
 * SERVICE NUMBERS MUST BE GREATER THAN 1024 TO MAKE ENOUGH SPACE FOR STANDARD NOTIFICATIONS
 * @author Andrey Mima (amima@yandex-team.ru)
 */
public class ForeignServicesService extends AbstractDbService {
    private UsersHostsService usersHostsService;
    private ServiceOptionsService serviceOptionsService;

    private static final String SELECT_ACTIVE_FOREIGN_SERVICES_QUERY =
            "SELECT " +
                        "service_id " +
                    "FROM " +
                        "tbl_foreign_services " +
                    "WHERE " +
                        "active = 1";

    private static final String SELECT_ALL_FOREIGN_SERVICES_QUERY =
            "SELECT " +
                        "service_id " +
                    "FROM " +
                        "tbl_foreign_services";

    private static final String SELECT_IS_SERVICE_ACTIVE_QUERY =
            "SELECT " +
                        "active " +
                    "FROM " +
                        "tbl_foreign_services " +
                    "WHERE " +
                        "service_id = ?";

    private static final String SELECT_ACTIVE_FOREIGN_SERVICES_EXTENDED_QUERY =
            "SELECT " +
                        "service_id, id, tbl_dic_service.name as name " +
                    "FROM " +
                        "tbl_foreign_services LEFT JOIN tbl_dic_service ON plugin_id = id " +
                    "WHERE " +
                        "active = 1 AND public = 1 " +
                    "AND " +
                        "plugin_id IS NOT NULL";

    private static final ParameterizedRowMapper<Integer> FOREIGN_SERVICE_ID_MAPPER =
            new ParameterizedRowMapper<Integer>() {
                @Override
                public Integer mapRow(ResultSet rs, int rowNum) throws SQLException {
                    return rs.getInt("service_id");
                }
            };

    private static final ParameterizedRowMapper<Boolean> BOOLEAN_MAPPER =
            new ParameterizedRowMapper<Boolean>() {
                @Override
                public Boolean mapRow(ResultSet rs, int rowNum) throws SQLException {
                    return rs.getBoolean("active");
                }
            };

    private final ParameterizedRowMapper<ForeignPairInfo> FOREIGN_SERVICE_EXTENDED_MAPPER =
            new ParameterizedRowMapper<ForeignPairInfo>() {
                @Override
                public ForeignPairInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
                    return new ForeignPairInfo(rs.getInt("service_id"), rs.getInt("id"), rs.getString("name"));
                }
            };

    public List<Integer> getActiveForeignServices()
            throws InternalException {
        IServiceJdbcTemplate jdbcTemplate = getJdbcTemplate(WMCPartition.nullPartition());

        return jdbcTemplate.query(
                SELECT_ACTIVE_FOREIGN_SERVICES_QUERY,
                FOREIGN_SERVICE_ID_MAPPER
        );
    }

    public List<Integer> getAllForeignServices()
            throws InternalException {
        IServiceJdbcTemplate jdbcTemplate = getJdbcTemplate(WMCPartition.nullPartition());

        return jdbcTemplate.query(
                SELECT_ALL_FOREIGN_SERVICES_QUERY,
                FOREIGN_SERVICE_ID_MAPPER
        );
    }

    public boolean isActiveService(Integer serviceId)
            throws InternalException {
        IServiceJdbcTemplate jdbcTemplate = getJdbcTemplate(WMCPartition.nullPartition());

        List<Boolean> result = jdbcTemplate.query(
                SELECT_IS_SERVICE_ACTIVE_QUERY,
                BOOLEAN_MAPPER,
                serviceId
        );

        return result.size() > 0 && result.get(0);
    }

    public List<ForeignPairInfo> getActiveForeignServicesExtended(Long userId)
            throws InternalException {
        IServiceJdbcTemplate jdbcTemplate = getJdbcTemplate(WMCPartition.nullPartition());

        List<ForeignPairInfo> activeForeignServicesExtended = jdbcTemplate.query(
                SELECT_ACTIVE_FOREIGN_SERVICES_EXTENDED_QUERY,
                FOREIGN_SERVICE_EXTENDED_MAPPER
        );

        Set<ForeignPairInfo> result = new HashSet<ForeignPairInfo>();

        List<UsersHostsInfo> hostsInfos = usersHostsService.getUsersHostsInfoList(usersHostsService.getUserMainMirrorHostIds(userId));
        for (UsersHostsInfo info : hostsInfos) {
            if (info.getVerificationState().isVerified()) {
                List<ServiceEnum> hostEnabledServices =
                        serviceOptionsService.getEnabledServices(new BriefHostInfo(info.getHostId(), info.getHostName(), null));
                for (ServiceEnum hostEnabledService : hostEnabledServices) {
                    for (ForeignPairInfo pair : activeForeignServicesExtended) {
                        if (pair.getServiceId() == hostEnabledService.getValue()) {
                            result.add(pair);
                        }
                    }
                }
            }
        }


        return new ArrayList<ForeignPairInfo>(result);
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }

    @Required
    public void setServiceOptionsService(ServiceOptionsService serviceOptionsService) {
        this.serviceOptionsService = serviceOptionsService;
    }
}
