package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.wmconsole.data.HostDailyStateForCode;
import ru.yandex.wmconsole.data.SeverityEnum;
import ru.yandex.wmconsole.data.UserErrorOptions;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDailyStateInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.UrlErrorsWithCodeInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.util.UrlErrorGrouper;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.SqlUtil;
import ru.yandex.wmtools.common.util.TimeFilter;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 20.03.2007
 * Time: 16:59:17
 */
public class HostDailyStateService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(HostDailyStateService.class);

    private static final String FIELD_HOST_ID = "host_id";
    private static final String FIELD_STATE_DATE = "state_date";
    private static final String FIELD_ERRORS_COUNT = "errors_count";
    private static final String FIELD_CODE = "code";
    private static final String FIELD_TIME = "time";

    private static final String SELECT_DAILY_STATE_FOR_HOST_QUERY =
            "SELECT " +
                    "    host_id AS " + FIELD_HOST_ID + ", " +
                    "    unix_timestamp(state_date) AS " + FIELD_STATE_DATE + ", " +
                    "    code AS " + FIELD_CODE + ", " +
                    "    count AS " + FIELD_ERRORS_COUNT + " " +
                    "FROM " +
                    "    tbl_host_daily_state " +
                    "WHERE " +
                    "    host_id = ? " +
                    "ORDER BY " +
                    "    state_date asc ";

    private static final String SELECT_ERROR_COUNT_HISTORY_FOR_HOST_AND_SECTION_QUERY =
            "SELECT " +
                    "    host_id AS " + FIELD_HOST_ID + ", " +
                    "    unix_timestamp(time) AS " + FIELD_TIME + ", " +
                    "    code AS " + FIELD_CODE + ", " +
                    "    count AS " + FIELD_ERRORS_COUNT + " " +
                    "FROM " +
                    "    tbl_trend_code_error_trees_count " +
                    "WHERE " +
                    "    host_id = ? AND " +
                    "    node_id = ? AND " +
                    "    code IN (%s) " +
                    "    %s  " +
                    "ORDER BY " +
                    "    time asc ";

    private static final String SELECT_ERROR_COUNT_HISTORY_DISTINCT_DATES_QUERY =
            "SELECT " +
                    "    distinct unix_timestamp(time) AS " + FIELD_TIME + " " +
                    "FROM " +
                    "    tbl_trend_code_error_trees_count " +
                    "WHERE " +
                    "    host_id = ? " +
                    "    %s  " +
                    "ORDER BY " +
                    "    time asc ";


    ParameterizedRowMapper<HostDailyStateForCode> hostDailyStateForCodeMapper = new ParameterizedRowMapper<HostDailyStateForCode>() {
        @Override
        public HostDailyStateForCode mapRow(ResultSet rs, int rowNum) throws SQLException {
            int hostId = rs.getInt(FIELD_HOST_ID);
            Date stateDate = new Date(TimeUnit.SECONDS.toMillis(rs.getInt(FIELD_STATE_DATE)));
            int code = rs.getInt(FIELD_CODE);
            int count = rs.getInt(FIELD_ERRORS_COUNT);
            return new HostDailyStateForCode(hostId, stateDate, code, count);
        }
    };

    ParameterizedRowMapper<HostDailyStateForCode> errorCountForCodeAndNodeMapper = new ParameterizedRowMapper<HostDailyStateForCode>() {
        @Override
        public HostDailyStateForCode mapRow(ResultSet rs, int rowNum) throws SQLException {
            int hostId = rs.getInt(FIELD_HOST_ID);
            Date stateDate = new Date(TimeUnit.SECONDS.toMillis(rs.getInt(FIELD_TIME)));
            int code = rs.getInt(FIELD_CODE);
            int count = rs.getInt(FIELD_ERRORS_COUNT);
            return new HostDailyStateForCode(hostId, stateDate, code, count);
        }
    };

    private HostDbHostInfoService hostDbHostInfoService;
    private ErrorInfoService errorInfoService;

    public HostDailyStateInfo getCurrentState(BriefHostInfo hostInfo, UserErrorOptions userErrorOptions) throws InternalException {
        List<UrlErrorsWithCodeInfo> list = errorInfoService.getErrorGroupsList(hostInfo, userErrorOptions);

        int siteErrors = 0;
        int disallowedByUser = 0;
        int unsupportedByRobot = 0;
        for (UrlErrorsWithCodeInfo urlErrorInfo : list) {
            SeverityEnum severity = userErrorOptions.getSeverityByCode(urlErrorInfo.getCode());
            if (SeverityEnum.SITE_ERROR.equals(severity)) {
                siteErrors += urlErrorInfo.getCount();
            } else if (SeverityEnum.DISALLOWED_BY_USER.equals(severity)) {
                disallowedByUser += urlErrorInfo.getCount();
            } else if (SeverityEnum.UNSUPPORTED_BY_ROBOT.equals(severity)) {
                unsupportedByRobot += urlErrorInfo.getCount();
            } else if (SeverityEnum.OK.equals(severity)) {
                // nothing to do
            } else {
                log.error("Unknown severity type: " + severity);
            }
        }

        return new HostDailyStateInfo(new Date(), siteErrors, disallowedByUser, unsupportedByRobot);
    }

    public List<HostDailyStateInfo> getHostDailyState(Long userId, BriefHostInfo hostInfo) throws InternalException {
        UserErrorOptions userErrorOptions = errorInfoService.getUserErrorOptions(userId, hostInfo.getId());
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());

        List <HostDailyStateForCode> statesForCode = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_DAILY_STATE_FOR_HOST_QUERY, hostDailyStateForCodeMapper, hostDbHostInfo.getHostDbHostId());

        List<HostDailyStateInfo> result = new ArrayList<HostDailyStateInfo>();
        for (HostDailyStateForCode state : statesForCode) {
            Date newDate = state.getStateDate();
            if ((result.size() == 0) || (!result.get(result.size() - 1).getStateDate().equals(newDate))) {
                result.add(new HostDailyStateInfo(newDate, 0, 0, 0));
            }

            Integer code = UrlErrorGrouper.getInstance().mapErrorToErrorGroup(state.getCode());
            if (code != null) {
                SeverityEnum severity = userErrorOptions.getSeverityByCode(code);
                if (severity == SeverityEnum.SITE_ERROR) {
                    result.get(result.size() - 1).addSiteErrors(state.getCount());
                } else if (severity == SeverityEnum.DISALLOWED_BY_USER) {
                    result.get(result.size() - 1).addDisallowedByUser(state.getCount());
                } else if (severity == SeverityEnum.UNSUPPORTED_BY_ROBOT) {
                    result.get(result.size() - 1).addUnsupportedByRobot(state.getCount());
                } else if (severity == SeverityEnum.OK) {
                    // nothing to do
                } else {
                    throw new IllegalStateException("Severity not found: " + severity);
                }
            }
        }

        // adding current state
        result.add(getCurrentState(hostInfo, userErrorOptions));

        return result;
    }

    /**
     * Получить историю исключенных страниц для раздела сайта и для кода ошибки.
     *
     * NOTE! Не проверяем код ошибки по настройкам пользователя, потому что на страницу с графиком переходим
     * только для кодов, которые должны показываться.
     *
     * @param hostInfo      информация о хосте из пользовательской базы
     * @param nodeId        идентификатор узла в дереве url
     * @param code          код ошибки
     * @param timeFilter    фильтр по времени
     * @return              данные для графика исключенных страниц: дата -> число исключенных страниц
     * @throws InternalException при ошибках с базой данных
     */
    public NavigableMap<Date, Integer> getErrorHistoryForNodeAndCode(
            final BriefHostInfo hostInfo,
            final long nodeId,
            final int code,
            final TimeFilter timeFilter) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());

        final NavigableMap<Date, Integer> result = new TreeMap<Date, Integer>();

        getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                String.format(SELECT_ERROR_COUNT_HISTORY_DISTINCT_DATES_QUERY,
                        SqlUtil.getTimeFilterPart(timeFilter, FIELD_TIME)),
                new ParameterizedRowMapper<Object>() {
                    @Override
                    public Object mapRow(ResultSet rs, int rowNum) throws SQLException {
                        result.put(new Date(TimeUnit.SECONDS.toMillis(rs.getInt(FIELD_TIME))), 0);
                        return null;
                    }
                },
                hostDbHostInfo.getHostDbHostId()
        );

        List <HostDailyStateForCode> statesForCode = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                String.format(SELECT_ERROR_COUNT_HISTORY_FOR_HOST_AND_SECTION_QUERY,
                        SqlUtil.getCommaSeparatedList(UrlErrorGrouper.expandGroups(Collections.singletonList(code))),
                        SqlUtil.getTimeFilterPart(timeFilter, FIELD_TIME)),
                errorCountForCodeAndNodeMapper,
                hostDbHostInfo.getHostDbHostId(),
                nodeId);

        for (HostDailyStateForCode state : statesForCode) {
            Integer count = result.get(state.getStateDate());
            if (count == null) {
                result.put(state.getStateDate(), state.getCount());
            } else {
                result.put(state.getStateDate(), count + state.getCount());
            }
        }

        return result;
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setErrorInfoService(ErrorInfoService errorInfoService) {
        this.errorInfoService = errorInfoService;
    }
}
