package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.common.util.db.BooleanRowMapper;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.service.AbstractDbService;

/**
 * @author baton
 */
public class HostDbHostInfoService extends AbstractDbService {
    private static final String FIELD_HOST_ID = "host_id";
    private static final String FIELD_HOSTNAME = "hostname";
    private static final String LOCK_STRING_FORMAT = "%d_in_progress";

    private static final String SELECT_HOST_INFO_BY_HOST_NAME_IN_HOST_DB_QUERY =
            "SELECT " +
                    "    host_id AS " + FIELD_HOST_ID + ", " +
                    "    name AS " + FIELD_HOSTNAME + " " +
                    "FROM " +
                    "    tbl_hosts " +
                    "WHERE " +
                    "    name = ? ";

    private static final String SELECT_CHECK_LOCK_FOR_HOST_QUERY =
            "SELECT IS_FREE_LOCK(?) ";

    private Map<String, HostDbHostInfo> cache;
    private int maxCacheSize = 5000;

    private static final ParameterizedRowMapper<HostDbHostInfo> hostDbHostInfoMapper = new ParameterizedRowMapper<HostDbHostInfo>() {
        @Override
        public HostDbHostInfo mapRow(ResultSet resultSet, int rowNumber) throws SQLException {
            return new HostDbHostInfo(
                    resultSet.getLong(FIELD_HOST_ID),
                    resultSet.getString(FIELD_HOSTNAME)
            );
        }
    };

    public void init() {
        final int MAX_SIZE = maxCacheSize;
        cache = Collections.synchronizedMap(
                new LinkedHashMap<String, HostDbHostInfo>(maxCacheSize, 0.75f, true) {
                    @Override
                    protected boolean removeEldestEntry(Map.Entry<String, HostDbHostInfo> eldest) {
                        return size() >= MAX_SIZE;
                    }
                }
        );
    }

    public HostDbHostInfo getHostDbHostInfo(String hostname) throws InternalException {
        return getHostDbHostInfo(hostname, false);
    }

    public HostDbHostInfo getHostDbHostInfo(String hostname, boolean nullable) throws InternalException {
        HostDbHostInfo hostDbHostInfo = cache.get(hostname);
        if (hostDbHostInfo != null) {
            return hostDbHostInfo;
        }

        List<HostDbHostInfo> hosts = getJdbcTemplate(new WMCPartition(null, hostname, null)).query(
                SELECT_HOST_INFO_BY_HOST_NAME_IN_HOST_DB_QUERY, hostDbHostInfoMapper, hostname);

        if (hosts.isEmpty()) {
            if (nullable) {
                return null;
            }
            throw new InternalException(InternalProblem.ILLEGAL_ARGUMENT, "Host " + hostname + " is NOT in the WMC-base");
        }

        if (hosts.size() > 1) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "Too many hosts for hostname: " + hostname);
        }
        hostDbHostInfo = hosts.get(0);
        cache.put(hostname, hostDbHostInfo);
        return hostDbHostInfo;
    }

    public boolean checkHostInProgress(HostDbHostInfo info) throws InternalException {
        Boolean isFree = getJdbcTemplate(new WMCPartition(info, null)).queryForObject(
                SELECT_CHECK_LOCK_FOR_HOST_QUERY,
                new BooleanRowMapper(),
                String.format(LOCK_STRING_FORMAT, info.getHostDbHostId()));
        return isFree != null && !isFree;
    }

    @Required
    public void setMaxCacheSize(int maxCacheSize) {
        this.maxCacheSize = maxCacheSize;
    }
}
