package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.common.framework.pager.Pager;
import ru.yandex.common.util.collections.Cu;
import ru.yandex.webmaster.common.host.dao.TblHostsMainDao;
import ru.yandex.webmaster.common.host.dao.TblUsersHostsDao;
import ru.yandex.webmaster.common.urltree.YandexSearchShard;
import ru.yandex.webmaster3.SafeNewWebmasterHttpService;
import ru.yandex.wmconsole.data.DisplayNameModerationStateEnum;
import ru.yandex.wmconsole.data.FakeUser;
import ru.yandex.wmconsole.data.HostInfoStatusEnum;
import ru.yandex.wmconsole.data.UpdateStateEnum;
import ru.yandex.wmconsole.data.UserErrorOptions;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.WMCHistoryObjectTypeEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.info.HostInfo;
import ru.yandex.wmconsole.data.info.HostStatusInfo;
import ru.yandex.wmconsole.data.info.HostUnavailableInfo;
import ru.yandex.wmconsole.data.info.ShortHostInfo;
import ru.yandex.wmconsole.data.info.UserDbHostInfo;
import ru.yandex.wmconsole.data.info.UsersHostsCacheInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.service.dao.*;
import ru.yandex.wmconsole.util.UrlErrorGrouper;
import ru.yandex.wmconsole.verification.VerificationTypeEnum;
import ru.yandex.wmtools.common.data.HistoryActionEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.service.HistoryService;
import ru.yandex.wmtools.common.util.ServiceTransactionCallback;
import ru.yandex.wmtools.common.util.SqlUtil;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 20.03.2007
 * Time: 10:09:24
 */
public class HostInfoService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(HostInfoService.class);

    private static final String FIELD_HOSTNAME = "host_name";
    private static final String FIELD_URL_COUNT = "url_count";
    private static final String FIELD_VERIFICATION_STATE = "verified";
    private static final String FIELD_VERIFICATION_TYPE = "verification_type";
    private static final String FIELD_HOST_ID = "host_id";
    private static final String FIELD_DOCS = "docs";
    private static final String FIELD_UPDATE_STATE = "update_state";
    private static final String FIELD_LAST_ACCESS = "last_access";
    private static final String FIELD_UPDATED_ON = "updated_on";
    private static final String FIELD_STATE = "state";
    private static final String FIELD_INDEX_COUNT = "index_count";
    private static final String FIELD_PRECISE_INDEX_COUNT = "precise_index_count";
    private static final String FIELD_INDEX_COUNT_TIME = "index_count_time";
    private static final String FIELD_BANNED_COUNT = "banned_count";
    private static final String FIELD_INTERNAL_LINKS_COUNT = "internal_links";
    private static final String FIELD_INTERNAL_LINKS_TIME = "internal_links_time";
    private static final String FIELD_LINKS_COUNT = "links";
    private static final String FIELD_LINKS_TIME = "links_time";
    private static final String FIELD_SITEMAP_COUNT = "sitemap_count";
    private static final String FIELD_VIRUSED_STATE = "state";
    private static final String FIELD_HOST_INFO_STATUS = "status";
    private static final String FIELD_HOST_INFO_STATUS_DATE = "status_date";
    private static final String FIELD_HOST_INFO_STATUS_HTARC = "status_htarc";
    private static final String FIELD_HOST_INFO_PENALTY_HTARC = "penalty_htarc";
    private static final String FIELD_HOST_INFO_STATUS_YA = "status_ya";
    private static final String FIELD_HOST_INFO_PENALTY_YA = "penalty_ya";
    private static final String FIELD_SPIDER_IP = "spider_ip";
    private static final String FIELD_TCY = "tcy";
    private static final String FIELD_URLS_TREND = "urls_trend";
    private static final String FIELD_INDEX_COUNT_TREND = "index_count_trend";
    private static final String FIELD_LINKS_COUNT_TREND = "links_count_trend";
    private static final String FIELD_TCY_TREND = "tcy_trend";
    private static final String FIELD_LAST_ACCESS_YA = "last_access_ya";
    private static final String FIELD_LAST_ACCESS_HTARC = "last_access_htarc";
    private static final String FIELD_LAST_LOADER_TIME = "last_loader_time";
    private static final String FIELD_HTARC_IN_INDEX = "htarc_in_index";
    private static final String FIELD_PREV_IN_INDEX = "prev_in_index";

    private static final String SELECT_GET_HOST_NAME_BY_HOST_ID_QUERY =
            "SELECT " +
                    "    name " +
                    "FROM " +
                    "    tbl_hosts " +
                    "WHERE " +
                    "    host_id = ? ";

    private static final String SELECT_GET_HOST_ID_BY_HOST_NAME_QUERY =
            "SELECT " +
                    "    host_id " +
                    "FROM " +
                    "    tbl_hosts " +
                    "WHERE " +
                    "    name = ? ";

    private static final String SELECT_HOST_INFO_QUERY =
            "SELECT " +
                    "    tc.tcy AS " + FIELD_TCY + ", " +
                    "    COALESCE(pi.last_access_ya, rdbi.last_access) AS " + FIELD_LAST_ACCESS + ", " +             // для показа даты последнего посещения
                    "    pi.last_access_htarc AS " + FIELD_LAST_ACCESS_HTARC + ", " + // честная дата с htarc для сообщений о недоступности
                    "    rdbi.updated_on AS " + FIELD_UPDATED_ON + ", " +
                    "    rdbi.docs AS " + FIELD_DOCS + ", " +
                    "    rdbi.state AS " + FIELD_UPDATE_STATE + ", " +
                    "    COALESCE(ut.index_count, rdbi.index_count) AS " + FIELD_PRECISE_INDEX_COUNT + ", " +        // стараемся показать число страниц для поисковой зоны, если нет, то показыаем дефолтное значение
                    "    rdbi.banned_count AS " + FIELD_BANNED_COUNT + ", " +
                    "    rdbi.urls AS " + FIELD_URL_COUNT + ", " +
                    "    rdbi.tcy_trend AS " + FIELD_TCY_TREND + ", " +
                    "    rdbi.urls_trend AS " + FIELD_URLS_TREND + ", " +
                    "    rdbi.index_count_trend AS " + FIELD_INDEX_COUNT_TREND + ", " +
                    "    rdbi.links_count_trend AS " + FIELD_LINKS_COUNT_TREND + ", " +
                    "    rdbi.last_loader_time AS " + FIELD_LAST_LOADER_TIME + ", " +
                    "    rdbi.htarc_in_index AS " + FIELD_HTARC_IN_INDEX + ", " +
                    "    rdbi.prev_in_index AS " + FIELD_PREV_IN_INDEX + ", " +
                    "    si.spider_ip AS " + FIELD_SPIDER_IP + ", " +
                    "    lc.index_count AS " + FIELD_INDEX_COUNT + ", " +
                    "    lc.index_count_time AS " + FIELD_INDEX_COUNT_TIME + ", " +
                    "    lc.internal_links_count AS " + FIELD_INTERNAL_LINKS_COUNT + ", " +
                    "    lc.internal_links_time AS " + FIELD_INTERNAL_LINKS_TIME + ", " +
                    "    lc.links_count AS " + FIELD_LINKS_COUNT + ", " +
                    "    lc.links_time AS " + FIELD_LINKS_TIME + ", " +
                    "    ih.state AS " + FIELD_VIRUSED_STATE + ", " +
                    "    (SELECT COUNT(*) FROM tbl_sitemaps WHERE host_id=h.host_id) AS " + FIELD_SITEMAP_COUNT + ", " +
                    "    hi.status AS " + FIELD_HOST_INFO_STATUS + ", " +
                    "    hi.status_date AS " + FIELD_HOST_INFO_STATUS_DATE + "," +
                    "    pi.status_htarc AS " + FIELD_HOST_INFO_STATUS_HTARC + ", " +
                    "    pi.penalty_htarc AS " + FIELD_HOST_INFO_PENALTY_HTARC + ", " +
                    "    pi.status_ya AS " + FIELD_HOST_INFO_STATUS_YA + ", " +
                    "    pi.penalty_ya AS " + FIELD_HOST_INFO_PENALTY_YA + " " +
                    "FROM ((((((" +
                    "    tbl_hosts h " +
                    "LEFT JOIN " +
                    "    tbl_tcy_cache tc ON (h.host_id = tc.host_id)) " +
                    "LEFT JOIN " +
                    "    tbl_robotdb_info rdbi ON (h.host_id = rdbi.host_id)) " +
                    "LEFT JOIN " +
                    "    tbl_url_trees ut ON (h.host_id = ut.host_id)" +
                    "LEFT JOIN " +
                    "    tbl_links_cache lc ON (h.host_id = lc.host_id)) " +
                    "LEFT JOIN " +
                    "    tbl_infected_hosts ih ON (h.host_id = ih.host_id)) " +
                    "LEFT JOIN " +
                    "    tbl_host_info hi ON (h.host_id = hi.host_id)) " +
                    "LEFT JOIN " +
                    "    tbl_penalty_info pi ON (h.host_id = pi.host_id)) " +
                    "LEFT JOIN " +
                    "    ( " +
                    "       SELECT " +
                    "           spider_ip, " +
                    "           host_id " +
                    "       FROM " +
                    "           tbl_spider_ips tmp " +
                    "       NATURAL JOIN " +
                    "           tbl_hosts h1 " +
                    "       WHERE " +
                    "           name = ? " +
                    "       ORDER BY " +
                    "           date DESC, " +
                    "           spider_ip DESC " +
                    "       LIMIT 1 " +
                    "    ) si " +
                    "ON (h.host_id = si.host_id )" +
                    "WHERE " +
                    "    h.name = ? " +
                    "AND " +
                    "   (ut.shard_id = ? OR ut.shard_id IS NULL) " +
                    "AND " +
                    "   ut.parent_id IS NULL";

    private static final String UPDATE_UPDATED_ON_QUERY =
            "SELECT " +
                    "    rdbi.updated_on AS " + FIELD_UPDATED_ON + " " +
                    "FROM (" +
                    "    tbl_hosts h " +
                    "LEFT JOIN " +
                    "    tbl_robotdb_info rdbi ON (h.host_id = rdbi.host_id)) " +
                    "WHERE " +
                    "    h.name = ? ";

    private static final String SELECT_SHORT_HOST_LIST_QUERY =
            "SELECT " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOSTNAME + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + " " +
                    "FROM (" +
                    "    tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (uh.host_id = h.host_id)) " +
                    "WHERE " +
                    "    uh.user_id = ?";

    private static final String SELECT_USER_DB_HOST_INFO_QUERY =
            "SELECT " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOSTNAME + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + " " +
                    "FROM " +
                    "    tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (uh.host_id = h.host_id) " +
                    "WHERE " +
                    "    uh.user_id = ? " +
                    "AND " +
                    "    uh.host_id = ?";

    private static final String SELECT_NOT_VERIFIED_SHORT_HOST_LIST_QUERY =
            "SELECT " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOSTNAME + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + " " +
                    "FROM (" +
                    "    tbl_users_hosts uh " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (uh.host_id = h.host_id)) " +
                    "WHERE " +
                    "    uh.user_id = ? " +
                    "AND " +
                    "    uh.state != 1 " +
                    " %1$s " +
                    " %2$s ";

    private static final String SELECT_NOT_VERIFIED_SHORT_HOST_LIST_COUNT_QUERY =
            "SELECT " +
                    "    count(*) " +
                    "FROM " +
                    "    tbl_users_hosts uh " +
                    "WHERE " +
                    "    uh.user_id = ? " +
                    "AND " +
                    "    uh.state != " + VerificationStateEnum.VERIFIED.getValue();

    private static final String SELECT_HOST_ERRORS_FOR_USER_QUERY =
            "SELECT " +
                    "    sum(cet.count) " +
                    " FROM " +
                    "    tbl_code_agr_error_trees cet " +
                    " JOIN " +
                    "    tbl_url_trees ut " +
                    " ON (cet.node_id = ut.id) " +
                    " WHERE " +
                    "    cet.host_id = ? " +
                    "    AND ut.parent_id IS NULL " +
                    "    AND ut.shard_id = " + YandexSearchShard.RU.value() +
                    "    AND code IN (%1$s) ";

    private static final String SELECT_ALL_HOSTS_PENALTY_QUERY =
            "SELECT " +
                    "   pi.host_id AS " + FIELD_HOST_ID + ", " +
                    "   h.name AS " + FIELD_HOSTNAME + ", " +
                    "   pi.penalty_htarc AS " + FIELD_HOST_INFO_PENALTY_HTARC + ", " +
                    "   pi.status_htarc AS " + FIELD_HOST_INFO_STATUS_HTARC + ", " +
                    "   pi.penalty_ya AS " + FIELD_HOST_INFO_PENALTY_YA +", " +
                    "   pi.status_ya AS " + FIELD_HOST_INFO_STATUS_YA + ", " +
                    "   pi.last_access_ya AS " + FIELD_LAST_ACCESS_YA + ", " +
                    "   pi.last_access_htarc AS " + FIELD_LAST_ACCESS_HTARC + ", " +
                    "   rdbi.htarc_in_index AS " + FIELD_HTARC_IN_INDEX + ", " +
                    "   rdbi.prev_in_index AS " + FIELD_PREV_IN_INDEX + " " +
                    "FROM " +
                    "   tbl_penalty_info pi " +
                    "LEFT JOIN " +
                    "   tbl_robotdb_info rdbi ON (pi.host_id = rdbi.host_id) " +
                    "JOIN "+
                    "   tbl_hosts h ON (pi.host_id = h.host_id) " +
                    "WHERE " +
                    "   ((pi.penalty_htarc IS NOT NULL AND pi.status_htarc IS NOT NULL) OR " +     // пенальти есть для htarc
                    "   (pi.penalty_ya IS NOT NULL AND pi.status_ya IS NOT NULL)) AND " +          // пенальти есть для ya
                    "   pi.last_access_htarc IS NOT NULL AND pi.last_access_ya IS NOT NULL ";      // загружены данные из htarc и из ya

    private static final String SELECT_SPIDER_IP_QUERY =
            "SELECT " +
                    "   spider_ip " + "" +
                    "FROM " +
                    "   tbl_spider_ips " +
                    "WHERE "+
                    "   host_id = ? " +
                    "ORDER BY "+
                    "   date DESC, " +
                    "   spider_ip DESC " +
                    "LIMIT 1";

    private static final String SELECT_HOST_INFO_STATUS_QUERY =
            "SELECT " +
                    "   hi.status AS " + FIELD_HOST_INFO_STATUS + ", " +
                    "   pi.penalty_htarc AS " + FIELD_HOST_INFO_PENALTY_HTARC + ", " +
                    "   pi.status_htarc AS " + FIELD_HOST_INFO_STATUS_HTARC + ", " +
                    "   pi.penalty_ya AS " + FIELD_HOST_INFO_PENALTY_YA +", " +
                    "   pi.status_ya AS " + FIELD_HOST_INFO_STATUS_YA + ", " +
                    "   rdbi.htarc_in_index AS " + FIELD_HTARC_IN_INDEX + ", " +
                    "   rdbi.prev_in_index AS " + FIELD_PREV_IN_INDEX + " " +
                    "FROM " +
                    "   tbl_hosts h " +
                    "LEFT JOIN " +
                    "   tbl_robotdb_info rdbi ON h.host_id = rdbi.host_id " +
                    "LEFT JOIN " +
                    "   tbl_host_info hi ON h.host_id = hi.host_id " +
                    "LEFT JOIN " +
                    "   tbl_penalty_info pi ON h.host_id = pi.host_id " +
                    "WHERE " +
                    "   h.host_id = ?";

    private LinksCacheService linksCacheService;
    private IndexHistoryService indexHistoryService;
    private XmlLimitsService xmlLimitsService;
    private HashSet<Long> hostWithDataLoaded = new HashSet<Long>();
    private HostDbHostInfoService hostDbHostInfoService;
    private ConsistentMainMirrorService consistentMainMirrorService;
    private HistoryService historyService;
    private ErrorInfoService errorInfoService;
    private SafeNewWebmasterHttpService safeNewWebmasterHttpService;
    private TblUsersHostsCacheDao tblUsersHostsCacheDao;
    private TblUsersDao tblUsersDao;
    private TblHostsMainDao tblHostsMainDao;
    private TblRobotdbInfoDao tblRobotdbInfoDao;
    private TblUsersHostsDao tblUsersHostsDao;
    private TblDisplayNameModerationDao tblDisplayNameModerationDao;
    private TblUrlTreesDao tblUrlTreesDao;

    private String getStringIp(Long ip) {
        if (ip == null) {
            return "";
        }
        String stringIp = "";
        for (int i = 0; i < 4; i++) {
            stringIp = (ip % 256) + ((i > 0) ? "." : "") + stringIp;
            ip /= 256;
        }
        return stringIp;
    }

    /**
     * Returns full host information for given host by the eyes of given user.
     *
     * @param userId   User, requesting host informtaion.
     * @param hostInfo Host, for which to return informtaion.
     * @return Returns full information for given host.
     * @throws ClientException   Thrown if client has provided wrong data. See UserException.getProblem() for details.
     * @throws InternalException Thrown if we have some internal problem. See InternalException.getProblem() for details.
     */
    public HostInfo getHostInfo(long userId, BriefHostInfo hostInfo) throws ClientException, InternalException {
        final List<ShortHostInfo> shortHostInfos = getShortHostInfo(userId, hostInfo);
        if (shortHostInfos.size() != 1) {
            throw new ClientException(ClientProblem.HOST_NOT_OWNED_BY_USER, "Host is not added to the users host list");
        }

        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        UsersHostsCacheInfo cacheInfo = checkCacheAndGetErrorsCount(userId, hostInfo, hostDbHostInfo);

        ShortHostInfo shortHostInfo = shortHostInfos.get(0);
        UserDbHostInfo userDbHostInfo = new UserDbHostInfo(
                shortHostInfo.getId(),
                shortHostInfo.getName(),
                shortHostInfo.getVerificationState(),
                shortHostInfo.getVerificationType(),
                cacheInfo.getErrorsCount(),
                cacheInfo.getErrorsCountTrend(),
                hostInfo.getMainMirrorId() == null);

        YandexSearchShard shard = tblUrlTreesDao.getOptimumShardId(hostDbHostInfo);
        final List<HostInfo> resultList = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_HOST_INFO_QUERY,
                new HostInfoRowMapper(hostDbHostInfo, userDbHostInfo, true),
                hostInfo.getName(),
                hostInfo.getName(),
                shard.value());

        if (resultList.isEmpty()) {
            return new HostInfo(userDbHostInfo);
        }

        final HostInfo result = Cu.first(resultList);

        updateXmlSearchValues(hostDbHostInfo, result, shard);

        return result;
    }

    public void updateXmlSearchValues(final HostDbHostInfo hostDbHostInfo, HostInfo hostInfo, YandexSearchShard searchShard) {

        if (hostInfo.getIndexCount() == null) {
            // Обновляем цифру из кеша xml-поиска, если точных данных нет и данные в кеше устарели
            try {
                Long indexCount = linksCacheService.checkCacheTtl(
                        hostDbHostInfo, hostInfo.getXmlsearchIndexCount(), hostInfo.getIndexCountUpdateTime(), LinkType.INDEX_URLS, false, searchShard);
                hostInfo.setIndexCount(indexCount);
            } catch (UserException e) {
                log.error("Exception in " + getClass().getName() + " : " + e.getMessage(), e);
            } catch (InternalException e) {
                log.error("Exception in " + getClass().getName() + " : " + e.getMessage(), e);
            }
        }

        // Обновляем цифру из кеша xml-поиска, если данные в кеше устарели
        try {
            Long internalLinksCount = linksCacheService.checkCacheTtl(
                    hostDbHostInfo, hostInfo.getInternalLinksCount(), hostInfo.getInternalLinksCountUpdateTime(),
                    LinkType.INTERNAL_LINKS, false, searchShard);
            hostInfo.setInternalLinksCount(internalLinksCount);
        } catch (UserException e) {
            log.error("Exception in " + getClass().getName() + " : " + e.getMessage(), e);
        } catch (InternalException e) {
            log.error("Exception in " + getClass().getName() + " : " + e.getMessage(), e);
        }

        // Обновляем цифру из кеша xml-поиска, если данные в кеше устарели
        try {
            Long linksCount = linksCacheService.checkCacheTtl(
                    hostDbHostInfo, hostInfo.getLinksCount(), hostInfo.getLinksCountUpdateTime(),
                    LinkType.LINKS, false, searchShard);
            hostInfo.setLinksCount(linksCount);
        } catch (InternalException e) {
            log.error("InternalException in " + getClass().getName() + " : " + e.getMessage(), e);
        } catch (UserException e) {
            log.error("UserException in " + getClass().getName() + " : " + e.getMessage(), e);
        }
    }

    public List<ShortHostInfo> getShortHostInfo(long userId, BriefHostInfo hostInfo)
            throws InternalException
    {
        return getJdbcTemplate(new WMCPartition(null, userId)).query(
                    SELECT_USER_DB_HOST_INFO_QUERY, shortHostInfoMapper, userId, hostInfo.getId());
    }

    /**
     * Checks cache record is not up to date
     *
     * @param cacheInfo         current value of cache record
     * @param userId            user id
     * @param hostDbHostInfo    host db info
     * @return true if it is necessary to update cache record otherwise false
     * @throws InternalException
     */
    private boolean needUpdateCache(UsersHostsCacheInfo cacheInfo, long userId, HostDbHostInfo hostDbHostInfo) throws InternalException {
        Date optionsUpdatedOn = tblUsersDao.getOptionsUpdatedOn(userId);
        boolean needUpdate;
        if (cacheInfo == null) {
            needUpdate = true;
        } else if (cacheInfo.getErrorsUpdatedOn() == null) {
            needUpdate = true;
        } else if (optionsUpdatedOn == null) {
            needUpdate = false;
        } else if (optionsUpdatedOn.after(cacheInfo.getErrorsUpdatedOn())) {
            needUpdate = true;
        } else {
            needUpdate = false;
        }

        if (!needUpdate) {
            Date updatedOn = tblRobotdbInfoDao.getUpdatedOn(hostDbHostInfo);
            needUpdate = updatedOn != null && cacheInfo.getErrorsUpdatedOn() != null && updatedOn.after(cacheInfo.getErrorsUpdatedOn());
        }

        return needUpdate;
    }

    private UsersHostsCacheInfo checkCacheAndGetErrorsCount(long userId, BriefHostInfo hostInfo, HostDbHostInfo hostDbHostInfo) throws InternalException {
        UsersHostsCacheInfo cacheInfo = tblUsersHostsCacheDao.getUsersHostsCache(userId, hostInfo.getId());

        if (needUpdateCache(cacheInfo, userId, hostDbHostInfo)) {
            UserErrorOptions userErrorOptions = errorInfoService.getUserErrorOptions(userId, hostInfo.getId());
            long errorsCount = calculateErrorCount(userErrorOptions, hostDbHostInfo);
            Long trend = calculateErrorCountTrend(userErrorOptions, hostDbHostInfo);

            // save new cache values
            try {
                tblUsersHostsCacheDao.updateUsersHostsCache(userId, hostInfo.getId(), errorsCount, trend, new Date());
            } catch (InternalException e) {
                log.error("Failed to update errors count cache for host " + hostInfo.getName(), e);
            }

            return new UsersHostsCacheInfo(errorsCount, trend, new Date());
        } else {
            return cacheInfo;
        }
    }

    public HostStatusInfo getHostInfoStatus(BriefHostInfo hostInfo) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostInfo.getName());
        return getHostInfoStatus(hostDbHostInfo);
    }

    public HostStatusInfo getHostInfoStatus(HostDbHostInfo hostDbHostInfo) throws InternalException {
        List<HostStatusInfo> results = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).query(
                SELECT_HOST_INFO_STATUS_QUERY, HOST_INFO_STATUS_MAPPER, hostDbHostInfo.getHostDbHostId());
        return results.size() > 0 ? results.iterator().next() : new HostStatusInfo(null, null, null, 0, 0, false, false);
    }

    private static HostInfoStatusEnum getHostInfoStatusEnum(ResultSet resultSet, String column) throws SQLException {
        Integer value = resultSet.getInt(column);
        if (resultSet.wasNull()) {
            return null;
        } else {
            return HostInfoStatusEnum.R.fromValueOrNull(value);
        }
    }

    public Date getUpdatedOn(HostDbHostInfo hostDbHostInfo) throws InternalException {
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).safeQueryForTimestamp(
                UPDATE_UPDATED_ON_QUERY, hostDbHostInfo.getName());
    }

    /**
     * Returns brief information about each host of given user.
     *
     * @param userId User, for which to get host information.
     * @return Returns brief information about each host of given user.
     * @throws InternalException if db error occurs
     */
    public List<ShortHostInfo> getShortHostList(Long userId) throws InternalException {
        List<ShortHostInfo> result = getJdbcTemplate(WMCPartition.nullPartition()).query(
                SELECT_SHORT_HOST_LIST_QUERY, shortHostInfoMapper, userId);
        Collections.sort(result);
        return result;
    }

    /**
     * Returns brief information about each not verified host of given user.
     *
     * @param userId User, for which to get host information.
     * @param pager  Pager to separate pages of host list.
     * @return Returns brief information about each not verified host of given user.
     * @throws InternalException if db error occurs
     */
    public List<ShortHostInfo> getNotVerifiedShortHostList(Long userId, Pager pager) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).select(
                SELECT_NOT_VERIFIED_SHORT_HOST_LIST_COUNT_QUERY, SELECT_NOT_VERIFIED_SHORT_HOST_LIST_QUERY,
                shortHostInfoMapper, null, pager, userId);
    }

    public int removeHosts(long userId, FakeUser fakeUser, Long... hostIds) throws InternalException {
        return removeHosts(userId, fakeUser.getId(), hostIds);
    }

    /**
     * Removes given hosts from user's list. It is guaranteed, that informtaion about that hosts will be saved in DB
     * if it is used by at least one another user.
     *
     * @param userId     From which user's list to delete hosts.
     * @param realUserId User that really performs this operation.
     * @param hostIds    Hosts to remove.
     * @return Removed given hosts from users' list,
     * @throws InternalException if db error occurs
     */
    public int removeHosts(final long userId, final long realUserId, final Long... hostIds) throws InternalException {
        Integer result = 0;
        try {
            result = (Integer) getServiceTransactionTemplate(WMCPartition.nullPartition()).executeInService(
                    new ServiceTransactionCallback() {
                        @Override
                        public Object doInTransaction(TransactionStatus transactionStatus) throws InternalException {
                            for (Long hostId : hostIds) {
                                try {
                                    xmlLimitsService.processRemoveHost(userId, hostId);
                                } catch (UserException e) {
                                    throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "unable to update host limit");
                                }
                            }

                            ArrayList<Long> toDelete = new ArrayList<Long>(hostIds.length);
                            toDelete.addAll(Arrays.asList(hostIds));
                            List<Long> mirrors = tblHostsMainDao.getHostsMirrors(toDelete);
                            toDelete.addAll(mirrors);
                            int res = tblUsersHostsDao.deleteHostsForUser(userId, toDelete);
                            historyService.addEvents(realUserId, userId, HistoryActionEnum.REMOVE,
                                    WMCHistoryObjectTypeEnum.HOST, hostIds);
                            return res;
                        }
                    });
        } catch (UserException e) {
            throw new AssertionError("user exception isn't thrown here");
        }
        for (Long hostId : hostIds) {
            BriefHostInfo briefHostInfo = tblHostsMainDao.getBriefHostInfoByHostId(hostId);
            if (briefHostInfo != null) {
                safeNewWebmasterHttpService.removeHost(userId, briefHostInfo);
            }
        }
        return result;
    }

    private long calculateErrorCount(UserErrorOptions userErrorOptions, HostDbHostInfo hostDbHostInfo) throws InternalException {
        List<Integer> notOkErrorCodes = UrlErrorGrouper.expandGroups(userErrorOptions.getNotOkSeverityCodesWithGroups());
        return getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).queryForLong(
                String.format(SELECT_HOST_ERRORS_FOR_USER_QUERY,
                        SqlUtil.getCommaSeparatedList(notOkErrorCodes)),
                hostDbHostInfo.getHostDbHostId());
    }

    private Long calculateErrorCountTrend(UserErrorOptions userErrorOptions, HostDbHostInfo hostDbHostInfo) throws InternalException {
        List<Integer> notOkErrorCodes = UrlErrorGrouper.expandGroups(userErrorOptions.getNotOkSeverityCodesWithGroups());
        // Получаем историю за неделю
        Calendar c = Calendar.getInstance();
        c.setTime(new Date());
        c.add(Calendar.DATE, -7);

        NavigableMap<Date, Long> errorCountHistory = indexHistoryService.getErrorsHistoryForTrend(
                hostDbHostInfo, notOkErrorCodes, c.getTime());
        return IndexHistoryService.calculateTrend(errorCountHistory);
    }

    public boolean isIndexed(BriefHostInfo briefHostInfo) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());
        return isIndexed(hostDbHostInfo);
    }

    public boolean isIndexed(HostDbHostInfo hostDbHostInfo) throws InternalException {
        Long urls = tblRobotdbInfoDao.getUrlsByHostId(hostDbHostInfo);

        return (urls != null) && (urls > 0);
    }

    public boolean isQuickIndexed(BriefHostInfo briefHostInfo) throws InternalException {
        try {
            HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());
            YandexSearchShard searchShard = tblUrlTreesDao.getOptimumShardId(hostDbHostInfo);
            final Long indexed = linksCacheService.checkCacheAndGetIndexCount(briefHostInfo, searchShard);
            return indexed != null && indexed > 0;
        } catch (UserException ue) {
            return false;
        }
    }

    /**
     * Checks, if host informtaion in DB is currently in a state of first loading.
     * If the informtaion about host is first loading, we cannot operate it and must wait until it will be loaded.
     *
     * @param briefHostInfo Host, for which to check a sate.
     * @return Returns, whether host information in DB is first loading.
     * @throws InternalException Thrown if something's gone wrong. See InternalException.getProblem() for details.
     */
    public boolean isHostDataFirstLoading(final BriefHostInfo briefHostInfo) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());

        if (hostWithDataLoaded.contains(hostDbHostInfo.getHostDbHostId())) {
            return false;
        }

        Integer state = tblRobotdbInfoDao.getState(hostDbHostInfo);
        if (state == null) {
            return false;
        }
        if ((UpdateStateEnum.NEW.getValue() == state) || (UpdateStateEnum.FIRST_IN_PROGRESS.getValue() == state)) {
            return true;
        } else {
            hostWithDataLoaded.add(hostDbHostInfo.getHostDbHostId());
            return false;
        }
    }

    public String getHostNameByHostId(Long hostId) throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).safeQueryForString(
                SELECT_GET_HOST_NAME_BY_HOST_ID_QUERY, hostId);
    }

    public String getDisplayHostNameByHostName(String hostName) throws InternalException {
        return tblDisplayNameModerationDao.getDisplayHostNameByHostName(hostName);
    }

    public DisplayNameModerationStateEnum getDisplayNameModerationStateByHostId(Long hostId) throws InternalException {
        return tblDisplayNameModerationDao.getDisplayNameModerationStateByHostId(hostId);
    }

    public Long getHostIdByHostName(String name, boolean ignoreMainMirror) throws InternalException {
        if (!ignoreMainMirror) {
            BriefHostInfo mainMirrorInfo = consistentMainMirrorService.getMainMirrorBriefHostInfo(name);
            return mainMirrorInfo != null ? mainMirrorInfo.getId() : null;
        }

        return getJdbcTemplate(WMCPartition.nullPartition()).safeQueryForLong(SELECT_GET_HOST_ID_BY_HOST_NAME_QUERY, name);
    }

    /**
     * Given a host name or host id, this method checks, if such host exists in our base and returns
     * its BriefHostInfo or <code>null</code>.
     *
     * @param host Host name or host id.
     * @return Returns BriefHostInfo if such host exists in our base, or <code>null</code> if it doesn't.
     * @throws InternalException If smth goes wrong.
     */
    public BriefHostInfo getBriefHostInfoByIdOrName(String host) throws InternalException {
        if ((host.matches("^\\d+$"))) { // we are given host id
            long hostId;
            try {
                hostId = Long.parseLong(host);
            } catch (NumberFormatException e) {
                return null;
            }

            return tblHostsMainDao.getBriefHostInfoByHostId(hostId);
        } else {    // we are given hostname
            return tblHostsMainDao.getHostIdByHostname(host);
        }
    }

    /**
     * Given a host name or host id, this method checks, if such host exists in our base and returns
     * its BriefHostInfo or <code>null</code>.
     *
     * @param host Host name or host id.
     * @return Returns BriefHostInfo if such host exists in our base, or <code>null</code> if it doesn't.
     * @throws InternalException If smth goes wrong.
     */
    public BriefHostInfo getMainMirrorBriefHostInfoByIdOrName(String host) throws InternalException {
        if ((host.matches("^\\d+$"))) {
            // we are given host id
            Long hostId;
            try {
                hostId = Long.parseLong(host);
            } catch (NumberFormatException e) {
                return null;
            }

            BriefHostInfo hostInfo = tblHostsMainDao.getBriefHostInfoByHostId(hostId);
            if (hostInfo == null) {
                return null;
            }

            if (hostInfo.getMainMirrorId() == null) {
                return hostInfo;
            }

            return tblHostsMainDao.getBriefHostInfoByHostId(hostInfo.getMainMirrorId());
        } else {
            return consistentMainMirrorService.getMainMirrorBriefHostInfo(host);
        }
    }

    /**
     * Получить информацию о статусе хостов и их пенальти для htarc и prew
     *
     * @param dbIndex индекс БД, с которой ведется работа
     * @return статусы для хостов, у которых есть информация о накопленных пенальти
     */
    public List<HostUnavailableInfo> getBadPenaltyHosts(final int dbIndex) throws InternalException {
        final List<HostUnavailableInfo> res = getJdbcTemplate(new WMCPartition(dbIndex)).query(
                SELECT_ALL_HOSTS_PENALTY_QUERY, hostUnavailableInfoMapper);
        // Получаем spider ip отдельными запросами
        for (HostUnavailableInfo info : res) {
            final Long spiderIpLong = getJdbcTemplate(new WMCPartition(dbIndex)).safeQueryForLong(
                    SELECT_SPIDER_IP_QUERY, info.getHostId());
            info.setSpiderIp(getStringIp(spiderIpLong));
        }
        return res;
    }

    private static final ParameterizedRowMapper<HostStatusInfo> HOST_INFO_STATUS_MAPPER =
            new ParameterizedRowMapper<HostStatusInfo>() {
                @Override
                public HostStatusInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
                    final HostInfoStatusEnum hostInfoStatus = getHostInfoStatusEnum(rs, FIELD_HOST_INFO_STATUS);

                    final HostInfoStatusEnum hostInfoStatusHtarc = getHostInfoStatusEnum(rs, FIELD_HOST_INFO_STATUS_HTARC);
                    Integer penaltyHtarc = null;
                    if (hostInfoStatusHtarc != null) {
                        penaltyHtarc = SqlUtil.getIntNullable(rs, FIELD_HOST_INFO_PENALTY_HTARC);
                    }

                    Boolean htarcInIndex = SqlUtil.getBooleanNullable(rs, FIELD_HTARC_IN_INDEX);
                    Boolean prevInIndex = SqlUtil.getBooleanNullable(rs, FIELD_PREV_IN_INDEX);

                    final HostInfoStatusEnum hostInfoStatusYa = getHostInfoStatusEnum(rs, FIELD_HOST_INFO_STATUS_YA);
                    Integer penaltyYa = null;
                    if (hostInfoStatusYa != null) {
                        penaltyYa = SqlUtil.getIntNullable(rs, FIELD_HOST_INFO_PENALTY_YA);
                    }
                    return new HostStatusInfo(hostInfoStatusHtarc, hostInfoStatusYa, hostInfoStatus, penaltyHtarc, penaltyYa, htarcInIndex, prevInIndex);
                }
            };


    private static final ParameterizedRowMapper<ShortHostInfo> shortHostInfoMapper = new ParameterizedRowMapper<ShortHostInfo>() {
        @Override
        public ShortHostInfo mapRow(ResultSet resultSet, int rowNumber) throws SQLException {
            return new ShortHostInfo(
                    resultSet.getLong(FIELD_HOST_ID),
                    resultSet.getString(FIELD_HOSTNAME),
                    VerificationStateEnum.R.fromValueOrNull(resultSet.getInt(FIELD_VERIFICATION_STATE)),
                    VerificationTypeEnum.R.fromValueOrNull(resultSet.getInt(FIELD_VERIFICATION_TYPE))
            );
        }
    };

    private class HostInfoRowMapper implements ParameterizedRowMapper<HostInfo> {
        private final HostDbHostInfo hostDbHostInfo;
        private final UserDbHostInfo userDbHostInfo;
        private final boolean getUncachedTcy;

        public HostInfoRowMapper(HostDbHostInfo hostDbHostInfo, UserDbHostInfo userDbHostInfo, boolean getUncachedTcy) {
            this.hostDbHostInfo = hostDbHostInfo;
            this.userDbHostInfo = userDbHostInfo;
            this.getUncachedTcy = getUncachedTcy;
        }

        @Override
        public HostInfo mapRow(ResultSet rs, int rowNum) throws SQLException {
            Long urlCount = rs.getLong(FIELD_URL_COUNT);
            if (rs.wasNull()) {
                urlCount = null;
            }


            Long indexCount = SqlUtil.getLongNullable(rs, FIELD_PRECISE_INDEX_COUNT);
            Date indexCountUpdateTime = SqlUtil.safeGetTimestamp(rs, FIELD_INDEX_COUNT_TIME);

            Long xmlsearchIndexCount = SqlUtil.getLongNullable(rs, FIELD_INDEX_COUNT);

            Long internalLinksCount = SqlUtil.getLongNullable(rs, FIELD_INTERNAL_LINKS_COUNT);
            Date internalLinksUpdateTime = SqlUtil.safeGetTimestamp(rs, FIELD_INTERNAL_LINKS_TIME);

            Long linksCount = SqlUtil.getLongNullable(rs, FIELD_LINKS_COUNT);
            Date linksUpdateTime = SqlUtil.safeGetTimestamp(rs, FIELD_LINKS_TIME);

            Date lastAccess = SqlUtil.safeGetTimestamp(rs, FIELD_LAST_ACCESS);
            Date lastAccessHtarc = SqlUtil.safeGetTimestamp(rs, FIELD_LAST_ACCESS_HTARC);
            Long lastLoaderTime = SqlUtil.getLongNullable(rs, FIELD_LAST_LOADER_TIME);
            Date updatedOn = SqlUtil.safeGetTimestamp(rs, FIELD_UPDATED_ON);
            Integer docs = rs.getInt(FIELD_DOCS);

            Integer updateStateInt = rs.getInt(FIELD_UPDATE_STATE);
            UpdateStateEnum updateState = UpdateStateEnum.R.fromValueOrNull(updateStateInt);
            Integer sitemapCount = rs.getInt(FIELD_SITEMAP_COUNT);

            Integer tcy = rs.getInt(FIELD_TCY);
            if (rs.wasNull()) {
                if (getUncachedTcy) {
                    tcy = linksCacheService.getAndCacheTcyByHostname(userDbHostInfo.getName());
                } else {
                    tcy = null;
                }
            }

            Boolean virusedState = false;
            if (rs.getObject(FIELD_VIRUSED_STATE) != null) {
                virusedState = true;
            }

            final HostInfoStatusEnum hostInfoStatus = getHostInfoStatusEnum(rs, FIELD_HOST_INFO_STATUS);
            final Date hostInfoStatusDate = SqlUtil.safeGetTimestamp(rs, FIELD_HOST_INFO_STATUS_DATE);

            final HostInfoStatusEnum hostInfoStatusHtarc = getHostInfoStatusEnum(rs, FIELD_HOST_INFO_STATUS_HTARC);
            Integer penaltyHtarc = null;
            if (hostInfoStatusHtarc != null) {
                penaltyHtarc = SqlUtil.getIntNullable(rs, FIELD_HOST_INFO_PENALTY_HTARC);
            }

            final HostInfoStatusEnum hostInfoStatusYa = getHostInfoStatusEnum(rs, FIELD_HOST_INFO_STATUS_YA);
            Integer penaltyYa = null;
            if (hostInfoStatusYa != null) {
                penaltyYa = SqlUtil.getIntNullable(rs, FIELD_HOST_INFO_PENALTY_YA);
            }

            final Boolean htarcInIndex = SqlUtil.getBooleanNullable(rs, FIELD_HTARC_IN_INDEX);
            final Boolean prevInIndex = SqlUtil.getBooleanNullable(rs, FIELD_PREV_IN_INDEX);

            final Long ip = rs.getLong(FIELD_SPIDER_IP);
            String stringIp;
            if (rs.wasNull()) {
                stringIp = null;
            } else {
                stringIp = getStringIp(ip);
            }

            final Long urlsTrend = SqlUtil.getLongNullable(rs, FIELD_URLS_TREND);
            final Long indexCountTrend = SqlUtil.getLongNullable(rs, FIELD_INDEX_COUNT_TREND);
            final Long linksCountTrend = SqlUtil.getLongNullable(rs, FIELD_LINKS_COUNT_TREND);
            final Long tcyTrend = SqlUtil.getLongNullable(rs, FIELD_TCY_TREND);

            return new HostInfo(userDbHostInfo.getId(), userDbHostInfo.getName(), tcy,
                    userDbHostInfo.getVerificationState(), userDbHostInfo.getVerificationType(), urlCount,
                    userDbHostInfo.getUrlErrors(), lastAccess, lastAccessHtarc, lastLoaderTime, updatedOn, docs, updateState, indexCount,
                    internalLinksCount, linksCount, sitemapCount, virusedState, hostInfoStatus, stringIp, hostInfoStatusHtarc, penaltyHtarc, hostInfoStatusYa, penaltyYa, hostInfoStatusDate,
                    urlsTrend, indexCountTrend, linksCountTrend, tcyTrend, userDbHostInfo.getUrlErrorsTrend(), htarcInIndex, prevInIndex,
                    xmlsearchIndexCount,
                    indexCountUpdateTime, linksUpdateTime, internalLinksUpdateTime,
                    userDbHostInfo.isMainMirror());
        }
    }

    private static final ParameterizedRowMapper<HostUnavailableInfo> hostUnavailableInfoMapper = new ParameterizedRowMapper<HostUnavailableInfo>() {
        @Override
        public HostUnavailableInfo mapRow(ResultSet resultSet, int rowNumber) throws SQLException {
            final Date lastAccessYa = SqlUtil.safeGetTimestamp(resultSet, FIELD_LAST_ACCESS_YA);
            final Date lastAccessHtarc = SqlUtil.safeGetTimestamp(resultSet, FIELD_LAST_ACCESS_HTARC);
            final Boolean htarcInIndex = SqlUtil.getBooleanNullable(resultSet,  FIELD_HTARC_IN_INDEX);
            final Boolean prevInIndex = SqlUtil.getBooleanNullable(resultSet, FIELD_PREV_IN_INDEX);
            return new HostUnavailableInfo(resultSet.getLong(FIELD_HOST_ID),
                    resultSet.getString(FIELD_HOSTNAME),
                    getHostInfoStatusEnum(resultSet, FIELD_HOST_INFO_STATUS_HTARC),
                    getHostInfoStatusEnum(resultSet, FIELD_HOST_INFO_STATUS_YA),
                    null,
                    resultSet.getInt(FIELD_HOST_INFO_PENALTY_HTARC),
                    resultSet.getInt(FIELD_HOST_INFO_PENALTY_YA),
                    lastAccessYa,
                    lastAccessHtarc,
                    htarcInIndex,
                    prevInIndex);
        }
    };

    @Required
    public void setIndexHistoryService(IndexHistoryService indexHistoryService) {
        this.indexHistoryService = indexHistoryService;
    }

    @Required
    public void setLinksCacheService(LinksCacheService linksCacheService) {
        this.linksCacheService = linksCacheService;
    }

    @Required
    public void setXmlLimitsService(XmlLimitsService xmlLimitsService) {
        this.xmlLimitsService = xmlLimitsService;
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setHistoryService(HistoryService historyService) {
        this.historyService = historyService;
    }

    @Required
    public void setErrorInfoService(ErrorInfoService errorInfoService) {
        this.errorInfoService = errorInfoService;
    }

    @Required
    public void setSafeNewWebmasterHttpService(SafeNewWebmasterHttpService safeNewWebmasterHttpService) {
        this.safeNewWebmasterHttpService = safeNewWebmasterHttpService;
    }

    @Required
    public void setTblUsersHostsCacheDao(TblUsersHostsCacheDao tblUsersHostsCacheDao) {
        this.tblUsersHostsCacheDao = tblUsersHostsCacheDao;
    }

    @Required
    public void setTblUsersDao(TblUsersDao tblUsersDao) {
        this.tblUsersDao = tblUsersDao;
    }

    @Required
    public void setTblRobotdbInfoDao(TblRobotdbInfoDao tblRobotdbInfoDao) {
        this.tblRobotdbInfoDao = tblRobotdbInfoDao;
    }

    @Required
    public void setTblUsersHostsDao(TblUsersHostsDao tblUsersHostsDao) {
        this.tblUsersHostsDao = tblUsersHostsDao;
    }

    @Required
    public void setTblHostsMainDao(TblHostsMainDao tblHostsMainDao) {
        this.tblHostsMainDao = tblHostsMainDao;
    }

    @Required
    public void setTblDisplayNameModerationDao(TblDisplayNameModerationDao tblDisplayNameModerationDao) {
        this.tblDisplayNameModerationDao = tblDisplayNameModerationDao;
    }

    @Required
    public void setTblUrlTreesDao(TblUrlTreesDao tblUrlTreesDao) {
        this.tblUrlTreesDao = tblUrlTreesDao;
    }

    @Required
    public void setConsistentMainMirrorService(ConsistentMainMirrorService consistentMainMirrorService) {
        this.consistentMainMirrorService = consistentMainMirrorService;
    }
}
