package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.common.util.concurrent.CommonThreadFactory;
import ru.yandex.webmaster3.SafeNewWebmasterHttpService;
import ru.yandex.wmconsole.common.service.AbstractUrlVisitorService;
import ru.yandex.wmconsole.data.NotificationTypeEnum;
import ru.yandex.wmconsole.data.VerificationStateEnum;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.UsersHostsInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmconsole.verification.VerificationTypeEnum;
import ru.yandex.wmconsole.verification.Verifier;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.UserService;
import ru.yandex.wmtools.common.util.SqlUtil;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 20.03.2007
 * Time: 10:09:24
 */
public class HostVisitorService extends AbstractUrlVisitorService<UsersHostsInfo> {
    private static final Logger log = LoggerFactory.getLogger(HostVisitorService.class);

    private static final String FIELD_HOST_ID = "host_id";
    private static final String FIELD_USER_ID = "user_id";
    private static final String FIELD_HOST_NAME = "hostname";
    private static final String FIELD_VERIFICATION_UIN = "verification_uin";
    private static final String FIELD_VERIFICATION_TYPE = "verification_type";
    private static final String FIELD_VERIFICATION_STATE = "verification_state";
    private static final String FIELD_VERIFY_FAULT_LOG = "verify_fault_log";
    private static final String FIELD_CODE = "code";

    private static final String SELECT_WAITING_QUERY =
            "SELECT " +
                    "    uh.user_id AS " + FIELD_USER_ID + ", " +
                    "    uh.host_id AS " + FIELD_HOST_ID + ", " +
                    "    h.name AS " + FIELD_HOST_NAME + ", " +
                    "    uh.verification_uin AS " + FIELD_VERIFICATION_UIN + ", " +
                    "    uh.verify_fault_log AS " + FIELD_VERIFY_FAULT_LOG + ", " +
                    "    uh.state AS " + FIELD_VERIFICATION_STATE + ", " +
                    "    uh.verification_type AS " + FIELD_VERIFICATION_TYPE + ", " +
                    "    code.code AS " + FIELD_CODE + " " +
                    "FROM " +
                    "   tbl_hosts h " +
                    "INNER JOIN " +
                    "   tbl_users_hosts uh " +
                    "USING " +
                    "   (host_id)" +
                    "LEFT JOIN " +
                    "   tbl_dns_verifications dns " +
                    "USING " +
                    "   (host_id, user_id) " +
                    "LEFT JOIN " +
                    "   tbl_verification_problem_http code " +
                    "USING " +
                    "   (host_id, user_id) " +
                    "WHERE " +
                    "   uh.state = " + VerificationStateEnum.RECHECK_WAITING.value() + " OR " +
                    "   uh.state = " + VerificationStateEnum.WAITING.value() + " OR " +
                    "   ( uh.state = " + VerificationStateEnum.LONG_DNS_WAITING.value() + " AND " +
                    "   dns.next_try_at < NOW() ) " +
                    "ORDER BY " +
                    "   uh.state asc " +
                    "LIMIT 1 ";// +
    //"FOR UPDATE";

    private UserService userService;
    private UsersHostsService usersHostsService;
    private XmlLimitsService xmlLimitsService;

    private final List<VerificationListener> listeners = new ArrayList<VerificationListener>();
    private SendInternalNotificationService sendInternalNotificationService;
    private SafeNewWebmasterHttpService safeNewWebmasterHttpService;
    private ExecutorService pool;
    private final AtomicInteger numActive = new AtomicInteger(0);

    private EnumMap<VerificationTypeEnum, Verifier> verifiers;

    public void init() {
        CommonThreadFactory threadFactory = new CommonThreadFactory(true, HostVisitorService.class.getSimpleName() + "-");
        pool = Executors.newFixedThreadPool(1, threadFactory);
    }

    public HostVisitorService() {
//        Sending messages
        listeners.add(new VerificationListener() {
            @Override
            public void verified(UsersHostsInfo info) {
                Map<String, Object> params = new HashMap<String, Object>();
                params.put("host_id", info.getHostId());
                params.put("user_id", info.getUserId());
                params.put("ver_type", info.getVerificationType().value());
                params.put("ver_date", info.getVerificationDate().getTime());

                try {
                    sendInternalNotificationService.sendInternalNotification(NotificationTypeEnum.HOST_VERIFIED, params);
                } catch (InternalException e) {
                    log.error("Failed to send verification notification: " + e.getMessage(), e);
                }
            }
        });
    }

    public void addListener(VerificationListener listener) {
        listeners.add(listener);
    }

    private final ParameterizedRowMapper<UsersHostsInfo> usersHostsInfoRowMapper = new ParameterizedRowMapper<UsersHostsInfo>() {
        @Override
        public UsersHostsInfo mapRow(ResultSet rs, int i) throws SQLException {
            long userId = rs.getLong(FIELD_USER_ID);

            return new UsersHostsInfo(
                    VerificationStateEnum.R.fromValueOrNull(rs.getByte(FIELD_VERIFICATION_STATE)),
                    rs.getBigDecimal(FIELD_VERIFICATION_UIN).longValue(),
                    VerificationTypeEnum.R.fromValueOrNull(rs.getInt(FIELD_VERIFICATION_TYPE)),
                    new Date(),
                    rs.getString(FIELD_VERIFY_FAULT_LOG),
                    userId,
                    rs.getLong(FIELD_HOST_ID),
                    rs.getString(FIELD_HOST_NAME),
                    userService.getUserInfo(userId),
                    SqlUtil.getIntNullable(rs, FIELD_CODE)
            );
        }
    };

    protected UsersHostsInfo checkVerification(UsersHostsInfo verInfo) {
        UsersHostsInfo info = tryAutoVerification(verInfo);
        if (info != null) {
            return info;
        }

        Verifier verifier = getVerifier(verInfo.getVerificationType());
        if (VerificationStateEnum.RECHECK_WAITING.equals(verInfo.getVerificationState())) {
            return verifier.cancel(verInfo);
        } else {    // WAITING OR LONG_DNS_WAITING
            return verifier.verify(verInfo);
        }
    }

    private UsersHostsInfo tryAutoVerification(UsersHostsInfo verInfo) {
        for (VerificationTypeEnum type : VerificationTypeEnum.getAutoVerifications()) {
            UsersHostsInfo info = getVerifier(type).verify(verInfo);
            if (VerificationStateEnum.VERIFIED.equals(info.getVerificationState())) {
                return info;
            }
        }

        return null;
    }

    @Override
    protected final UsersHostsInfo listFirstWaiting(int databaseIndex) throws InternalException {
        List<UsersHostsInfo> res = getJdbcTemplate(WMCPartition.nullPartition()).query(SELECT_WAITING_QUERY,
                usersHostsInfoRowMapper);
        if (res == null) {
            return null;
        }
        for (UsersHostsInfo info : res) {
            if (info != null) {
                return info;
            }
        }

        return null;
    }

    @Override
    protected final UsersHostsInfo check(UsersHostsInfo oldInfo) {
        return oldInfo;
    }

    public UsersHostsInfo checkAutoVerification(UsersHostsInfo oldInfo) throws InternalException {
        UsersHostsInfo newInfo = tryAutoVerification(oldInfo);

        if (newInfo != null) {
            // записываем изменения в базу
            usersHostsService.updateVerificationInfoInTransaction(newInfo);

            // обрабатываем изменение состояния
            onStateChanged(oldInfo, newInfo);
        }

        return newInfo;
    }

    public UsersHostsInfo checkExternalVerification(UsersHostsInfo oldInfo, VerificationTypeEnum verificationType) throws InternalException {
        UsersHostsInfo newInfo = getVerifier(verificationType).verify(oldInfo);

        if (newInfo != null) {
            // записываем изменения в базу
            usersHostsService.updateVerificationInfoInTransaction(newInfo);

            // обрабатываем изменение состояния
            onStateChanged(oldInfo, newInfo);
        }

        return newInfo;
    }

    private Verifier getVerifier(VerificationTypeEnum verificationType) {
        Verifier verifier = verifiers.get(verificationType);
        if (verifier == null) {
            throw new RuntimeException("Unregistered verifier: " + verificationType);
        }
        return verifier;
    }

    private void notifyListeners(UsersHostsInfo info) {
        log.debug("Notifying verification listeners...");
        for (VerificationListener listener : listeners) {
            try {
                listener.verified(info);
            } catch (RuntimeException e) {
                log.error("RuntimeException occured when notifying verification listener " +
                        listener.getClass().getName() + " for host " + info.getHostName() +
                        " and user " + info.getUserId(), e);
            }
        }
        log.debug("All verification listeners have been notified.");
    }

    @Override
    protected final void updateStateToInProgress(UsersHostsInfo verInfo) throws InternalException {
        usersHostsService.resetVerificationsToInProgressForHost(verInfo);
    }

    @Override
    protected final void updateStateToCurrentStateValue(UsersHostsInfo verInfo) throws InternalException {
        usersHostsService.updateVerificationInfoInTransaction(verInfo);
    }

    @Override
    protected void onStateChanged(UsersHostsInfo oldInfo, UsersHostsInfo newInfo) throws InternalException {
        // Сообщения для пользователей о подтверждении прав
        if (VerificationStateEnum.VERIFIED.equals(newInfo.getVerificationState()) && !oldInfo.getVerificationState().isVerified()) {
            notifyListeners(newInfo);
        }

        final boolean oldVerified = oldInfo.getVerificationState().isVerified();
        final boolean newVerified = newInfo.getVerificationState().isVerified();

        if (oldVerified && !newVerified) {
            // перепроверка прав не подтвердила права пользователя
            try {
                xmlLimitsService.processRemoveHost(newInfo.getUserId(), newInfo.getHostId());
            } catch (InternalException e) {
                log.error("error in processRemoveHost", e);
            } catch (UserException e) {
                log.error("error in processRemoveHost", e);
            }

            try {
                BriefHostInfo hostInfo = new BriefHostInfo(oldInfo.getHostId(), oldInfo.getHostName(), null);
                safeNewWebmasterHttpService.cancelVerification(oldInfo.getUserId(), hostInfo);
            } catch (InternalException ie) {
                log.error("Unable to cancel webmaster3 verification for " + oldInfo.getUserId() +
                        " and " + oldInfo.getHostName(), ie);
            }
        } else if (!oldVerified && newVerified) {
            try {
                xmlLimitsService.processHostVerified(newInfo);
            } catch (InternalException e) {
                log.error("error in processHostVerified", e);
            }

            try {
                BriefHostInfo hostInfo = new BriefHostInfo(oldInfo.getHostId(), oldInfo.getHostName(), null);
                safeNewWebmasterHttpService.verifyHost(oldInfo.getUserId(), hostInfo);
            } catch (InternalException ie) {
                log.error("Unable to verify host in webmaster3 for " + oldInfo.getUserId() +
                          " and " + oldInfo.getHostName(), ie);
            }
        }
    }

    @Override
    public final void fireCheck() {
        if (numActive.compareAndSet(0, 1)) {
            try {
                pool.submit(new Runnable() {
                    @Override
                    public void run() {
                        logUserDbConnections();
                        try {
                            checkDatabase(getDatabaseCount() - 1);
                        } finally {
                            logUserDbConnections();
                            numActive.decrementAndGet();
                        }
                    }
                });
            } catch (Exception e) {
                log.error("Error while submitting new task to HostVisitor", e);
                numActive.set(0);
            }
        } else {
            // Предыдущие потоки еще не завершились
            log.info("Previous threads are still running. Suspend checks");
            // Восстанавливаем работу системы, если пришли в недопустимое состояние
            int num = numActive.get();
            if (num < 0 || num > 1) {
                log.error("Changing numActive from " + num + " to 0");
                numActive.set(0);
            }
        }
    }

    public interface VerificationListener {
        void verified(UsersHostsInfo info);
    }

    @Required
    public void setSendInternalNotificationService(SendInternalNotificationService sendInternalNotificationService) {
        this.sendInternalNotificationService = sendInternalNotificationService;
    }

    @Required
    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    @Required
    public void setUsersHostsService(UsersHostsService usersHostsService) {
        this.usersHostsService = usersHostsService;
    }

    @Required
    public void setXmlLimitsService(XmlLimitsService xmlLimitsService) {
        this.xmlLimitsService = xmlLimitsService;
    }

    @Required
    public void setVerifiers(Map<VerificationTypeEnum, Verifier> verifiers) {
        this.verifiers = new EnumMap<>(verifiers);
    }

    @Required
    public void setSafeNewWebmasterHttpService(SafeNewWebmasterHttpService safeNewWebmasterHttpService) {
        this.safeNewWebmasterHttpService = safeNewWebmasterHttpService;
    }
}
