package ru.yandex.wmconsole.service;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.cyberneko.html.parsers.SAXParser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import ru.yandex.misc.ip.Ipv4Address;
import ru.yandex.wmconsole.data.ServerResponseInfo;
import ru.yandex.wmconsole.error.ClientException;
import ru.yandex.wmconsole.error.ClientProblem;
import ru.yandex.wmconsole.error.WMCExtraTagNameEnum;
import ru.yandex.wmconsole.service.error.WMCUserProblem;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.IService;
import ru.yandex.wmtools.common.sita.DocumentFormatEnum;
import ru.yandex.wmtools.common.sita.SitaException;
import ru.yandex.wmtools.common.sita.SitaIncompleteResponseException;
import ru.yandex.wmtools.common.sita.SitaRequestTimeout;
import ru.yandex.wmtools.common.sita.SitaService;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequest;
import ru.yandex.wmtools.common.sita.SitaUrlFetchRequestBuilder;
import ru.yandex.wmtools.common.sita.SitaUrlFetchResponse;
import ru.yandex.wmtools.common.sita.UserAgentEnum;
import ru.yandex.wmtools.common.sita.YandexCharset;
import ru.yandex.wmtools.common.sita.YandexMimeType;
import ru.yandex.wmtools.common.util.GZipException;
import ru.yandex.wmtools.common.util.HttpConnector;
import ru.yandex.wmtools.common.util.HttpResponse;
import ru.yandex.wmtools.common.util.URLUtil;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * HTTP Response servantlet methods
 */
public class HttpResponseService implements IService {
    private static final Logger log = LoggerFactory.getLogger(HttpResponseService.class);

    private static final String LOCATION_HEADER = "Location";
    private static final String IF_MODIFIED_SINCE_HEADER = "If-Modified-Since";
    private static final String X_YANDEX_HEADER_PREFIX = "X-Yandex-";

    private SitaService sitaService;

    private class MetaTagContentHandler extends DefaultHandler {
        private String charset = null;

        public String getCharset() {
            return charset;
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attrs) throws SAXException {
            if (!"meta".equalsIgnoreCase(localName)) {
                return;
            }

            if (attrs.getValue("charset") != null) {
                this.charset = attrs.getValue("charset");
            } else if ("Content-Type".equalsIgnoreCase(attrs.getValue("http-equiv"))) {
                String contentType = attrs.getValue("content");
                int charsetIndex = contentType.indexOf("charset");
                if (charsetIndex != -1) {
                    this.charset = contentType.substring(charsetIndex + 8);
                }
            }
        }
    }

    public void appendBodyIfPresent(ServerResponseInfo info, HttpResponse httpResponse) {
        try {
            if (httpResponse.getContent() != null) {
                String body = "";

                try {
                    ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
                    IOUtils.copy(httpResponse.getContent(), byteArrayOutputStream);

                    // determine enconding
                    String encoding = null;
                    Boolean isText = true;
                    try {
                        Collection<String> ctHeader = httpResponse.getHeaders().get("Content-Type");
                        if (ctHeader != null) {
                            String contentType = ctHeader.iterator().next().toLowerCase();
                            int charsetIndex = contentType.indexOf("charset");
                            if (charsetIndex != -1) {
                                encoding = contentType.substring(charsetIndex + 8);
                            }
                            if (!contentType.contains("text")) {
                                isText = false;
                            }
                        }
                    } catch (Exception e) {
                        log.debug("Failed to guess encoding: " + e.getMessage());
                    }
                    // Скрываем показ содержимого в рамках WMCON-4147
                    isText = false;

                    info.setText(isText);
                    if (isText) {
                        try {
                            MetaTagContentHandler metaTagContentHandler = new MetaTagContentHandler();
                            SAXParser saxParser = new SAXParser();
                            saxParser.setContentHandler(metaTagContentHandler);
                            saxParser.parse(new InputSource(new ByteArrayInputStream(byteArrayOutputStream.toByteArray())));
                            if (metaTagContentHandler.getCharset() != null) {
                                encoding = metaTagContentHandler.getCharset();
                            }
                        } catch (Exception e) {
                            log.debug("Sax parser exception: " + e.getMessage());
                        }

                        if (encoding == null) {
                            encoding = "UTF-8";
                        }

                        try {
                            body = byteArrayOutputStream.toString(encoding);
                        } catch (UnsupportedEncodingException e) {
                            encoding = "UTF-8";
                            body = byteArrayOutputStream.toString(encoding);
                        }
                        info.setEncoding(encoding);
                    }
                } finally {
                    httpResponse.getContent().close();
                }

                long size = (long) body.length();

                double smallSize = (double) size / (double) 1024;
                info.setSize(smallSize);
                info.setBody(body);
            }
        } catch (IOException e) {
            log.error("Exception while parsing http response body", e);
        }
    }

    /**
     * Проверить ответ сервера
     *
     * @param url                   проверяемый url
     * @param userAgent             определяет заголовок UserAgent в запросе
     * @param ifModifiedSinceOn     использовать ли заголовок if-modified-since
     * @param ifModifiedSinceDate   дата для заголовка if-modified-since
     */
    public ServerResponseInfo getHttpResponse(
            @NotNull
            URL url,
            @NotNull
            UserAgentEnum userAgent,
            @NotNull
            Boolean ifModifiedSinceOn,
            @Nullable
            Date ifModifiedSinceDate) throws UserException {
        HttpResponse response;
        InetAddress ip;
        HttpConnector.RequestBuilder builder = new HttpConnector.RequestBuilder(url)
                .userAgent(userAgent.getValue())
                .socketTimeout(10000)
                .allowRedirect(false);
        if (ifModifiedSinceOn) {
            DateFormat httpDateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
            String date = httpDateFormat.format(ifModifiedSinceDate);
            builder = builder.header(IF_MODIFIED_SINCE_HEADER, date);
        }
        try {
            response = builder.execute();
            ip = InetAddress.getByName(url.getHost());
        } catch (GZipException e) {
            throw new ClientException(ClientProblem.GZIP_STREAM_READ_ERROR,
                    "Error while reading gzip stream from url:" + url.toString());
        } catch (IOException e) {
            throw new UserException(WMCUserProblem.URL_CONNECTION_PROBLEM, "Error connecting to url: " + url.toString());
        }

        String statusName = YandexHttpStatus.getStatusString(YandexHttpStatus.parseCode(response.getStatusCode()));

        String location = response.getHeaders().get(LOCATION_HEADER) != null ? getLocation(response, url) : null;
        ServerResponseInfo info = new ServerResponseInfo(
                response.getStatusCode(),
                statusName,
                response.getHeaders(),
                ip.getHostAddress(),
                response.getRequestTime(),
                location,
                null
        );

        appendBodyIfPresent(info, response);

        return info;
    }

    public ServerResponseInfo getHttpResponseWithSita(
            @NotNull URL url,
            @NotNull UserAgentEnum userAgent,
            boolean ifModifiedSinceOn,
            @Nullable Date ifModifiedSinceDate) throws UserException, InternalException
    {
        return getHttpResponseWithSita(url, userAgent, ifModifiedSinceOn, ifModifiedSinceDate, false);
    }

    public ServerResponseInfo getHttpResponseWithSita(
            @NotNull URL url,
            @NotNull UserAgentEnum userAgent,
            boolean ifModifiedSinceOn,
            @Nullable Date ifModifiedSinceDate,
            boolean retainSitaRequestResponse) throws UserException, InternalException
    {
        SitaUrlFetchRequestBuilder builder = new SitaUrlFetchRequestBuilder(url)
                .setUserAgent(userAgent)
                .setDocumentFormat(DocumentFormatEnum.DF_HTTP_RESPONSE)
                .setRequestTimeout(SitaRequestTimeout._15_SECONDS);

        if (ifModifiedSinceOn) {
            Date d = ifModifiedSinceDate != null ? ifModifiedSinceDate : new Date();
            log.debug("d.getTime() = " + d.getTime());
            builder.setIfModifiedSince(Long.valueOf(d.getTime() / 1000L).intValue());
        }

        try {
            SitaUrlFetchRequest urlFetchRequest = builder.createSitaUrlFetchRequest();

            SitaUrlFetchResponse urlFetchResponse = sitaService.request(urlFetchRequest, retainSitaRequestResponse);
            if (urlFetchResponse.isAllowedInRobotsTxt() != null && !urlFetchResponse.isAllowedInRobotsTxt()) {
                throw new UserException(WMCUserProblem.URL_IS_DISALLOWED_IN_ROBOTS_TXT, "");
            } else if (urlFetchResponse.getSitaHttpStatus() == null ||
                    urlFetchResponse.getSitaHttpStatus() == YandexHttpStatus.UNKNOWN) {
                throw new UserException(WMCUserProblem.URL_CONNECTION_PROBLEM, "Error connecting to url: " + url.toString());
            } else {
                YandexHttpStatus status = urlFetchResponse.getSitaHttpStatus();
                String ip = urlFetchResponse.getIp4() != null ?
                        Ipv4Address.valueOf(urlFetchResponse.getIp4().intValue()).toString() : null;
                Integer httpCode = getStandardHttpCode(url, status.getCode(), urlFetchResponse);
                String statusName = YandexHttpStatus.getStatusString(status);
                Map<String, Collection<String>> headers = getHeaders(urlFetchResponse);
                String location =
                        headers.get(LOCATION_HEADER) != null ? headers.get(LOCATION_HEADER).iterator().next() : null;

                ServerResponseInfo serverResponseInfo = new ServerResponseInfo(
                        httpCode,
                        statusName,
                        headers,
                        ip,
                        urlFetchResponse.getResponseTime(),
                        location,
                        null
                );
                if (retainSitaRequestResponse) {
                    serverResponseInfo.setSitaRequestJson(urlFetchRequest.getRequestJson());
                    serverResponseInfo.setSitaResponseJson(urlFetchResponse.getResponseJson());
                }
                try {
                    appendTextBody(serverResponseInfo, urlFetchResponse);
                } catch (InternalException e) {
                    log.error("Exception reading http content", e);
                } catch (IOException e) {
                    log.error("Exception reading http content", e);
                }

                return serverResponseInfo;
            }
        } catch (SitaIncompleteResponseException e) {
            throw new UserException(WMCUserProblem.URL_CONNECTION_PROBLEM, "Error connecting to url: " + url.toString());
        } catch (SitaException e) {
            throw new InternalException(InternalProblem.PROCESSING_ERROR, "SitaException", e);
        }
    }

    private int getStandardHttpCode(URL url, int httpCode, SitaUrlFetchResponse responseInfo) throws UserException {
        if (!YandexHttpStatus.isStandardHttpCode(YandexHttpStatus.parseCode(httpCode))) {
            log.debug("zora httpCode for " + url + " is " + httpCode);
            if (responseInfo.hasDocument()) {
                try {
                    org.apache.http.HttpResponse httpHeaders = responseInfo.getParsedHttpHeaders();
                    Integer parsedHttpCode = httpHeaders.getStatusLine().getStatusCode();
                    if (YandexHttpStatus.isStandardHttpCode(YandexHttpStatus.parseCode(parsedHttpCode))) {
                        return parsedHttpCode;
                    }
                } catch (SitaException exception) {
                    log.error("Sita exception while parsing http code", exception);
                }
            }
            throw new UserException(
                    WMCUserProblem.URL_CONNECTION_PROBLEM, "Error connecting to url: " + url.toString(),
                    new ExtraTagInfo(WMCExtraTagNameEnum.EXTENDED_HTTP_CODE, Integer.toString(httpCode)));
        } else {
            return httpCode;
        }
    }

    private void appendTextBody(ServerResponseInfo info, SitaUrlFetchResponse sitaUrlFetchResponse)
            throws InternalException, IOException
    {
        InputStream documentContentStream = sitaUrlFetchResponse.getDocumentContentStream();
        if (documentContentStream == null) {
            return;
        }

        String body = "";

        // determine enconding

        Boolean isText = sitaUrlFetchResponse.getMimeType() == YandexMimeType.MIME_TEXT
                || sitaUrlFetchResponse.getMimeType() == YandexMimeType.MIME_HTML
                || sitaUrlFetchResponse.getMimeType() == YandexMimeType.MIME_XML;

        info.setText(isText);
        if (isText) {
            YandexCharset yandexCharset = sitaUrlFetchResponse.getCharset();

            Charset charset = SitaService.UTF_8;
            if (yandexCharset.getCharsetName() != null) {
                try {
                    charset = Charset.forName(yandexCharset.getCharsetName());
                } catch (UnsupportedCharsetException | IllegalCharsetNameException e) {
                    log.warn("Unknown charset, fallback to UTF-8: " + yandexCharset);
                }
            }

            body = sitaUrlFetchResponse.getDocumentContentAsString(charset);
            body = StringUtils.trimToEmpty(body);
            info.setEncoding(getCharsetDisplayName(charset));
        }

        long size = (long) body.length();

        double smallSize = (double) size / (double) 1024;
        info.setSize(smallSize);
        info.setBody(body);
    }

    private String getCharsetDisplayName(Charset charset) {
        StringBuilder sb = new StringBuilder(charset.displayName());
        Set<String> aliases = charset.aliases();
        if (!aliases.isEmpty()) {
            sb.append('(').append(StringUtils.join(aliases, ", ")).append(')');
        }
        return sb.toString();
    }

    private static String getLocation(HttpResponse httpResponse, URL url) {
        String location = httpResponse.getHeaders().get(LOCATION_HEADER).iterator().next();
        if (location.startsWith("/")) {
            location = URLUtil.getHostName(url, true) + location;
        }
        return location;
    }

    private static Map<String, Collection<String>> getHeaders(SitaUrlFetchResponse responseInfo) {
        if (!responseInfo.hasDocument()) {
            return Collections.emptyMap();
        }
        Map<String, Collection<String>> headers = new LinkedHashMap<String, Collection<String>>();
        for (Header header : responseInfo.getParsedHttpHeaders().getAllHeaders()) {
            if (header.getName().startsWith(X_YANDEX_HEADER_PREFIX)) {
                continue;
            }

            Collection<String> hs = headers.get(header.getName());
            if (hs == null) {
                hs = new LinkedList<String>();
                headers.put(header.getName(), hs);
            }
            hs.add(header.getValue());
        }
        return headers;
    }

    @Required
    public void setNewSitaService(SitaService sitaService) {
        this.sitaService = sitaService;
    }
}
