package ru.yandex.wmconsole.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;

import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.wmconsole.data.info.KPIInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.util.ParameterizedMapRowMapper;

/**
 * @author avhaliullin
 */
public class KPIInfoService extends AbstractDbService {
    private static final String FIELD_NAME = "name";
    private static final String FIELD_DATE = "date";
    private static final String FIELD_COUNTER = "cntr";
    private static final String FIELD_ID = "id";

    private static final String SELECT_AVAILABLE_KPIS_QUERY =
            "SELECT DISTINCT " +
                    "   dkc.name AS " + FIELD_NAME + ", " +
                    "   kc.id AS " + FIELD_ID + " " +
                    "FROM " +
                    "   tbl_kpi_cntrs kc " +
                    "JOIN " +
                    "   tbl_dic_kpi_cntrs dkc " +
                    "USING(id) ";

    private static final String SELECT_KPIS_PLOT_DATA_QUERY =
            "SELECT " +
                    "   kc.cntr AS " + FIELD_COUNTER + ", " +
                    "   kc.date AS " + FIELD_DATE + ", " +
                    "   dkc.name AS " + FIELD_NAME + " " +
                    "FROM " +
                    "   tbl_kpi_cntrs kc " +
                    "JOIN " +
                    "   tbl_dic_kpi_cntrs dkc " +
                    "USING(id) " +
                    "WHERE " +
                    "   kc.id = ? " +
                    "GROUP BY kc.date " +
                    "ORDER BY kc.date ASC ";

    private static final ParameterizedRowMapper<KPIInfo> kpiInfoMapper = new ParameterizedRowMapper<KPIInfo>() {
        @Override
        public KPIInfo mapRow(ResultSet rs, int i) throws SQLException {
            return new KPIInfo(rs.getString(FIELD_NAME), rs.getInt(FIELD_ID));
        }
    };

    public List<KPIInfo> getAvailableKPIs() throws InternalException {
        return getJdbcTemplate(WMCPartition.nullPartition()).query(SELECT_AVAILABLE_KPIS_QUERY, kpiInfoMapper);
    }

    public Map<String, NavigableMap<Date, Double>> getPlotData(List<Integer> kpiIds) throws InternalException {
        StringBuilder label = new StringBuilder();
        Map<String, NavigableMap<Date, Double>> res = new HashMap<String, NavigableMap<Date, Double>>();
        for (Integer kpiId : kpiIds) {
            NavigableMap<Date, Double> plot = getJdbcTemplate(WMCPartition.nullPartition()).queryForNavigableMap(
                    SELECT_KPIS_PLOT_DATA_QUERY,
                    new KPIPlotDataMapper(label),
                    kpiId);
            res.put(label.toString(), plot);
        }
        return res;
    }

    static class KPIPlotDataMapper implements ParameterizedMapRowMapper<Date, Double> {
        private final StringBuilder label;

        public KPIPlotDataMapper(StringBuilder label) {
            this.label = label;
            label.setLength(0);
        }

        @Override
        public Pair<Date, Double> mapRow(ResultSet rs, int i) throws SQLException {
            if (label.length() == 0) {
                label.append(rs.getString(FIELD_NAME));
            }
            return new Pair<Date, Double>(rs.getDate(FIELD_DATE), rs.getDouble(FIELD_COUNTER));
        }
    }
}
