package ru.yandex.wmconsole.service;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;

import ru.yandex.common.util.concurrent.CommonThreadFactory;
import ru.yandex.webmaster.common.urltree.YandexSearchShard;
import ru.yandex.wmconsole.data.info.BriefHostInfo;
import ru.yandex.wmconsole.data.info.HostDbHostInfo;
import ru.yandex.wmconsole.data.partition.WMCPartition;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.AbstractDbService;
import ru.yandex.wmtools.common.service.IndexInfoService;
import ru.yandex.wmtools.common.util.SqlUtil;

/**
 * User: azakharov
 * Date: 25.09.12
 * Time: 12:41
 */
public class LinksCacheService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(LinksCacheService.class);

    private static final String FIELD_LINKS_COUNT = "links_count";
    private static final String FIELD_TIME = "time";
    private static final String FIELD_TCY = "tcy";

    private static final String INSERT_TCY_CACHE_QUERY =
            "INSERT IGNORE INTO " +
                    "    tbl_tcy_cache (host_id, tcy) " +
                    "VALUES " +
                    "    (?, ?) ";

    private static final String SELECT_TCY_QUERY =
            "SELECT " +
                    "    tc.tcy AS " + FIELD_TCY + " " +
                    "FROM " +
                    "    tbl_tcy_cache tc " +
                    "LEFT JOIN " +
                    "    tbl_hosts h ON (h.host_id = tc.host_id) " +
                    "WHERE " +
                    "    h.name = ? ";

    private static final String SELECT_HOST_LINKS_COUNT_QUERY =
            "SELECT " +
                    "       %1$s AS " + FIELD_LINKS_COUNT + ", " +
                    "       %2$s AS " + FIELD_TIME + " " +
                    "   FROM " +
                    "       tbl_links_cache " +
                    "   WHERE " +
                    "       host_id = ? ";

    private static final String INSERT_LINKS_COUNT_DB_CACHE_QUERY =
            "INSERT INTO " +
                    "    tbl_links_cache(host_id, %1$s, %2$s, shard_id) " +
                    "VALUES " +
                    "    (?, ?, NOW(), ?) " +
                    "ON DUPLICATE KEY UPDATE " +
                    "    %1$s = ?, %2$s = NOW() ";


    private HostDbHostInfoService hostDbHostInfoService;
    private TcyProviderService tcyProviderService;
    private IndexInfoService indexInfoService;
    private IndexInfoService uaIndexInfoService;
    private IndexInfoService comIndexInfoService;
    private IndexInfoService trIndexInfoService;

    private int indexCountCacheTtlMinutes = 3 * 60;

    private final CommonThreadFactory threadFactory = new CommonThreadFactory(true, LinksCacheService.class.getSimpleName() + "-");

    private static final ParameterizedRowMapper<CacheRecord> linksCountMapper = new ParameterizedRowMapper<CacheRecord>() {
        @Override
        public CacheRecord mapRow (ResultSet resultSet,int rowNumber)throws SQLException {
            return new CacheRecord(
                    SqlUtil.getLongNullable(resultSet, FIELD_LINKS_COUNT),
                    SqlUtil.getDateNullable(resultSet, FIELD_TIME));
        }
    };

    private static final ParameterizedRowMapper<Integer> tcyMapper = new ParameterizedRowMapper<Integer>() {
        @Override
        public Integer mapRow(ResultSet resultSet, int rowNumber) throws SQLException {
            return SqlUtil.getIntNullable(resultSet, FIELD_TCY);
        }
    };

    public Integer checkCacheAndGetTcy(URL hostname) throws InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(
                SupportedProtocols.getCanonicalHostname(hostname));

        List<Integer> tcyList = getJdbcTemplate(new WMCPartition(hostDbHostInfo, null))
                .query(SELECT_TCY_QUERY, tcyMapper, SupportedProtocols.getCanonicalHostname(hostname));
        Integer tcy = tcyList.size() > 0 ? tcyList.iterator().next() : null;

        return (tcy == null) ? getAndCacheTcy(hostname) : tcy;
    }

    public Integer getAndCacheTcyByHostname(String hostname) throws AssertionError {
        URL url;

        try {
            url = SupportedProtocols.getURL(hostname);
        } catch (MalformedURLException e) {
            throw new AssertionError("invalid hostname in a database!");
        } catch (URISyntaxException e) {
            throw new AssertionError("invalid hostname in a database!");
        } catch (SupportedProtocols.UnsupportedProtocolException e) {
            throw new AssertionError("there is a host in a database with unsupported protocol!");
        }

        return getAndCacheTcy(url);
    }

    public Integer getAndCacheTcy(final URL url) {
        final Integer tcy = tcyProviderService.getTcy(url);

        // starting thread, that will cache tcy in async mode.
        if (tcy != null) {
            threadFactory.newThread(new Runnable() {
                @Override
                public void run() {
                    try {
                        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(
                                SupportedProtocols.getCanonicalHostname(url));
                        getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).update(INSERT_TCY_CACHE_QUERY,
                                hostDbHostInfo.getHostDbHostId(), tcy);
                    } catch (InternalException e) {
                        log.warn("InternalException occured while caching TCY", e);
                    }
                }
            }).start();
        }

        return tcy;
    }

    public Long checkCacheAndGetIndexCount(final BriefHostInfo briefHostInfo, final YandexSearchShard searchShard) throws InternalException, UserException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(briefHostInfo.getName());

        return checkCacheAndGetIndexCount(hostDbHostInfo, searchShard);
    }

    public Long checkCacheAndGetIndexCount(final HostDbHostInfo hostDbHostInfo, final YandexSearchShard searchShard) throws InternalException, UserException {
        CacheRecord indexCount = getLinksCountFromCache(hostDbHostInfo, LinkType.INDEX_URLS);
        if (indexCount == null) {
            return extractLinksCountAndUpdateCache(hostDbHostInfo, LinkType.INDEX_URLS, true, searchShard);
        }
        return checkCacheTtl(hostDbHostInfo, indexCount.getValue(), indexCount.getValueUpdateTime(), LinkType.INDEX_URLS, true, searchShard);
    }

    public Long getAndCacheIndexCount(final String hostname, final YandexSearchShard searchShard) throws UserException, InternalException {
        HostDbHostInfo hostDbHostInfo = hostDbHostInfoService.getHostDbHostInfo(hostname);
        return extractLinksCountAndUpdateCache(hostDbHostInfo, LinkType.INDEX_URLS, true, searchShard);
    }

    public Long checkCacheAndGetLinksCount(final HostDbHostInfo hostDbHostInfo, final YandexSearchShard searchShard) throws InternalException, UserException {
        final CacheRecord cacheRecord = getLinksCountFromCache(hostDbHostInfo, LinkType.LINKS);
        if (cacheRecord == null) {
            return extractLinksCountAndUpdateCache(hostDbHostInfo, LinkType.LINKS, true, searchShard);
        }
        return checkCacheTtl(hostDbHostInfo, cacheRecord.getValue(), cacheRecord.getValueUpdateTime(), LinkType.LINKS, true, searchShard);
    }

    public Long checkCacheAndGetIntLinksCount(final HostDbHostInfo hostDbHostInfo, final YandexSearchShard searchShard) throws InternalException, UserException {
        final CacheRecord cacheRecord = getLinksCountFromCache(hostDbHostInfo, LinkType.INTERNAL_LINKS);
        if (cacheRecord == null) {
            return extractLinksCountAndUpdateCache(hostDbHostInfo, LinkType.INTERNAL_LINKS, true, searchShard);
        }
        return checkCacheTtl(hostDbHostInfo, cacheRecord.getValue(), cacheRecord.getValueUpdateTime(), LinkType.INTERNAL_LINKS, true, searchShard);
    }

    public long getLinksCountAndUpdateCache(
            HostDbHostInfo hostInfo,
            LinkType linkType,
            YandexSearchShard searchShard,
            boolean allowCache) throws InternalException, UserException {
        if (allowCache) {
            CacheRecord cacheRecord = getLinksCountFromCache(hostInfo, linkType);
            if (cacheRecord == null) {
                return extractLinksCountAndUpdateCache(hostInfo, linkType, false, searchShard);
            }
            return checkCacheTtl(hostInfo, cacheRecord.getValue(), cacheRecord.getValueUpdateTime(), linkType, false, searchShard);
        }
        return extractLinksCountAndUpdateCache(hostInfo, linkType, false, searchShard);
    }

    /**
     * Adds (or updates, if it is already present) information about links count in DB.
     * Now exception is not thrown if operation was not finished successfully, because caching is not
     * a necessary operation. Error is just logged.
     *
     * @param hostDbHostInfo Host, for which to update links count.
     * @param linksCount     New count of links.
     * @param type           Type of links, information for which is being updated.
     * @param searchShard    Search shard to cache information for.
     */
    public void saveLinksCountToDbCache(HostDbHostInfo hostDbHostInfo, Long linksCount, LinkType type, YandexSearchShard searchShard) {
        try {
            String tt = String.format(INSERT_LINKS_COUNT_DB_CACHE_QUERY,
                    type.getDbFieldName(), type.getTimeDbFieldName());
            getJdbcTemplate(new WMCPartition(hostDbHostInfo, null)).update(
                    tt,
                    hostDbHostInfo.getHostDbHostId(), linksCount, searchShard.value(), linksCount);
        } catch (InternalException e) {
            log.warn("Links count of type " + type.getName() + " were not saved into DB.", e);
        }
    }

    /**
     * Позволяет проверить время жизни кэша в случае и перезапросить при превышении времени жизни
     * Может использоваться при выборке данных из кеша совместно с другими данными по хосту.
     *
     * @param hostDbHostInfo    информация по хосту
     * @param linksCount        количество ссылок, выбранное отдельным запросом
     * @param updatedOn         время последнего обновления кэша
     * @param linkType          тип сохраняемых ссылок
     * @return
     * @throws UserException
     * @throws InternalException
     */
    public Long checkCacheTtl(
            HostDbHostInfo hostDbHostInfo,
            Long linksCount,
            Date updatedOn,
            LinkType linkType,
            boolean runUpdateInThread,
            YandexSearchShard searchShard) throws UserException, InternalException {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MINUTE, -indexCountCacheTtlMinutes);
        Date minSatisfiableTime = new Date(c.getTimeInMillis());
        if (linksCount == null ||  // нет записи
                updatedOn == null ||   // нет записи
                minSatisfiableTime.after(updatedOn)) {  // запись устарела
            return extractLinksCountAndUpdateCache(hostDbHostInfo, linkType, runUpdateInThread, searchShard);
        } else {
            return linksCount;
        }
    }

    /**
     * Получить данные из кэша без проверки времени жизни
     * @param hostInfo  информация о хосте
     * @param linkType  тип числа ссылок
     * @return          запись из кэша
     * @throws InternalException
     */
    private CacheRecord getLinksCountFromCache(HostDbHostInfo hostInfo, LinkType linkType) throws InternalException {
        return getJdbcTemplate(new WMCPartition(hostInfo, null)).safeQueryForObject(
                String.format(SELECT_HOST_LINKS_COUNT_QUERY, linkType.getDbFieldName(), linkType.getTimeDbFieldName()),
                linksCountMapper,
                hostInfo.getHostDbHostId());
    }

    /**
     * Получить значение числа ссылок заданного типа и сохранить в кэше
     *
     * @param hostInfo      информация о хосте
     * @param linkType      тип числа ссылок
     * @param runInThread   запускать обновление кэша синхронно или асинхронно
     * @return              число ссылок заданного типа
     * @throws UserException
     * @throws InternalException
     */
    private Long extractLinksCountAndUpdateCache(
            final HostDbHostInfo hostInfo,
            final LinkType linkType,
            final boolean runInThread,
            final YandexSearchShard searchShard) throws UserException, InternalException {
        final IndexInfoService actualIndexInfoService;
        switch (searchShard) {
            case UA:        actualIndexInfoService = uaIndexInfoService; break;
            case COM:       actualIndexInfoService = comIndexInfoService; break;
            case COM_TR:    actualIndexInfoService = trIndexInfoService; break;
            default:        actualIndexInfoService = indexInfoService; break;
        }

        final Long count = actualIndexInfoService.extractLinksCount(linkType.getXmlSearchRequest(hostInfo.getName()));
        if (runInThread) {
            // starting thread, that will cache linksCount in async mode.
            threadFactory.newThread(new Runnable() {
                @Override
                public void run() {
                    saveLinksCountToDbCache(hostInfo, count, linkType, searchShard);
                }
            }).start();
        } else {
            saveLinksCountToDbCache(hostInfo, count, linkType, searchShard);
        }
        return count;
    }

    public static class CacheRecord {
        private Long value;
        private Date valueUpdateTime;

        public CacheRecord(Long value, Date valueUpdateTime) {
            this.value = value;
            this.valueUpdateTime = valueUpdateTime;
        }

        public Long getValue() {
            return value;
        }

        public Date getValueUpdateTime() {
            return valueUpdateTime;
        }
    }

    @Required
    public void setHostDbHostInfoService(HostDbHostInfoService hostDbHostInfoService) {
        this.hostDbHostInfoService = hostDbHostInfoService;
    }

    @Required
    public void setTcyProviderService(TcyProviderService tcyProviderService) {
        this.tcyProviderService = tcyProviderService;
    }

    @Required
    public void setIndexInfoService(IndexInfoService indexInfoService) {
        this.indexInfoService = indexInfoService;
    }

    @Required
    public void setUaIndexInfoService(IndexInfoService uaIndexInfoService) {
        this.uaIndexInfoService = uaIndexInfoService;
    }

    @Required
    public void setComIndexInfoService(IndexInfoService comIndexInfoService) {
        this.comIndexInfoService = comIndexInfoService;
    }

    @Required
    public void setTrIndexInfoService(IndexInfoService trIndexInfoService) {
        this.trIndexInfoService = trIndexInfoService;
    }

    @Required
    public void setIndexCountCacheTtlMinutes(int indexCountCacheTtlMinutes) {
        this.indexCountCacheTtlMinutes = indexCountCacheTtlMinutes;
    }
}
